############################################################################
#                                                                          #
#             copyright (c) 2004 ITB, Humboldt-University Berlin           #
#             written by: Raphael Ritz, r.ritz@biologie.hu-berlin.de       #
#                                                                          #
############################################################################

"""PDFRenderer class"""

# Python stuff
import os, tempfile, shutil

# Zope stuff
from Globals import InitializeClass
from App.Dialogs import MessageDialog
from OFS.PropertyManager import PropertyManager


# CMF stuff
from Products.CMFCore.utils import getToolByName

# Bibliography stuff
from Products.CMFBibliographyAT.tool.renderers.base \
     import IBibliographyRenderer, BibliographyRenderer

DEFAULT_TEMPLATE = r"""
\documentclass[12pt,a4paper]{article}
\usepackage[latin1]{inputenc}
\usepackage{bibmods}
\usepackage{bibnames}
\usepackage{showtags}
\renewcommand{\refname}{~}

\begin{document}
\begin{center}
{\large \bf %s}\\
(URL: %s)
\end{center}

~\hfill \today

\nocite{*}
\bibliographystyle{abbrv}
\bibliography{references}

\end{document}
"""

class PDFRenderer(BibliographyRenderer, PropertyManager):
    """
    specific pdf renderer
    depends on the BibTeX renderer
    """

    __implements__ = (IBibliographyRenderer ,)

    meta_type = "PDF Renderer"

    format = {'name':'PDF',
              'extension':'pdf'}
    template = ''
##     _properties = BibliographyRenderer._properties + \
    _properties = ({'id':'template', 'type':'text', 'mode':'w'},)

    manage_options = PropertyManager.manage_options + \
                     BibliographyRenderer.manage_options

    def __init__(self,
                 id = 'pdf',
                 title = "PDF renderer"):
        """
        initializes id, title and the default LaTeX template
        """
        self.id = id
        self.title = title
        self.template = DEFAULT_TEMPLATE

    def render(self, object):
        """
        renders a bibliography object in PDF format
        """
        bib_tool = getToolByName(object, 'portal_bibliography')
        bibtex_source = bib_tool.render(object, 'bib')
        # rr XXX: ugly hack
        bibtex_source = unicode(bibtex_source, 'utf-8').encode('latin-1')
        return self.processSource(bibtex_source,
                                  object.title_or_id(),
                                  object.absolute_url())

    def processSource(self, source, title, url):
        """
        use latex/bibtex/pdflatex to generate the pdf
        from the passed in BibTeX file in 'source' using
        the (LaTeX) source tempalte from the renderer's
        'template' property
        """
        template = self.getProperty('template', DEFAULT_TEMPLATE)
        template = template % (title, url)
        wd = self.getWorkingDirectory()
        tex_path = os.path.join(wd, 'template.tex')
        bib_path = os.path.join(wd, 'references.bib')
        tex_file = open(tex_path, 'w')
        bib_file = open(bib_path, 'w')
        tex_file.write(template)
        bib_file.write(source)
        tex_file.close()
        bib_file.close()
        os.system("cd %s; latex %s"% (wd, tex_path))
        os.system("cd %s; bibtex %s"% (wd, 'template'))
        os.system("cd %s; latex %s"% (wd, 'template.tex'))
        os.system("cd %s; pdflatex %s"% (wd, tex_path))
        pdf_file= open(os.path.join(wd, "template.pdf"), 'r')
        pdf = pdf_file.read()
        pdf_file.close()
        self.clearWorkingDirectory(wd)
        return pdf


    def getWorkingDirectory(self):
        """
        returns the full path to a newly created
        temporary working directory
        """
        tmp_dir = tempfile.mkdtemp()
        renderer_dir = '/'.join(os.path.split(__file__)[:-1])
        resource_dir = os.path.join(renderer_dir, 'latex_resources')
        for file in os.listdir(resource_dir):
            source = os.path.join(resource_dir, file)
            destination = os.path.join(tmp_dir, file)
            if os.path.isfile(source):
                shutil.copy(source, destination)
        return tmp_dir

    def clearWorkingDirectory(self, wd):
        """
        removes the temporary working directory
        """
        for file in os.listdir(wd):
            try:
                path = os.path.join(wd, file)
                os.remove(path)
            except OSError:
                pass
        os.rmdir(wd)

# Class instanciation
InitializeClass(PDFRenderer)


def manage_addPDFRenderer(self, REQUEST=None):
    """ """
    ## self._setObject('pdf', PDFRenderer())
    try:
        self._setObject('pdf', PDFRenderer())
    except:
        return MessageDialog(
            title='Bibliography tool warning message',
            message='The renderer you attempted to add already exists.',
            action='manage_main')
    return self.manage_main(self, REQUEST)
