#
# LinguaPlone Tests
#

import os, sys
if __name__ == '__main__':
    execfile(os.path.join(sys.path[0], 'framework.py'))

from Testing import ZopeTestCase

from Products.LinguaPlone.tests import LinguaPloneTestCase
from Products.LinguaPlone.tests import dummy
from Products.LinguaPlone.tests.utils import makeContent
from Products.LinguaPlone.tests.utils import makeTranslation
from Products.LinguaPlone.tests.utils import sortTuple
from Products.LinguaPlone import config

from Acquisition import aq_base
from OFS.ObjectManager import BeforeDeleteException


class TestLinguaPlone(LinguaPloneTestCase.LinguaPloneTestCase):

    def afterSetUp(self):
        # Speed things up
        self.portal._delObject('portal_catalog')

    # Test setup

    def testTools(self):
        # Check all tools are installed
        portal = aq_base(self.portal)
        self.failUnless(hasattr(portal, 'archetype_tool'))
        self.failUnless(hasattr(portal, 'portal_languages'))

    def testPortalTypes(self):
        # Check all content types are installed
        types = aq_base(self.portal.portal_types)
        self.failUnless(hasattr(types, 'Lingua Item'))
        self.failUnless(hasattr(types, 'Lingua Folder'))
        # Dummies
        self.failUnless(hasattr(types, 'SimpleType'))
        self.failUnless(hasattr(types, 'DerivedType'))
        self.failUnless(hasattr(types, 'SimpleFolder'))
        self.failUnless(hasattr(types, 'OrderedFolder'))
        self.failUnless(hasattr(types, 'BTreeFolder'))

    def testExampleTypes(self):
        self.folder.invokeFactory('Lingua Item', id='foo')
        self.failUnless(hasattr(aq_base(self.folder), 'foo'))
        self.folder.invokeFactory('Lingua Folder', id='bar')
        self.failUnless(hasattr(aq_base(self.folder), 'bar'))

    # Test interface

    def testCanonicalWithoutTranslation(self):
        self.addLanguage('de')
        self.addLanguage('fr')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        self.failIf(english.isTranslation())
        self.failUnless(english.isCanonical())
        self.assertEqual(english, english.getCanonical())

    def testAddAndGetTranslation(self):
        # addTranslation, getTranslation
        self.addLanguage('de')
        self.addLanguage('fr')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        self.failUnless(german)
        self.failIfEqual(english, german)
        # French translation
        french = makeTranslation(german, 'fr')
        self.failUnless(french)
        # Check getTranslation
        self.assertEqual(french, german.getTranslation('fr'))
        self.assertEqual(french, english.getTranslation('fr'))
        self.assertEqual(german, french.getTranslation('de'))
        self.assertEqual(german, english.getTranslation('de'))
        self.assertEqual(english, german.getTranslation('en'))
        self.assertEqual(english, french.getTranslation('en'))

    def testAddAndGetSecondTranslation(self):
        # addTranslation, getTranslation
        self.addLanguage('no')
        self.setLanguage('no')
        self.portal.portal_languages.getPreferredLanguage()
        norwegian = makeContent(self.folder, 'SimpleType', 'doc')
        self.assertEqual(norwegian.getLanguage(), 'no')
        german = makeTranslation(norwegian, 'de')
        self.failUnless(german)
        self.failIfEqual(norwegian, german)
        self.failIfEqual(norwegian.getLanguage(), german.getLanguage())
        self.assertEqual(german.getLanguage(), 'de')

    def testRemoveTranslation(self):
        self.addLanguage('de')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        self.failUnless(english)
        # German translation
        german = makeTranslation(english, 'de')
        self.failUnless(german)
        self.failIfEqual(english, german)
        # Remove translation
        english.removeTranslation('de')
        self.failIf(english.getTranslation('de'))

    def testCanonical(self):
        # isCanonical, getCanonicalLanguage, getCanonical
        self.addLanguage('de')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        self.failUnless(english.isCanonical())
        self.failIf(german.isCanonical())
        self.assertEqual('en', english.getCanonicalLanguage())
        self.assertEqual('en', german.getCanonicalLanguage())
        self.assertEqual(english, english.getCanonical())
        self.assertEqual(english, german.getCanonical())

    def testChangeCanonical(self):
        # isCanonical, getCanonicalLanguage, getCanonical
        self.addLanguage('de')
        self.setLanguage('en')
        self.addLanguage('fr')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        german.setCanonical()
        self.failIf(english.isCanonical())
        self.failUnless(german.isCanonical())
        self.assertEqual('de', english.getCanonicalLanguage())
        self.assertEqual('de', german.getCanonicalLanguage())
        self.assertEqual(german, english.getCanonical())
        self.assertEqual(german, german.getCanonical())
        french = makeTranslation(german, 'fr')
        french.setCanonical()
        self.failIf(english.isCanonical())
        self.failIf(german.isCanonical())
        self.failUnless(french.isCanonical())
        self.assertEqual('fr', english.getCanonicalLanguage())
        self.assertEqual('fr', french.getCanonicalLanguage())
        self.assertEqual('fr', german.getCanonicalLanguage())
        self.assertEqual(french, english.getCanonical())
        self.assertEqual(french, french.getCanonical())
        self.assertEqual(french, german.getCanonical())
        english.setCanonical()
        self.failUnless(english.isCanonical())
        self.failIf(german.isCanonical())
        self.failIf(french.isCanonical())
        self.assertEqual('en', english.getCanonicalLanguage())
        self.assertEqual('en', french.getCanonicalLanguage())
        self.assertEqual('en', german.getCanonicalLanguage())
        self.assertEqual(english, english.getCanonical())
        self.assertEqual(english, french.getCanonical())
        self.assertEqual(english, german.getCanonical())
        
    def testRemoveCanonicalTranslation(self):
        self.addLanguage('de')
        self.addLanguage('fr')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        self.failUnless(english)
        # German translation
        german = makeTranslation(english, 'de')
        self.failUnless(german)
        french = makeTranslation(english, 'fr')
        self.failUnless(french)
        # Remove canonical translation
        german.removeTranslation('en')
        self.failIf(german.getTranslation('en'))
        self.failUnless(german.getTranslation('fr'))

    def testGetTranslationLanguages(self):
        self.addLanguage('de')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        self.assertEqual(sortTuple(('en','de')), sortTuple(english.getTranslationLanguages()))

    def testGetTranslations(self):
        self.addLanguage('de')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        self.assertEqual(sortTuple(('en','de')), sortTuple(english.getTranslationLanguages()))
        translations = english.getTranslations()
        self.assertEqual(translations['en'][0], english)
        self.assertEqual(translations['en'][1], 'visible')
        self.assertEqual(translations['de'][0], german)
        self.assertEqual(translations['de'][1], 'visible')

    def testBasicFolderishMultilingual(self):
        self.addLanguage('de')
        self.setLanguage('en')
        folder = makeContent(self.folder, 'SimpleFolder', 'folder')
        folder_de = makeTranslation(folder, 'de')
        english = makeContent(folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        self.failUnless(english.getId() in folder.objectIds())
        self.failUnless(german.getId() in folder_de.objectIds())

    def testTranslationId(self):
        self.addLanguage('de')
        self.setLanguage('en')
        folder = makeContent(self.folder, 'SimpleFolder', 'folder')
        folder_de = makeTranslation(folder, 'de')
        english = makeContent(folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        # As the english and german are in different folders, ids are the same.
        self.assertEqual(english.getId(), german.getId())

    def testFolderishMultilingual(self):
        self.addLanguage('de')
        self.setLanguage('en')
        folder = makeContent(self.folder, 'SimpleFolder', 'folder')
        english = makeContent(folder, 'SimpleType', 'doc')
        english2 = makeContent(folder, 'SimpleType', 'doc2')
        german = makeTranslation(english, 'de')
        german2 = makeTranslation(english2, 'de')
        get_transaction().commit(1)
        # When translating the folder, translated content should be moved
        # to the new folder.
        folder_de = makeTranslation(folder, 'de')
        self.failUnless(english.getId() in folder.objectIds())
        self.failUnless(english2.getId() in folder.objectIds())
        self.failUnless(german.getId() in folder_de.objectIds())
        self.failUnless(german2.getId() in folder_de.objectIds())

    def testReferences(self):
        self.addLanguage('de')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        reftool = self.portal.reference_catalog
        ref = reftool.getReferences(german, 'translationOf')[0]
        self.assertEqual(german, ref.getSourceObject())
        self.assertEqual(english, ref.getTargetObject())
        self.assertEqual(german, english.getBRefs('translationOf')[0])
        self.assertEqual(english, german.getRefs('translationOf')[0])

    def testRenameTranslation(self):
        self.addLanguage('de')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        get_transaction().commit(1)
        self.folder.manage_renameObject(german.getId(), 'german')
        self.failUnless('de' in english.getTranslationLanguages())
        self.assertEqual(english.getTranslation('de').getId(), 'german')

    def testProcessFormNotifyTranslations(self):
        self.addLanguage('de')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        self.failIf(german.outdated())
        # processForm is used for all AT content on updates.
        english.processForm(values={'title':'English'})
        self.failUnless(german.outdated())
        german.processForm(values={'title':'German'})
        self.failIf(german.outdated())
        self.failIf(english.outdated())

    def testCanonicalInvalidateTranslations(self):
        self.addLanguage('de')
        self.addLanguage('fr')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        french = makeTranslation(english, 'fr')
        # Call invalidate on the canonical object
        english.invalidateTranslations()
        self.failIf(english.outdated())
        self.failUnless(german.outdated())
        self.failUnless(french.outdated())

    def testNonCanonicalInvalidateTranslations(self):
        self.addLanguage('de')
        self.addLanguage('fr')
        self.setLanguage('en')
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        french = makeTranslation(english, 'fr')
        # Call invalidate on a translation
        german.invalidateTranslations()
        self.failIf(english.outdated())
        self.failUnless(german.outdated())
        self.failUnless(french.outdated())


class TestLanguageIndependentFields(LinguaPloneTestCase.LinguaPloneTestCase):

    def afterSetUp(self):
        self.addLanguage('de')
        self.setLanguage('en')
        # Speed things up
        self.portal._delObject('portal_catalog')

    def testLanguageIndependentField(self):
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')

        contact = 'Test string'
        english.setTitle('English title')
        english.setContactName(contact)
        german.setTitle('German title')
        self.failIfEqual(english.Title(), german.Title())
        self.assertEqual(english.getContactName(), contact)
        self.assertEqual(english.getRawContactName(), contact)
        self.assertEqual(german.getContactName(), contact)
        self.assertEqual(german.getRawContactName(), contact)
        self.failUnless(english.contactName)
        self.failUnless(german.contactName)
        self.failUnless(hasattr(german, 'testing'))
        self.assertEqual(german.testing, english.contactName)
        self.assertEqual(english.contactName, german.contactName)

        contact = 'First name'
        german.setContactName(contact)
        self.assertEqual(english.getContactName(), contact)
        self.assertEqual(english.getRawContactName(), contact)
        self.assertEqual(german.getContactName(), contact)
        self.assertEqual(german.getRawContactName(), contact)
        self.failUnless(english.contactName)
        self.failUnless(german.contactName)
        self.failUnless(hasattr(english, 'testing'))
        self.assertEqual(english.testing, german.contactName)
        self.assertEqual(english.contactName, german.contactName)

    # Test derived type

    def testBaseSchemaSetup(self):
        schema = dummy.SimpleType.schema
        self.assertEqual(schema['langIndependentInBase'].languageIndependent, 1)
        self.assertEqual(schema['langIndependentInDerived'].languageIndependent, 0)
        self.assertEqual(schema['langIndependentInBoth'].languageIndependent, 1)

    def testDerivedSchemaSetup(self):
        schema = dummy.DerivedType.schema
        self.assertEqual(schema['langIndependentInBase'].languageIndependent, 0)
        self.assertEqual(schema['langIndependentInDerived'].languageIndependent, 1)
        self.assertEqual(schema['langIndependentInBoth'].languageIndependent, 1)

    def testLangIndependentInBase(self):
        english = makeContent(self.folder, 'DerivedType', 'doc')
        german = makeTranslation(english, 'de')
        teststring = 'Test string'
        # Known issue - you can not override a 'true' languageIndependent field
        # from a base class. The derived class sets languageIndependent to 'false',
        # but it is still in effect.
        english.setLangIndependentInBase(teststring)
        ##self.failIfEqual(german.getLangIndependentInBase(), teststring)
        ##self.failIfEqual(german.getRawLangIndependentInBase(), teststring)
        self.assertEqual(german.getLangIndependentInBase(), teststring)
        self.assertEqual(german.getRawLangIndependentInBase(), teststring)

    def testLangIndependentInDerived(self):
        english = makeContent(self.folder, 'DerivedType', 'doc')
        german = makeTranslation(english, 'de')
        teststring = 'Test string'
        # Note that you *can* override a 'false' languageIndependent field
        # from a base class...
        english.setLangIndependentInDerived(teststring)
        self.assertEqual(german.getLangIndependentInDerived(), teststring)
        self.assertEqual(german.getRawLangIndependentInDerived(), teststring)

    def testLangIndependentInBoth(self):
        english = makeContent(self.folder, 'DerivedType', 'doc')
        german = makeTranslation(english, 'de')
        teststring = 'Test string'
        english.setLangIndependentInBoth(teststring)
        self.assertEqual(german.getLangIndependentInBoth(), teststring)
        self.assertEqual(german.getRawLangIndependentInBoth(), teststring)


class TestLanguageLists(LinguaPloneTestCase.LinguaPloneTestCase):

    def afterSetUp(self):
        self.addLanguage('de')
        self.addLanguage('fr')
        self.addLanguage('es')
        self.setLanguage('en')
        # Speed things up
        self.portal._delObject('portal_catalog')

    def testGetTranslatedLanguagesDefault(self):
        # List should contain only the canonical translation
        english = makeContent(self.folder, 'SimpleType', 'doc')
        langs = english.getTranslatedLanguages()
        langs = [l[0] for l in langs]
        self.assertEqual(langs, ['en'])

    def testGetTranslatedLanguages(self):
        # List should contain all existing translations
        english = makeContent(self.folder, 'SimpleType', 'doc')
        makeTranslation(english, 'de')
        makeTranslation(english, 'fr')
        langs = english.getTranslatedLanguages()
        langs = [l[0] for l in langs]
        self.assertEqual(langs, ['de', 'en', 'fr'])

    def testGetUntranslatedLanguagesEmpty(self):
        # List should be emtpy
        english = makeContent(self.folder, 'SimpleType', 'doc')
        makeTranslation(english, 'de')
        makeTranslation(english, 'fr')
        makeTranslation(english, 'es')
        langs = english.getUntranslatedLanguages()
        self.assertEqual(langs, [])

    def testGetUntranslatedLanguages(self):
        # List should contain all translations waiting to be done
        english = makeContent(self.folder, 'SimpleType', 'doc')
        makeTranslation(english, 'de')
        langs = english.getUntranslatedLanguages()
        langs = [l[0] for l in langs]
        self.assertEqual(langs, ['es', 'fr'])

    def testGetDeletableLanguagesEmpty(self):
        # List should be emtpy
        english = makeContent(self.folder, 'SimpleType', 'doc')
        langs = english.getDeletableLanguages()
        self.assertEqual(langs, [])

    def testGetDeletableLanguages(self):
        # List should contain all translations par the canonical one
        english = makeContent(self.folder, 'SimpleType', 'doc')
        makeTranslation(english, 'de')
        makeTranslation(english, 'fr')
        langs = english.getDeletableLanguages()
        langs = [l[0] for l in langs]
        self.assertEqual(langs, ['de', 'fr'])


class TestDeleteTranslations(LinguaPloneTestCase.LinguaPloneTestCase):

    def afterSetUp(self):
        self.addLanguage('de')
        self.setLanguage('en')
        # Speed things up
        self.portal._delObject('portal_catalog')

    def testDeleteTranslationClearsReference(self):
        # Deleting a translation object should remove 
        # the relation in the canonical object.
        english = makeContent(self.folder, 'SimpleType', 'doc')
        german = makeTranslation(english, 'de')
        self.assertEqual(len(english.getTranslations()), 2)
        self.folder._delObject(german.getId())
        self.assertEqual(len(english.getTranslations()), 1)

    def testDeleteCanonical(self):
        # Deletion should be possible if there are no translations
        english = makeContent(self.folder, 'SimpleType', 'doc')
        self.folder._delObject('doc')

    if config.CANONICAL_DELETE_PROTECTION:

        def testDeleteCanonicalWithTranslations(self):
            # Must delete translations first
            english = makeContent(self.folder, 'SimpleType', 'doc')
            makeTranslation(english, 'de')
            self.assertRaises(BeforeDeleteException, 
                              self.folder._delObject, 'doc')

        def testDeleteCanonicalWithTranslationsFolder(self):
            # Must delete translations first
            english = makeContent(self.folder, 'SimpleFolder', 'foo')
            makeTranslation(english, 'de')
            self.assertRaises(BeforeDeleteException, 
                              self.folder._delObject, 'foo')

        def testDeleteCanonicalWithTranslationsOrderedFolder(self):
            # Must delete translations first
            english = makeContent(self.folder, 'OrderedFolder', 'foo')
            makeTranslation(english, 'de')
            self.assertRaises(BeforeDeleteException, 
                              self.folder._delObject, 'foo')

        def testDeleteCanonicalWithTranslationsBTreeFolder(self):
            # Must delete translations first
            english = makeContent(self.folder, 'BTreeFolder', 'foo')
            makeTranslation(english, 'de')
            self.assertRaises(BeforeDeleteException, 
                              self.folder._delObject, 'foo')

        def testBulkDeleteCanonical(self):
            # Should be able to delete if translations are deleted as well
            english = makeContent(self.folder, 'SimpleType', 'doc')
            german = makeTranslation(english, 'de')
            self.folder.manage_delObjects(['doc', german.getId()])

        # XXX: Fails :-(
        def DISABLED_testBulkDeleteCanonicalReverseOrder(self):
            # Should be able to delete if translations are deleted as well
            english = makeContent(self.folder, 'SimpleType', 'doc')
            german = makeTranslation(english, 'de')
            self.folder.manage_delObjects([german.getId(), 'doc'])

        # XXX: Fails :-(
        def DISABLED_testBulkDeleteCanonicalParentFolder(self):
            # Should be able to delete the parent folder
            english = makeContent(self.folder, 'SimpleType', 'doc')
            german = makeTranslation(english, 'de')
            self.folder.Members.manage_delObjects(self.folder.getId())


class TestMultilingualCatalog(LinguaPloneTestCase.LinguaPloneTestCase):

    def afterSetUp(self):
        self.addLanguage('de')
        self.setLanguage('en')
        self.english = makeContent(self.folder, 'SimpleType', 'doc')
        self.english.edit(title='Foo')
        self.german = makeTranslation(self.english, 'de')
        self.german.edit(title='Foo')
        self.catalog = self.portal.portal_catalog

    def testSearchEnglish(self):
        search = self.catalog(Language='en')
        self.failUnless(self.english in [x.getObject() for x in search])
        self.failIf(self.german in [x.getObject() for x in search])

    def testSearchGerman(self):
        search = self.catalog(Language='de')
        self.failIf(self.english in [x.getObject() for x in search])
        self.failUnless(self.german in [x.getObject() for x in search])
    
    def testSearchAll(self):
        search = self.catalog(Language='all')
        self.failUnless(self.english in [x.getObject() for x in search])
        self.failUnless(self.german in [x.getObject() for x in search])

    def testSearchTitle(self):
        search = self.catalog(Title='Foo')  # default Language=['en', 'neutral']
        self.failUnless(self.english in [x.getObject() for x in search])
        self.failIf(self.german in [x.getObject() for x in search])

    def testSearchAllAndTitle(self):
        search = self.catalog(Language='all', Title='Foo')
        self.failUnless(self.english in [x.getObject() for x in search])
        self.failUnless(self.german in [x.getObject() for x in search])

    def testSearchAllAndBadTitle(self):
        search = self.catalog(Language='all', Title='Bar')
        self.failIf(self.english in [x.getObject() for x in search])
        self.failIf(self.german in [x.getObject() for x in search])

    def testSearchGermanAndTitle(self):
        search = self.catalog(Language='de', Title='Foo')
        self.failIf(self.english in [x.getObject() for x in search])
        self.failUnless(self.german in [x.getObject() for x in search])

    def testMultipleLanguages(self):
        search = self.catalog(Language=['en', 'de'])
        self.failUnless(self.english in [x.getObject() for x in search])
        self.failUnless(self.german in [x.getObject() for x in search])

    def testMultipleLanguagesAndTitle(self):
        search = self.catalog(Language=['en', 'de'], Title='Foo')
        self.failUnless(self.english in [x.getObject() for x in search])
        self.failUnless(self.german in [x.getObject() for x in search])

    def testSearchNeutral(self):
        search = self.catalog(Language='neutral')
        self.failIf(self.english in [x.getObject() for x in search])
        self.failIf(self.german in [x.getObject() for x in search])

    def testSearchGermanAndNeutral(self):
        search = self.catalog(Language=['de', 'neutral'])
        self.failIf(self.english in [x.getObject() for x in search])
        self.failUnless(self.german in [x.getObject() for x in search])


def test_suite():
    from unittest import TestSuite, makeSuite
    suite = TestSuite()
    suite.addTest(makeSuite(TestLinguaPlone))
    suite.addTest(makeSuite(TestLanguageIndependentFields))
    suite.addTest(makeSuite(TestLanguageLists))
    suite.addTest(makeSuite(TestDeleteTranslations))
    suite.addTest(makeSuite(TestMultilingualCatalog))
    return suite

if __name__ == '__main__':
    framework()
