#
# An example ZopeTestCase testing a PythonScript in the default fixture.
#
# Note that you are encouraged to call any of the following methods
# from your own tests to modify the test user's security credentials:
#
#   - setRoles()
#   - setPermissions()
#   - login()
#   - logout()
#

# $Id: testPythonScript.py,v 1.6 2003/11/01 18:48:18 shh42 Exp $

import os, sys
if __name__ == '__main__':
    execfile(os.path.join(sys.path[0], 'framework.py'))

from Testing import ZopeTestCase

ZopeTestCase.installProduct('PythonScripts')

standard_permissions = ZopeTestCase.standard_permissions
access_permissions   = ['View management screens']
change_permissions   = ['Change Python Scripts']
all_permissions      = standard_permissions+access_permissions+change_permissions

params1 = 'a=1'
body1   = 'return a'
params2 = 'a'
body2   = 'return a+1'

from AccessControl import getSecurityManager, Unauthorized


class TestPythonScript(ZopeTestCase.ZopeTestCase):
    '''Tries various things allowed by the ZopeTestCase API.'''

    def afterSetUp(self):
        '''Adds a PythonScript object to the default fixture'''
        dispatcher = self.folder.manage_addProduct['PythonScripts']
        dispatcher.manage_addPythonScript('ps')
        self.ps = self.folder['ps']
        self.ps.ZPythonScript_edit(params1, body1)

    # Test the fixture ##############################################

    def test01(self):
        # The PythonScript should exist and be properly set up
        self.failUnless(hasattr(self.folder, 'ps'))
        self.assertEqual(self.ps.body(), body1+'\n')
        self.assertEqual(self.ps.params(), params1)
        owner = self.ps.getOwner()
        self.assertEqual(owner.getUserName(), ZopeTestCase.user_name)

    # Test callability ##############################################

    def testCanCall(self):
        # PythonScript should return 1
        self.assertEqual(self.ps(), 1)

    def testCannotCall(self):
        # PythonScript should raise a TypeError
        self.ps.ZPythonScript_edit(params2, body2)
        self.assertRaises(TypeError, self.ps, ())

    def testCanCall2(self):
        # PythonScript should return 2
        self.ps.ZPythonScript_edit(params2, body2)
        self.assertEqual(self.ps(1), 2)

    # Test edit protection ##########################################

    def testCanEdit(self):
        # PythonScript should be editable
        self.setPermissions(all_permissions)
        self.ps.restrictedTraverse('ZPythonScript_edit')(params2, body2)
        self.assertEqual(self.ps.body(), body2+'\n')
        self.assertEqual(self.ps.params(), params2)

    def testManagerCanEdit(self):
        # PythonScript should be editable for Managers
        self.setRoles(['Manager'])
        self.ps.restrictedTraverse('ZPythonScript_edit')(params2, body2)
        self.assertEqual(self.ps.body(), body2+'\n')
        self.assertEqual(self.ps.params(), params2)

    def testCannotEdit(self):
        # PythonScript should not be editable
        self.setPermissions(standard_permissions+access_permissions)
        try:
            self.ps.restrictedTraverse('ZPythonScript_edit')(params2, body2)
        except Unauthorized:
            pass    # test passed
        else:
            self.assertEqual(self.ps.body(), body2+'\n', 'Method was unprotected')
            self.assertEqual(self.ps.body(), body1+'\n', 'Method was protected but no exception was raised')

    def testLoggedOutCannotEdit(self):
        # PythonScript should not be editable
        self.logout()
        try:
            self.ps.restrictedTraverse('ZPythonScript_edit')(params2, body2)
        except Unauthorized:
            pass    # test passed
        else:
            self.assertEqual(self.ps.body(), body2+'\n', 'Method was unprotected')
            self.assertEqual(self.ps.body(), body1+'\n', 'Method was protected but no exception was raised')

    # Test access protection ########################################

    def testAccessAllowed(self):
        # manage_main should be accessible
        self.setPermissions(standard_permissions+access_permissions)
        try:
            dummy = self.ps.restrictedTraverse('manage_main')
        except Unauthorized:
            self.fail('Access to manage_main was denied')

    def testManagerAccessAllowed(self):
        # manage_main should be accessible to Managers
        self.setRoles(['Manager'])
        try:
            dummy = self.ps.restrictedTraverse('manage_main')
        except Unauthorized:
            self.fail('Access to manage_main was denied to Manager')

    def testAccessDenied(self):
        # manage_main should be protected
        self.setPermissions(standard_permissions)
        self.assertRaises(Unauthorized, self.ps.restrictedTraverse, 'manage_main')

    # Test access protection with SecurityManager ###################

    def testSecurityManagerAllowed(self):
        # SecurityManager should allow access
        self.setPermissions(standard_permissions+access_permissions)
        security = getSecurityManager()
        try:
            security.validateValue(self.ps.manage_main)
        except Unauthorized:
            self.fail('Access to manage_main was denied')

    def testSecurityManagerAllowed2(self):
        # SecurityManager should allow access
        self.setRoles(['Manager'])
        security = getSecurityManager()
        try:
            security.validateValue(self.ps.manage_main)
        except Unauthorized:
            self.fail('Access to manage_main was denied')

    def testSecurityManagerDenied(self):
        # SecurityManager should deny access
        self.setPermissions(standard_permissions)
        security = getSecurityManager()
        self.assertRaises(Unauthorized, security.validateValue, self.ps.manage_main)


if __name__ == '__main__':
    framework()
else:
    # While framework.py provides its own test_suite()
    # method the testrunner utility does not.
    from unittest import TestSuite, makeSuite
    def test_suite():
        suite = TestSuite()
        suite.addTest(makeSuite(TestPythonScript))
        return suite

