/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: packsock.c,v 1.36.2.8 2003/12/02 12:12:32 sasa Exp $
 *
 * Author  : yeti
 * Auditor : bazsi
 * Last audited version: 1.39
 * Notes:
 *
 ***************************************************************************/

#include <zorp/packsock.h>
#include <zorp/log.h>

#if ENABLE_CONNTRACK
#include <zorp/sockaddr.h>
#include <zorp/socket.h>
#include <zorp/zorp.h>
#include <zorp/io.h>
#include <zorp/stream.h>
#include <zorp/cap.h>
#include <zorp/sysdep.h>

#include <time.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <sys/uio.h>
#endif

typedef struct _ZPacksockFuncs
{
  gint (*open)(guint flags, ZSockAddr *remote, ZSockAddr *local, GError **error);
  GIOStatus (*recv)(gint fd, ZPacket **pack, ZSockAddr **from, ZSockAddr **to, GError **error);
  GIOStatus (*read)(gint fd, ZPacket **pack, GError **error);
  GIOStatus (*write)(gint fd, ZPacket *pack, GError **error);
} ZPacksockFuncs;

#if ENABLE_LINUX22_TPROXY || ENABLE_NETFILTER_LINUX22_FALLBACK

gint
z_l22_packsock_open(guint flags, ZSockAddr *remote, ZSockAddr *local, GError **error G_GNUC_UNUSED)
{
  gint fd;
  gint tmp = 1;
  
  z_enter();
  fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  
  if (fd < 0)
    {
      z_log(NULL, CORE_ERROR, 3, "Error opening socket; error='%m'");
      close(fd);
      z_leave();
      return -1;
    }
  setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &tmp, sizeof(tmp));
  tmp = 1;
  setsockopt(fd, SOL_SOCKET, SO_BROADCAST, &tmp, sizeof(tmp));
  
  if (flags & ZPS_LISTEN)
    {
      if (z_bind(fd, local) != G_IO_STATUS_NORMAL)
        {
          /* z_bind already issued a log message */
          z_leave();
          return -1;
        }
    }
  else if (flags & ZPS_ESTABLISHED)
    {
      if (local && z_bind(fd, local) != G_IO_STATUS_NORMAL)
        {
          close(fd);
          z_leave();
          return -1;
        }
      if (connect(fd, &remote->sa, remote->salen) == -1)
        {
          z_log(NULL, CORE_ERROR, 3, "Error connecting UDP socket (l22); error='%m'");
          close(fd);
          z_leave();
          return -1;
        }
        
    }
  return fd;
}

GIOStatus
z_l22_packsock_recv(gint fd, ZPacket **packet, 
                    ZSockAddr **from_addr, ZSockAddr **to_addr, 
                    GError **error G_GNUC_UNUSED)
{
  struct sockaddr_in from;
  gchar buf[65536], ctl_buf[64];
  struct msghdr msg;
  struct iovec iov;
            
  gint rc;
  
  z_enter();

  memset(&msg, 0, sizeof(msg));
  msg.msg_name = &from;
  msg.msg_namelen = sizeof(from);
  msg.msg_controllen = sizeof(ctl_buf);
  msg.msg_control = ctl_buf;
  msg.msg_iovlen = 1;
  msg.msg_iov = &iov;
  iov.iov_base = buf;
  iov.iov_len = sizeof(buf);

  do
    {                
      rc = recvmsg(fd, &msg, MSG_PROXY);
      
    }
  while (rc < 0 && errno == EINTR);
  
  if (rc < 0)
    {
      z_leave();
      return errno == EAGAIN ? G_IO_STATUS_AGAIN : G_IO_STATUS_ERROR;
    }
  *packet = z_packet_new();
  
  z_packet_set_data(*packet, buf, rc);
  *from_addr = z_sockaddr_inet_new2(&from);
  
  if (to_addr)
    {
      
      if (((struct sockaddr_in *) &from.sin_zero)->sin_family)
        {
          *to_addr = z_sockaddr_inet_new2((struct sockaddr_in *) &from.sin_zero);
        }
      else
        {
          struct sockaddr_in to;
          socklen_t tolen = sizeof(to);
          
          getsockname(fd, (struct sockaddr *) &to, &tolen);
          *to_addr = z_sockaddr_inet_new2(&to);
        }
    }

  z_leave();
  return G_IO_STATUS_NORMAL;
  
}

/* destination address is implicit, specified by socket creation */
GIOStatus
z_l22_packsock_write(gint fd, ZPacket *packet, GError **error G_GNUC_UNUSED)
{
  int rc;

  z_enter();
  do
    {
      rc = send(fd, packet->data, packet->length, 0);
    }
  while (rc < 0 && errno == EINTR);
  if (rc < 0)
    {
      if (errno == EAGAIN)
        rc = G_IO_STATUS_AGAIN;
      else
        rc = G_IO_STATUS_ERROR;
    }
  else
    rc = G_IO_STATUS_NORMAL;
    
  z_leave();
  return rc;
}

GIOStatus
z_l22_packsock_read(gint fd, ZPacket **packet, GError **error G_GNUC_UNUSED)
{
  gchar buf[65536];
  GIOStatus res;
  gint rc;

  z_enter();
  
  do
    {
      rc = recv(fd, buf, sizeof(buf), 0);
    }
  while (rc < 0 && errno == EINTR);
  
  if (rc < 0)
    {
      if (errno == EAGAIN)
        res = G_IO_STATUS_AGAIN;
      else
        res = G_IO_STATUS_ERROR;
    }
  else
    {
      res = G_IO_STATUS_NORMAL;
      *packet = z_packet_new();
      z_packet_set_data(*packet, buf, rc);
    }

  z_leave();
  return res;
}

ZPacksockFuncs z_l22_packsock_funcs = 
{
  z_l22_packsock_open,
  z_l22_packsock_recv,
  z_l22_packsock_read,
  z_l22_packsock_write
};

#endif

#if ENABLE_NETFILTER_TPROXY

#include <zorp/nfiptproxy-kernel.h>

#ifndef IP_RECVORIGADDRS

#define IP_RECVORIGADDRS 16
#define IP_ORIGADDRS     17

struct in_origaddrs {
        struct in_addr ioa_srcaddr;
        struct in_addr ioa_dstaddr;
        unsigned short int ioa_srcport;
        unsigned short int ioa_dstport;
};

#endif

gint
z_nf_packsock_open(guint flags, ZSockAddr *remote, ZSockAddr *local, GError **error G_GNUC_UNUSED)
{
  gint fd;
  gint tmp = 1;
  
  z_enter();
  fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  
  if (fd < 0)
    {
      z_log(NULL, CORE_ERROR, 3, "Error creating socket; error='%m'");
      close(fd);
      z_leave();
      return -1;
    }
  setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &tmp, sizeof(tmp));
  tmp = 1;
  setsockopt(fd, SOL_SOCKET, SO_BROADCAST, &tmp, sizeof(tmp));
  
  if (flags & ZPS_LISTEN)
    {
      if (setsockopt(fd, SOL_IP, IP_RECVORIGADDRS, &tmp, sizeof(tmp)) < 0)
        {
          /* FIXME: log error */
          z_log(NULL, CORE_ERROR, 3, "Error during setsockopt(SOL_IP, IP_RECVORIGADDRS); error='%m'");
          close(fd);
          z_leave();
          return -1;
        }
      if (z_bind(fd, local) != G_IO_STATUS_NORMAL)
        {
          /* z_bind already issued a log message */
          z_leave();
          return -1;
        }
      tmp = ITP_LISTEN | ITP_UNIDIR;
      if (setsockopt(fd, SOL_IP, IP_TPROXY_FLAGS, &tmp, sizeof(tmp)) < 0)
        {
          /* FIXME: error message */
          z_log(NULL, CORE_ERROR, 3, "Error during setsockopt(SOL_IP, IP_TPROXY_FLAGS, ITP_LISTEN | ITP_UNIDIR); error='%m'");
          close(fd);
          z_leave();
          return -1;
        }
    }
  else if (flags & ZPS_ESTABLISHED)
    {

      if (local && z_bind(fd, local) != G_IO_STATUS_NORMAL)
        {
          close(fd);
          z_leave();
          return -1;
        }

      /* NOTE: we use connect instead of z_connect, as we do tproxy calls ourselves */
      if (connect(fd, &remote->sa, remote->salen) < 0)
        {
          z_log(NULL, CORE_ERROR, 3, "Error connecting UDP socket (nf); error='%m'");
          close(fd);
          z_leave();
          return -1;
        }
      
      if (!local)
        {
          struct sockaddr disconnect;
          
          if (z_getsockname(fd, &local) != G_IO_STATUS_NORMAL)
            {
              z_log(NULL, CORE_ERROR, 3, "Error getting dynamic local address (nf); error='%m'");
              close(fd);
              z_leave();
              return -1;
            }
          disconnect.sa_family = AF_UNSPEC;
          if (connect(fd, &disconnect, sizeof(disconnect)) < 0)
            {
              z_log(NULL, CORE_ERROR, 3, "Error unconnecting for dynamic binding (nf); error='%m'");
              z_sockaddr_unref(local);
              close(fd);
              z_leave();
              return -1;
            }
          if (z_bind(fd, local) != G_IO_STATUS_NORMAL)
            {
              z_sockaddr_unref(local);
              close(fd);
              z_leave();
              return -1;
            }
          z_sockaddr_unref(local);
          local = NULL;
          if (connect(fd, &remote->sa, remote->salen) < 0)
            {
              z_log(NULL, CORE_ERROR, 3, "Error reconnecting UDP socket (nf); error='%m'");
              close(fd);
              z_leave();
              return -1;
            }
        }
        
      tmp = ITP_ESTABLISHED;
      if (setsockopt(fd, SOL_IP, IP_TPROXY_FLAGS, &tmp, sizeof(tmp)) < 0)
        {
          z_log(NULL, CORE_ERROR, 3, "Error during setsockopt(SOL_IP, IP_TPROXY_FLAGS, ITP_ESTABLISHED); error='%m'");
          close(fd);
          z_leave();
          return -1;
        }
    }
  return fd;
}

GIOStatus
z_nf_packsock_recv(gint fd, ZPacket **packet, ZSockAddr **from_addr, ZSockAddr **to_addr, GError **error G_GNUC_UNUSED)
{
  struct sockaddr_in from, to;
  gchar buf[65536], ctl_buf[64];
  struct msghdr msg;
  struct cmsghdr *cmsg;
  struct iovec iov;
            
  gint rc;
  
  z_enter();

  memset(&msg, 0, sizeof(msg));
  msg.msg_name = &from;
  msg.msg_namelen = sizeof(from);
  msg.msg_controllen = sizeof(ctl_buf);
  msg.msg_control = ctl_buf;
  msg.msg_iovlen = 1;
  msg.msg_iov = &iov;
  iov.iov_base = buf;
  iov.iov_len = sizeof(buf);
                
  do
    {
      rc = recvmsg(fd, &msg, 0);
    }
  while (rc < 0 && errno == EINTR);
  
  if (rc < 0)
    {
      z_leave();
      return errno == EAGAIN ? G_IO_STATUS_AGAIN : G_IO_STATUS_ERROR;
    }
  *packet = z_packet_new();
  
  z_packet_set_data(*packet, buf, rc);
  *from_addr = z_sockaddr_inet_new2(&from);
  
  if (to_addr)
    {
      *to_addr = NULL;
      for (cmsg = CMSG_FIRSTHDR(&msg); cmsg; cmsg = CMSG_NXTHDR(&msg,cmsg))
        {
          if (cmsg->cmsg_level == SOL_IP && cmsg->cmsg_type == IP_ORIGADDRS)
            {
              struct in_origaddrs *ioa = (struct in_origaddrs *) CMSG_DATA(cmsg);

	      to.sin_family = AF_INET;
	      to.sin_addr = ioa->ioa_dstaddr;
	      to.sin_port = ioa->ioa_dstport;
	      *to_addr = z_sockaddr_inet_new2(&to);
	      break;
	    }
	}
      if (*to_addr == NULL)
        {
          struct sockaddr_in to;
          socklen_t tolen = sizeof(to);
                             
          getsockname(fd, (struct sockaddr *) &to, &tolen);
          *to_addr = z_sockaddr_inet_new2(&to);
        }
    }

  z_leave();
  return G_IO_STATUS_NORMAL;
  
}

/* destination address is implicit, specified by socket creation */
GIOStatus
z_nf_packsock_write(gint fd, ZPacket *packet, GError **error G_GNUC_UNUSED)
{
  int rc;

  z_enter();
  do
    {
      rc = send(fd, packet->data, packet->length, 0);
    }
  while (rc < 0 && errno == EINTR);
  if (rc < 0)
    {
      z_leave();
      return errno == EAGAIN ? G_IO_STATUS_AGAIN : G_IO_STATUS_ERROR;
    }
  z_leave();
  return G_IO_STATUS_NORMAL;
}

GIOStatus 
z_nf_packsock_read(gint fd, ZPacket **packet, GError **error G_GNUC_UNUSED)
{
  gchar buf[65536];
  gint rc;

  z_enter();
  
  do
    {
      rc = recv(fd, buf, sizeof(buf), 0);
    }
  while (rc < 0 && errno == EINTR);
  
  if (rc < 0)
    {
      z_leave();
      return errno == EAGAIN ? G_IO_STATUS_AGAIN : G_IO_STATUS_ERROR;
    }
  *packet = z_packet_new();
  z_packet_set_data(*packet, buf, rc);

  z_leave();
  return G_IO_STATUS_NORMAL;
}

ZPacksockFuncs z_nf_packsock_funcs = 
{
  z_nf_packsock_open,
  z_nf_packsock_recv,
  z_nf_packsock_read,
  z_nf_packsock_write
};

#endif

ZPacksockFuncs *packsock_funcs;

gint 
z_packsock_open(guint flags, ZSockAddr *remote, ZSockAddr *local, GError **error)
{
  return packsock_funcs->open(flags, remote, local, error);
}

GIOStatus 
z_packsock_recv(gint fd, ZPacket **pack, ZSockAddr **from, ZSockAddr **to, GError **error)
{
  return packsock_funcs->recv(fd, pack, from, to, error);
}

GIOStatus 
z_packsock_read(gint fd, ZPacket **pack, GError **error)
{
  return packsock_funcs->read(fd, pack, error);
}

GIOStatus 
z_packsock_write(gint fd, ZPacket *pack, GError **error)
{
  return packsock_funcs->write(fd, pack, error);
}

gboolean
z_packsock_init(gint sysdep_tproxy)
{
  switch (sysdep_tproxy)
    {
#if ENABLE_LINUX22_TPROXY || ENABLE_NETFILTER_LINUX22_FALLBACK
    case Z_SD_TPROXY_LINUX22:
      packsock_funcs = &z_l22_packsock_funcs;
      break;
#endif
#if ENABLE_NETFILTER_TPROXY
    case Z_SD_TPROXY_NETFILTER:
      packsock_funcs = &z_nf_packsock_funcs;
      break;
#endif
    default:
      z_log(NULL, CORE_ERROR, 0, "Required transparency support not compiled in (UDP); sysdep_tproxy='%d'", sysdep_tproxy);
      return FALSE;
    }
  return TRUE;
} 
