/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: pyattach.c,v 1.4.2.7 2003/07/29 16:45:54 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/pyattach.h>
#include <zorp/attach.h>
#include <zorp/log.h>
#include <zorp/policy.h>
#include <zorp/pysockaddr.h>
#include <zorp/freeq.h>
#include <zorp/socket.h>
#include <zorp/stream.h>
#include <zorp/streamfd.h>
#include <zorp/pystream.h>

/*
 * struct ZorpAttach
 *
 * Author:  Bazsi, 2000/03/27
 * Purpose: this class encapsulates a connector
 *
 */

typedef struct _ZorpAttach
{
  PyObject_HEAD
  ZPolicy *policy;
  ZAttach *attach;
  ZSockAddr *local;
  PyObject *handler;
} ZorpAttach;

static PyObject *z_py_zorp_attach_new_instance(PyObject *s, PyObject *args);
static void z_py_zorp_attach_free(ZorpAttach *self);
static PyObject *z_py_zorp_attach_getattr(PyObject *o, char *name);

static PyObject *z_py_zorp_attach_start_method(ZorpAttach *self, PyObject *args);
static PyObject *z_py_zorp_attach_block_method(ZorpAttach *self, PyObject *args);
static PyObject *z_py_zorp_attach_cancel_method(ZorpAttach *self, PyObject *args);


PyMethodDef z_py_zorp_attach_funcs[] =
{
  { "Attach",  z_py_zorp_attach_new_instance, METH_VARARGS, NULL },
  { NULL,      NULL, 0, NULL }   /* sentinel*/
};

static PyMethodDef py_zorp_attach_methods[] =
{
  { "start",       (PyCFunction) z_py_zorp_attach_start_method, 0, NULL },
  { "block",       (PyCFunction) z_py_zorp_attach_block_method, 0, NULL },
  { "cancel",      (PyCFunction) z_py_zorp_attach_cancel_method, 0, NULL },
  { NULL,          NULL, 0, NULL }   /* sentinel*/
};

PyTypeObject z_py_zorp_attach_type = 
{
  PyObject_HEAD_INIT(&z_py_zorp_attach_type)
  0,
  "ZorpAttach",
  sizeof(ZorpAttach),
  0,
  (destructor) z_py_zorp_attach_free,
  0,
  (getattrfunc) z_py_zorp_attach_getattr,
  0,
  0,
  0, /*(reprfunc) z_py_zorp_attach_repr,*/
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  "ZorpAttach class for Zorp",
  0, 0, 0, 0
};

/* called by the low level connector, under the main thread */
static void
z_py_zorp_attach_connected(ZConnection *conn, gpointer user_data)
{
  ZorpAttach *self = (ZorpAttach *) user_data;
  PyObject *res, *sobj;

  z_enter();

  if (!self->handler)
    {
      /* we have already been cancelled */
      z_connection_destroy(conn, TRUE);
      z_leave();
      return;
    }
    
  z_policy_acquire_main(self->policy);
 
  if (self->handler)
    {
      if (conn)
        sobj = z_py_stream_new(conn->stream);
      else 
        {
          Py_XINCREF(Py_None);
          sobj = Py_None;
        }
      res = PyEval_CallFunction(self->handler, "(O)", sobj);
      if (!res)
        {
	  PyErr_Print();
	  z_stream_close(conn->stream, NULL);
        }
      Py_XDECREF(sobj);
    }
  else
    z_stream_close(conn->stream, NULL);
  z_policy_release_main(self->policy);
  z_connection_destroy(conn, FALSE);
  z_leave();
}

static void
z_py_zorp_attach_unref(PyObject *attach)
{
  ZPolicy *policy;
  
  policy = z_policy_ref(((ZorpAttach *) attach)->policy);
  
  z_policy_acquire_main(policy);
  Py_XDECREF(attach);
  z_policy_release_main(policy);
}

static PyObject *
z_py_zorp_attach_start_method(ZorpAttach *self, PyObject *args G_GNUC_UNUSED)
{
  gboolean res;
  z_enter();

  if (self->attach)
    {
      Py_BEGIN_ALLOW_THREADS;
      res = z_attach_start(self->attach);
      Py_END_ALLOW_THREADS;
      if (res)
        {
          self->local = z_attach_get_local(self->attach);
        }
      else
        {
          PyErr_SetString(PyExc_IOError, "Error in z_attach_start");
          return NULL;
        }
    }

  Py_XINCREF(Py_None);
  z_leave();
  return Py_None;
}

static PyObject *
z_py_zorp_attach_block_method(ZorpAttach *self, PyObject *args G_GNUC_UNUSED)
{
  z_enter();
  if (self->handler == Py_None)
    {
      PyObject *res;
      ZConnection *conn;
      
      Py_BEGIN_ALLOW_THREADS
      conn = z_attach_block(self->attach);
      Py_END_ALLOW_THREADS
      
      if (conn != NULL)
        {
          /* Note: we don't assign a name to this stream now, it will be assigned later */
          res = z_py_stream_new(conn->stream);
          z_connection_destroy(conn, FALSE);
        }
      else
        {
          Py_XINCREF(Py_None);
          res = Py_None;
        }
      z_leave();
      return res;
    }
  PyErr_SetString(PyExc_ValueError, "ZorpAttach.block called with handler != None");
  z_leave();
  return NULL;
}


/* called from Python to destroy this instance */
static PyObject *
z_py_zorp_attach_cancel_method(ZorpAttach *self, PyObject *args G_GNUC_UNUSED)
{
  PyObject *handler;
  
  z_enter();

  /* this would race with _connected, but we are protected by the interpreter lock */

  z_attach_cancel(self->attach);

  /* handler being null indicates we have been cancelled */  
  /* NOTE: self->handler is protected by the interpreter lock */
  
  handler = self->handler;
  self->handler = NULL;
  Py_XDECREF(handler); 

  Py_XINCREF(Py_None);
  z_leave();
  return Py_None;
}

/* called automatically to free up this instance */
static void
z_py_zorp_attach_free(ZorpAttach *self)
{
  z_enter();
  if (self->handler)
    {
      Py_XDECREF(self->handler);
    }
  if (self->attach)
    {
      z_attach_unref(self->attach);
    }
  z_sockaddr_unref(self->local);

  PyMem_DEL(self);
  z_leave();
}

static PyObject *
z_py_zorp_attach_getattr(PyObject *o, char *name)
{
  ZorpAttach *self = (ZorpAttach *) o;
  PyObject * back;
  z_enter();
  
  if (strcmp(name, "local") == 0)
    {
      if (self->local)
        {
          back = z_py_zorp_sockaddr_new(self->local);
          z_leave();
          return back;
        }
      else
        {
          Py_XINCREF(Py_None);
          z_leave();
          return Py_None;
        }
    }
  else
    {
      back = Py_FindMethod(py_zorp_attach_methods, o, name);
      z_leave();
      return back;
    }
}

static PyObject *
z_py_zorp_attach_new_instance(PyObject *s G_GNUC_UNUSED, PyObject *args)
{
  ZorpAttach *self;
  ZorpSockAddr *local, *remote;
  PyObject *handler, *keywords, *fake_args;
  ZAttachParams params;
  static gchar *tcp_keywords[] = { "timeout", NULL };
  static gchar *udp_keywords[] = { "tracker", NULL };
  gchar buf1[MAX_SOCKADDR_STRING], buf2[MAX_SOCKADDR_STRING], *session_id;
  guint protocol;

  z_enter();
  /* called by python, no need to lock the interpreter */
   
  if (!PyArg_ParseTuple(args, "siOOO|O", &session_id, &protocol, &local, &remote, &handler, &keywords))
    return NULL;

  if (handler != Py_None && !PyCallable_Check(handler))
    {
      PyErr_SetString(PyExc_TypeError, "Handler parameter must be callable");
      z_leave();
      return NULL;
    }
  if ((((PyObject *) local != Py_None) && !z_py_zorp_sockaddr_check(local)) ||
      (!z_py_zorp_sockaddr_check(remote)))
    {
      PyErr_SetString(PyExc_TypeError, "Local and remote arguments must be ZorpSockAddr or None");
      z_leave();
      return NULL;
    }

  fake_args = PyTuple_New(0);

  switch (protocol)
    {
    case ZD_PROTO_TCP:
      params.tcp.timeout = 30;
      if (!PyArg_ParseTupleAndKeywords(fake_args, keywords, "|i", tcp_keywords, &params.tcp.timeout))
        {
          Py_XDECREF(fake_args);
          return NULL;
        }
      break;
    case ZD_PROTO_UDP:

      /* NOTE: params.udp.tracker is a (gchar *) valid only as long as we
       * return, it is discarded by Python afterwards.  This is not a
       * problem as this name is copied to a private location, and never
       * referenced again */
       
      params.udp.tracker = NULL;
      if (!PyArg_ParseTupleAndKeywords(fake_args, keywords, "|s", udp_keywords, &params.udp.tracker))
        {
          Py_XDECREF(fake_args);
          return NULL;
        }
      break;
      
    }
  Py_XDECREF(fake_args);


  /*LOG
    This message indicates that Zorp began establishing connection
    with the indicated remote host.
   */
  z_log(NULL, CORE_DEBUG, 7, "Connecting to remote host; protocol='%d', local='%s', remote='%s'", 
        protocol,
	(PyObject *) local != Py_None ? z_sockaddr_format(local->sa, buf1, sizeof(buf1)) : "NULL",
	z_sockaddr_format(remote->sa, buf2, sizeof(buf2)));

  self = PyObject_NEW(ZorpAttach, &z_py_zorp_attach_type);
  if (!self)
    {
      z_leave();
      return NULL;
    }
  
  /* NOTE: this pointer is used as a mark to indicate that connection has
   * not succeeded yet. Every reference to self->conn _MUST_ check this
   * value prior to using the value in _ANY_ way
   */

  self->local = NULL;
  self->policy = NULL;
  self->handler = NULL;
        
  /* z_attach_new stores a reference to self, which is freed by z_py_zorp_attach_unref */
  
  if (handler != Py_None)
    {
      self->attach = z_attach_new(session_id, protocol, (PyObject *) local == Py_None ? NULL : local->sa, remote->sa, 
                                  &params,
                                  z_py_zorp_attach_connected, 
                                  self, (GDestroyNotify) z_py_zorp_attach_unref);
      Py_XINCREF((PyObject *) self); /* returned reference */
    }
  else
    self->attach = z_attach_new(session_id, protocol, (PyObject *) local == Py_None ? NULL : local->sa, remote->sa, 
                                &params,
                                NULL, NULL, NULL); 
  if (!self->attach)
    {
      PyErr_SetString(PyExc_IOError, "Error during connect");
      
      Py_XDECREF(self);
      if (handler != Py_None)
        {
          Py_XDECREF(self);
        }
      z_leave();
      return NULL;
    }
  Py_XINCREF(handler);
  self->handler = handler;
  self->policy = z_policy_ref(current_policy);

  z_leave();
  return (PyObject *) self;
}

void
z_py_zorp_attach_init(void)
{
  /* PyImport_AddModule("Zorp.Zorp"); */
  Py_InitModule("Zorp.Zorp", z_py_zorp_attach_funcs);
}
