/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: http.c,v 1.124.2.12 2004/05/14 14:33:12 bazsi Exp $
 * 
 * Author: Balazs Scheidler <bazsi@balabit.hu>
 * Auditor: 
 * Last audited version: 
 * Notes:
 *   Based on the code by: Viktor Peter Kovacs <vps__@freemail.hu>
 *   
 ***************************************************************************/

#include "http.h"

#include <zorp/thread.h>
#include <zorp/registry.h>
#include <zorp/log.h>
#include <zorp/policy.h>
#include <zorp/authprovider.h>
#include <zorp/misc.h>

#include <ctype.h>
#include <sys/socket.h>
#include <unistd.h>
#include <fcntl.h>

/* case insensitive compare */
static int
http_filter_hash_compare(gconstpointer a, gconstpointer b)
{
  z_enter();
  if (strcasecmp((char *) a, (char *) b) == 0)
    {
      z_leave();
      return 1;
    }
  z_leave();
  return 0;
}

static int
http_filter_hash_bucket(gconstpointer a)
{
  int sum = 0;
  char *s = (char *) a;
  
  z_enter();
  while (*s != 0)
    {
      sum += toupper(*s);     
      s++;
    }

  z_leave();
  return (sum % 16);          
}

static void
http_config_set_defaults(HttpProxy *self)
{
  z_proxy_enter(self);
  
  self->connection_mode = HTTP_CONNECTION_CLOSE;
  self->transparent_mode = TRUE;
  self->permit_server_requests = TRUE;
  self->permit_proxy_requests = FALSE;
  self->permit_unicode_url = FALSE;

  self->rewrite_host_header = TRUE;
  self->require_host_header = TRUE;
  self->strict_header_checking = TRUE;
  self->permit_null_response = TRUE;
  self->max_line_length = 4096;
  self->max_url_length = 4096;
  self->max_header_lines = 50;
  self->max_hostname_length = 256;
  self->max_chunk_length = 262144;
  self->timeout_request = 10000;
  self->timeout = 300000;
  self->default_port = 80;
  self->max_body_length = 0;

  http_init_headers(&self->headers[EP_CLIENT]);
  http_init_headers(&self->headers[EP_SERVER]);

  self->request_url = g_string_sized_new(128);
  self->current_header_name = g_string_sized_new(16);
  self->current_header_value = g_string_sized_new(32);

  self->parent_proxy = g_string_sized_new(0);
  self->parent_proxy_port = 3128;
  
  self->target_port_range = g_string_new("80,443");

  self->request_method_policy =
    g_hash_table_new((GHashFunc) http_filter_hash_bucket,
                     (GCompareFunc) http_filter_hash_compare);
  self->request_header_policy =
    g_hash_table_new((GHashFunc) http_filter_hash_bucket,
                     (GCompareFunc) http_filter_hash_compare);

  self->response_policy = 
    z_dim_hash_table_new(1, 2, DIMHASH_WILDCARD, DIMHASH_CONSUME);
  self->response_header_policy =
    g_hash_table_new((GHashFunc) http_filter_hash_bucket,
                     (GCompareFunc) http_filter_hash_compare);


  self->error_info = g_string_sized_new(0);
  self->error_headers = g_string_sized_new(0);
  self->error_code = -1;
  self->error_status = 500;
  self->error_files_directory = g_string_new(ZORP_DATADIR "/http");

  self->error_silent = FALSE;
  
  self->auth_inband_supported = 1;
  self->auth_realm = g_string_new("Zorp HTTP auth");
  self->old_auth_header = g_string_sized_new(0);
  z_proxy_leave(self);
}

static void
http_config_init(HttpProxy *self)
{
  z_proxy_enter(self);
  if (self->max_line_length > HTTP_MAX_LINE)
    self->max_line_length = HTTP_MAX_LINE;
  self->super.endpoints[EP_CLIENT]->timeout = self->timeout_request;
  z_proxy_leave(self);
}

static ZPolicyObj *
http_query_url(HttpProxy *self, gchar *name, gpointer value G_GNUC_UNUSED)
{
  gchar *proto, *user, *passwd, *host, *file;
  guint proto_len, user_len, passwd_len, host_len, file_len, remote_port;
  ZPolicyObj *res = NULL;

  z_proxy_enter(self);
  if (!http_split_url(self->request_url->str,
                      &proto, &proto_len,    
                      &user, &user_len,      
                      &passwd, &passwd_len,  
                      &host, &host_len,
                      &remote_port,
                      &file, &file_len)) 
    {
      PyErr_SetString(PyExc_ValueError, "request_url format error");
      z_proxy_leave(self);
      return NULL;
    }
  if (self->remote_port == 0)
    self->remote_port = self->default_port;
  
  if (strcmp(name, "request_url_proto") == 0)
    {
      res = z_policy_var_build("s#", proto, proto_len);
    }
  else if (strcmp(name, "request_url_username") == 0)
    {
      res = z_policy_var_build("s#", user, user_len);
    }
  else if (strcmp(name, "request_url_passwd") == 0)
    {
      res = z_policy_var_build("s#", passwd, passwd_len);
    }
  else if (strcmp(name, "request_url_host") == 0)
    {
      res = z_policy_var_build("s#", host, host_len);
    }
  else if (strcmp(name, "request_url_port") == 0)
    {
      if (remote_port == 0)
        remote_port = self->default_port;
      res = z_policy_var_build("i", remote_port);
    }
  else if (strcmp(name, "request_url_file") == 0)
    {
      res = z_policy_var_build("s#", file, file_len);
    }
  z_proxy_leave(self);
  return res;
}

static void
http_register_vars(HttpProxy *self)
{
  z_proxy_enter(self);
  
  z_proxy_var_new(&self->super, "transparent_mode",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->transparent_mode);

  z_proxy_var_new(&self->super, "permit_server_requests",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_server_requests);

  z_proxy_var_new(&self->super, "permit_null_response",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_null_response);

  z_proxy_var_new(&self->super, "permit_proxy_requests",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_proxy_requests);

  z_proxy_var_new(&self->super, "permit_unicode_url",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_unicode_url);

  /* close or keep-alive */
  z_proxy_var_new(&self->super, "connection_mode", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET,
                  &self->connection_mode);

  /* string containing parent proxy */
  z_proxy_var_new(&self->super, "parent_proxy",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  self->parent_proxy);

  /* parent proxy port */
  z_proxy_var_new(&self->super, "parent_proxy_port",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->parent_proxy_port);

  /* default port if portnumber is not specified in urls */
  z_proxy_var_new(&self->super, "default_port",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->default_port);

  /* rewrite host header when redirecting */
  z_proxy_var_new(&self->super, "rewrite_host_header",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->rewrite_host_header);

  /* require host header */
  z_proxy_var_new(&self->super, "require_host_header",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->require_host_header);

  /* enable strict header checking */
  z_proxy_var_new(&self->super, "strict_header_checking",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->strict_header_checking);

  /* integer */
  z_proxy_var_new(&self->super, "max_line_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_line_length);

  /* integer */
  z_proxy_var_new(&self->super, "max_url_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_url_length);

  /* integer */
  z_proxy_var_new(&self->super, "max_hostname_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_hostname_length);

  /* integer */
  z_proxy_var_new(&self->super, "max_header_lines",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_header_lines);

  /* integer */
  z_proxy_var_new(&self->super, "max_keepalive_requests",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_keepalive_requests);

  /* integer */
  z_proxy_var_new(&self->super, "max_body_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_body_length);

  /* integer */
  z_proxy_var_new(&self->super, "max_chunk_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_chunk_length);

  /* integer */
  z_proxy_var_new(&self->super, "request_count",
                  Z_VAR_TYPE_INT | Z_VAR_GET,
                  &self->request_count);

  /* timeout value in milliseconds */
  z_proxy_var_new(&self->super, "timeout", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->timeout);

  /* timeout value in milliseconds */
  z_proxy_var_new(&self->super, "timeout_request", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG,
                  &self->timeout_request);

  /* hash indexed by request method */
  z_proxy_var_new(&self->super, "request",
                  Z_VAR_TYPE_HASH | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  self->request_method_policy);

  /* hash indexed by header name */
  z_proxy_var_new(&self->super, "request_header",
                  Z_VAR_TYPE_HASH | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  self->request_header_policy);

  /* hash indexed by response code */
  z_proxy_var_new(&self->super, "response",
                  Z_VAR_TYPE_DIMHASH | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  self->response_policy);

  /* hash indexed by header name */
  z_proxy_var_new(&self->super, "response_header",
                  Z_VAR_TYPE_HASH | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  self->response_header_policy);

  /* string containing current url */
  z_proxy_var_new(&self->super, "request_url",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
		  self->request_url);

  /* string containing current url proto */
  z_proxy_var_new(&self->super, "request_url_proto",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_url, NULL, NULL);

  /* string containing current url username */
  z_proxy_var_new(&self->super, "request_url_username",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_url, NULL, NULL);

  /* string containing current url passwd */
  z_proxy_var_new(&self->super, "request_url_passwd",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_url, NULL, NULL);

  /* string containing current url hostname */
  z_proxy_var_new(&self->super, "request_url_host",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_url, NULL, NULL);

  /* string containing current url port */
  z_proxy_var_new(&self->super, "request_url_port",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_url, NULL, NULL);

  /* string containing current url file */
  z_proxy_var_new(&self->super, "request_url_file",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_url, NULL, NULL);

  /* string containing current header name */
  z_proxy_var_new(&self->super, "current_header_name",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
		  self->current_header_name);

  /* string containing current header value */
  z_proxy_var_new(&self->super, "current_header_value",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
		  self->current_header_value);

  /* error response */
  z_proxy_var_new(&self->super, "error_status", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->error_status);

  /* error silence */
  z_proxy_var_new(&self->super, "error_silent", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG, 
                  &self->error_silent);

  /* string inserted into error messages */
  z_proxy_var_new(&self->super, "error_info",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
		  self->error_info);

  /* inband authentication support */
  z_proxy_var_new(&self->super, "auth_inband_supported",
  		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_GET_CONFIG,
  		  &self->auth_inband_supported);

  z_proxy_var_new(&self->super, "auth",
  		  Z_VAR_TYPE_OBJECT | Z_VAR_GET | Z_VAR_SET_CONFIG,
  		  &self->auth);

  z_proxy_var_new(&self->super, "auth_realm",
  		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET_CONFIG,
  		  self->auth_realm);

  z_proxy_var_new(&self->super, "target_port_range", 
                  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG, 
                  self->target_port_range);

  z_proxy_var_new(&self->super, "error_files_directory",
  		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG,
  		  self->error_files_directory);

  /* compatibility with Zorp 0.8.x */
  z_proxy_var_new(&self->super, "transparent_server_requests",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  "permit_server_requests");  

  z_proxy_var_new(&self->super, "transparent_proxy_requests",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  "permit_proxy_requests");  

  z_proxy_var_new(&self->super, "request_timeout",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_SET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG,
		  "timeout_request");

  z_proxy_var_new(&self->super, "request_headers",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  "request_header");

  z_proxy_var_new(&self->super, "response_headers",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  "response_header");
  
  z_proxy_var_new(&self->super, "url_proto",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_proto");

  z_proxy_var_new(&self->super, "url_username",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_username");

  z_proxy_var_new(&self->super, "url_passwd",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_passwd");

  z_proxy_var_new(&self->super, "url_host",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_host");

  z_proxy_var_new(&self->super, "url_port",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_port");

  z_proxy_var_new(&self->super, "url_file",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_file");

  z_proxy_var_new(&self->super, "error_response",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_SET,
                  "error_status");

  z_proxy_leave(self);
}

static gboolean
http_error_message(HttpProxy *self, gint response_code, guint message_code, GString *infomsg)
{
  gchar *messages[] = 
    { 
      NULL, 
      "clientsyntax.html", 
      "serversyntax.html", 
      "policysyntax.html", 
      "policyviolation.html", 
      "invalidurl.html", 
      "connecterror.html", 
      "ioerror.html", 
      "auth.html",
      "clienttimeout.html",
      "servertimeout.html"
    };
  gchar response[64], filename[256];
  int fd;
 
  z_proxy_enter(self); 
  if (message_code >= (sizeof(messages) / sizeof(char *)))
    {
      /*LOG
        This message is triggered when the proxy attempts to write
        an error message which doesn't exist. 
       */
      z_proxy_log(self, HTTP_ERROR, 2, "Internal error, error code out of range; error_code='%d'", message_code);
      z_proxy_leave(self);
      return FALSE;
    }
    
  if (message_code == 0)
    {
      z_proxy_leave(self);
      return TRUE;
    }
    
  if (self->proto_version[EP_CLIENT] >= 0x0100)
    {
      g_snprintf(response, sizeof(response), "HTTP/1.0 %d Error encountered\r\n", response_code);
      
      if (http_write(self, EP_CLIENT, response, strlen(response)) != G_IO_STATUS_NORMAL)
        {
	/* error writing error message */
	  z_proxy_leave(self);
          return FALSE;
        }
      g_string_truncate(self->headers[EP_SERVER].flat, 0);
      if (!self->transparent_mode)
	g_string_append(self->headers[EP_SERVER].flat, "Proxy-Connection: close\r\n");
      else
	g_string_append(self->headers[EP_SERVER].flat, "Connection: close\r\n");
      
      g_string_append(self->headers[EP_SERVER].flat, self->error_headers->str);
      g_string_append(self->headers[EP_SERVER].flat, "Content-Type: text/html\r\n\r\n");
      if (http_write(self, EP_CLIENT, self->headers[EP_SERVER].flat->str, self->headers[EP_SERVER].flat->len) != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self);
          return FALSE;
        }
    }
  if (self->error_silent)
    {
      z_proxy_leave(self);
      return FALSE;
    }
  g_snprintf(filename, sizeof(filename), "%s/%s", self->error_files_directory->str, messages[message_code]);

  /*LOG
    This message is written to log file when zorp serving an error file
    to client
   */
  z_proxy_log(self, HTTP_DEBUG, 6, "An error occurred, serving error file; filename='%s'", filename);
  fd = open(filename, O_RDONLY);
  if (fd == -1)
    {
      /*LOG
        This message is appear when Zorp cannot open error
        file.
       */
      z_proxy_log(self, HTTP_ERROR, 3, "I/O error opening error file; filename='%s', error='%m'", filename);
      http_write(self, EP_CLIENT, "Proxy error encountered, error file not found.\r\n", 49);
    }
  else
    {
      GString *new_contents = g_string_sized_new(4096);
      gchar contents[4096], *src;
      gint count;
      
      count = read(fd, contents, sizeof(contents) - 1);
      close(fd);
      if (count == -1)
	count = 0;
      contents[count] = 0;
      src = contents;
      while (*src)
        {
          if (*src == '@')
            {
              if (strncmp(src, "@INFO@", 6) == 0)
                {
                  gchar *p;
                  /* add info */
                  src += 5;
                  
                  for (p = infomsg->str; *p; p++)
                    {
                      if (*p == '<')
                        g_string_append(new_contents, "&lt;");
                      else if (*p == '>')
                        g_string_append(new_contents, "&gt;");
                      else if (*p == '"')
                        g_string_append(new_contents, "&quot;");
                      else if (*p == '&')
                        g_string_append(new_contents, "&amp;");
                      else
                        g_string_append_c(new_contents, *p);
                    }
                }
              else if (strncmp(src, "@VERSION@", 9) == 0)
                {
                  src += 8;
		  g_string_append(new_contents, VERSION);
                }
              else if (strncmp(src, "@DATE@", 6) == 0)
		{
		  time_t t;
		  gchar timebuf[64];
		  struct tm tm;

		  src += 5;
		  t = time(NULL);
		  localtime_r(&t, &tm);
		  strftime(timebuf, sizeof(timebuf), "%a %b %e %H:%M:%S %Z %Y", &tm);
		  g_string_append(new_contents, timebuf);
		}
	      else
		g_string_append_c(new_contents, *src);
	    }
	  else
	    g_string_append_c(new_contents, *src);
            
	  src++;
	}
      http_write(self, EP_CLIENT, new_contents->str, new_contents->len);
      g_string_free(new_contents, TRUE);
    }
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_client_stream_init(HttpProxy *self)
{
  ZStream *tmpstream;
  z_proxy_enter(self);
  
  tmpstream = self->super.endpoints[EP_CLIENT];
  self->super.endpoints[EP_CLIENT] = z_stream_line_new(tmpstream, self->max_line_length, ZRL_EOL_CRLF);
  z_stream_unref(tmpstream);
  /* timeout is initialized after the config event */
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_server_stream_init(HttpProxy *self)
{
  ZStream *tmpstream;
  z_proxy_enter(self);
  
  tmpstream = self->super.endpoints[EP_SERVER];
  self->super.endpoints[EP_SERVER] = z_stream_line_new(tmpstream, self->max_line_length, ZRL_EOL_CRLF);
  z_stream_unref(tmpstream);
  self->super.endpoints[EP_SERVER]->timeout = self->timeout;
  z_proxy_leave(self);
  return TRUE;
}

GIOStatus
http_read(HttpProxy *self, guint side, gchar *buf, size_t buflen, guint *bytes_read)
{
  GIOStatus res;
  
  z_proxy_enter(self);
  res = z_stream_read(self->super.endpoints[side], buf, buflen, bytes_read, NULL);
  if (res != G_IO_STATUS_NORMAL && res != G_IO_STATUS_EOF)
    {
      /*LOG
        This message appear when zorp cannot read from
        network.
       */
      z_proxy_log(self, HTTP_ERROR, 1, "Error reading %s stream; res='%x', error='%m'", side == EP_CLIENT ? "client" : "server", res);
      z_proxy_leave(self);
      return res;
    }
  z_proxy_leave(self);
  return res;
}

GIOStatus
http_write(HttpProxy *self, guint side, gchar *buf, size_t buflen)
{
  guint res, bytes_written;
  
  z_proxy_enter(self);
  if (!self->super.endpoints[side])
    {
      /*LOG
        This message appear when Zorp try to write a stream
        that doesn't exists, or destroyed.
       */
      z_proxy_log(self, HTTP_ERROR, 1, "Error writing %s stream, stream is NULL;", side == EP_CLIENT ? "client" : "server");
      z_proxy_leave(self);
      return G_IO_STATUS_ERROR;
    }
  res = z_stream_write(self->super.endpoints[side], buf, buflen, &bytes_written, NULL);
  if (res != G_IO_STATUS_NORMAL || buflen != bytes_written)
    {
      /* FIXME: move this to a separate function */
      /*LOG
        This message appears when Zorp tries to write a stream
        but some error occured.
       */
      z_proxy_log(self, HTTP_ERROR, 1, "Error writing %s stream; res='%x', error='%m'", side == EP_CLIENT ? "client" : "server", res);

      self->error_code = HTTP_MSG_IO_ERROR;
      self->error_status = 502;
      g_string_sprintf(self->error_info, "Error writing to %s (%m)", side == EP_CLIENT ? "client" : "server");
      z_proxy_leave(self);
      return G_IO_STATUS_ERROR;
    }

  z_proxy_leave(self);
  return res;
}   

gboolean
http_hash_get_type(ZPolicyObj *tuple, guint *filter_type)
{
  ZPolicyObj *tmp;

  z_enter();
  if (!z_policy_seq_check(tuple))
    {
      if (z_policy_var_parse(tuple, "i", filter_type))
        {
          z_leave();
          return TRUE;
        }
      /* not a sequence nor an int */
      z_leave();
      return FALSE;
    }
    
  tmp = z_policy_seq_getitem(tuple, 0);
  if (!z_policy_var_parse(tmp, "i", filter_type))
    {
      /* policy syntax error */
      z_policy_var_unref(tmp);
      z_leave();
      return FALSE;
    }
  z_policy_var_unref(tmp);
  z_leave();
  return TRUE;
}

static gboolean
http_fetch_request(HttpProxy *self)
{
  gchar *line;
  guint line_length;
  gint empty_lines = 0;
  guint res;

  z_proxy_enter(self); 

  http_clear_headers(&self->headers[EP_CLIENT]);
  while (empty_lines < HTTP_MAX_EMPTY_REQUESTS) 
    {
      self->super.endpoints[EP_CLIENT]->timeout = self->timeout_request;
      res = z_stream_line_get(self->super.endpoints[EP_CLIENT], &line, &line_length, NULL);
      self->super.endpoints[EP_CLIENT]->timeout = self->timeout;
      if (res == G_IO_STATUS_EOF)
        {
          self->error_code = HTTP_MSG_OK;
          z_proxy_leave(self);
          return FALSE;
        }
      else if (res != G_IO_STATUS_NORMAL)
        {
          self->error_code = HTTP_MSG_OK; 
          if (errno == ETIME)
            {
              if (self->request_count == 0)
                {
	          self->error_code = HTTP_MSG_CLIENT_TIMEOUT;
                  self->error_status = 408;
                }
            }
          z_proxy_leave(self);
	  return FALSE;
	}
      if (line_length != 0)
	break;
      empty_lines++;
    }
  
  if (!http_split_request(self, line, line_length))
    {
      g_string_assign(self->error_info, "Invalid request line.");
      z_proxy_leave(self);
      return FALSE;
    }
  self->request_flags = http_proto_request_lookup(self->request_method);

  if (!http_parse_version(self, EP_CLIENT, self->request_version))
    {
      z_proxy_leave(self);
      return FALSE;
    }

  if (!http_fetch_headers(self, EP_CLIENT))
    {
      z_proxy_leave(self);
      return FALSE;
    }

  z_proxy_leave(self);
  return TRUE;
}

static gint
http_translate_base64_char(guchar c)
{
  static char base64_alphabet[128] = 
    {
     -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,  /*  0*/
     -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,  /* 16*/
     -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,  /* 32*/
     52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1,  /* 48*/
     -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,  /* 64*/
     15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,  /* 80*/
     -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,  /* 96*/
     41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1   /*112*/
    };
  if (c == '=')
    return -2;
  else if (c > 127)
    return -1;
  else
    return base64_alphabet[c];
}

static gboolean
http_process_base64_quantum(guchar *dst, guint *dstlen, guchar *src, guint srclen)
{
  gchar c[4];
  guint checked = 0;
  guint i, j;
  
  if (srclen < 4)
    {
      return FALSE;
    }
  c[0] = http_translate_base64_char(src[0]);
  c[1] = http_translate_base64_char(src[1]);
  c[2] = http_translate_base64_char(src[2]);
  c[3] = http_translate_base64_char(src[3]);

  if (*dstlen < 3)
    return FALSE;
    
  *dstlen = 3;
  
  /* sanity check quantum */
  for (i = 0; i < 4; i++)
    if (c[i] == -2)
      {
        c[i] = 0;
        if (!checked)
          {
            if (i == 2)
              *dstlen = 1;
            else if (i == 3)
              *dstlen = 2;
            else
              return FALSE;
              
            checked = 1;
            for (j = i + 1; j < 4; j++)
              if (c[j] != -2)
                return FALSE;
          }
      }
  *dst = (c[0] << 2) | ((c[1] & 0x30) >> 4); 
  dst++;
  *dst = ((c[1] & 0x0f) << 4) | ((c[2] & 0x3e) >> 2);
  dst++;
  *dst = ((c[2] & 0x03) << 6) | (c[3] & 0x3f);
  dst++;
  return TRUE;
}

static gboolean
http_process_base64(gchar *dst, guint dstlen, gchar *src, guint srclen)
{
  while (srclen > 0 && dstlen > 0)
    {
      gint l;
      
      l = dstlen;
      
      if (!http_process_base64_quantum(dst, &l, src, srclen))
        return FALSE;
        
      dstlen -= l;
      srclen -= 4;
      src += 4;
      dst += l;
    }
  if (srclen != 0)
    return FALSE;
  *dst = 0;
  return TRUE;
}

/* FIXME: optimize header processing a bit (no need to copy hdr into a
   buffer) */
static gboolean
http_process_auth_info(HttpProxy *self, HttpHeader *h)
{
  gchar userpass[128];
  gchar *p;
  gchar **up;

  z_proxy_enter(self);

  if (self->old_auth_header->len &&
      strcmp(h->value->str, self->old_auth_header->str) == 0)
    {
      z_proxy_leave(self);
      return TRUE;
    }
  
  if (strncmp(h->value->str, "Basic", 5) != 0)
    {
      /*LOG
        This message is appear, when client try to authenticate
        myself with an unknown authentication method
       */
      z_proxy_log(self, HTTP_ERROR, 3, "Only Basic authentication is supported;");
      /* not basic auth */
      z_proxy_leave(self);
      return FALSE;
    }
    
  p = h->value->str + 5;
  while (*p == ' ') 
    {
      p++;
    }
  if (!http_process_base64(userpass, sizeof(userpass), p, strlen(p)))
    {
      /*LOG
        This message is about when client send an
        invalid encoded username or password
       */
      z_proxy_log(self, HTTP_VIOLATION, 1, "Invalid base64 encoded username:password pair;");
      z_proxy_leave(self);
      return FALSE;
    }
  
  up = g_strsplit(userpass, ":", 2);
  if (up)
    {
      gboolean res;
      
      z_policy_lock(self->super.thread);
      res = z_auth_provider_check_passwd(self->auth, self->super.session_id, up[0], up[1]);
      z_policy_unlock(self->super.thread);
      if (res)
        z_proxy_user_authenticated(&self->super, up[0]);
      g_strfreev(up);
      z_proxy_leave(self);
      return res;
    }
  z_proxy_leave(self);
  return FALSE;
}

static gboolean
http_process_request(HttpProxy *self)
{
  HttpHeader *h;

  z_proxy_enter(self);  
  if (self->proto_version[EP_CLIENT] > 0x0100)
    self->connection_mode = HTTP_CONNECTION_KEEPALIVE;
  else    
    self->connection_mode = HTTP_CONNECTION_CLOSE;
    
    
  if (self->request_flags & HTTP_REQ_FLG_CONNECT)
    {
      if (self->proto_version[EP_CLIENT] >= 0x0100)
        {
          self->request_type = HTTP_REQTYPE_PROXY;
          z_proxy_leave(self);
          return TRUE;
        }
      else
        {
          self->error_code = HTTP_MSG_CLIENT_SYNTAX;
          g_string_sprintf(self->error_info, "CONNECT method requires HTTP/1.0 or later");
          /*LOG
            This message appear when client
            try to use CONNECT method without
            proper header.
           */
          z_proxy_log(self, HTTP_VIOLATION, 1, "CONNECT method without version specification;");
          z_proxy_leave(self);
          return FALSE;
        }
    }
    
  if (self->proto_version[EP_CLIENT] < 0x0100)
    {
      /* no proxy protocol for version 0.9 */
      self->request_type = HTTP_REQTYPE_SERVER;
    }
  else if (http_lookup_header(&self->headers[EP_CLIENT], "Proxy-Connection", &h))
    {
      if (http_lookup_header(&self->headers[EP_CLIENT], "Connection", &h))
	{
	  /* both Proxy-Connection & Connection headers ... */
	  self->error_code = HTTP_MSG_CLIENT_SYNTAX;
	  g_string_sprintf(self->error_info, "Both Proxy-Connection and Connection headers exist.");
	  /*LOG
	    This message appear when both proxy-connection and
	    connection header exists in request. This may be
	    because a buggy proxy in client side.
	   */
	  z_proxy_log(self, HTTP_VIOLATION, 1, "Both Proxy-Connection and Connection headers exist;");
	  z_proxy_leave(self);
	  return FALSE;
	}
      self->request_type = HTTP_REQTYPE_PROXY;
    }
  else if (http_lookup_header(&self->headers[EP_CLIENT], "Connection", &h))
    {
      self->request_type = HTTP_REQTYPE_SERVER;
    }
  else if (self->request_url->str[0] != '/')
    {
      /* neither Connection nor Proxy-Connection header exists, and URI
         doesn't seem to be a simple filename */
      self->request_type = HTTP_REQTYPE_PROXY;
    }
  else
    {
      /* default */
      self->request_type = HTTP_REQTYPE_SERVER;
    }

  if (self->auth)
    {
      /* authentication is required */
      if (self->transparent_mode)
        {
          self->error_code = HTTP_MSG_POLICY_SYNTAX;
          g_string_sprintf(self->error_info, "Authentication is not available in transparent mode.");
          /*LOG
            This is a placeholder 1.
           */
          z_proxy_log(self, HTTP_POLICY, 1, "Authentication is not available in transparent mode;");
          z_proxy_leave(self);
          return FALSE;
        }
      if (!http_lookup_header(&self->headers[EP_CLIENT], "Proxy-Authorization", &h) ||
          !http_process_auth_info(self, h))
        {
          self->error_code = HTTP_MSG_AUTH_REQUIRED;
          self->error_status = 407;
          g_string_sprintf(self->error_info, "Authentication is required.");
          g_string_sprintfa(self->error_headers, "Proxy-Authenticate: Basic realm=\"%s\"\r\n", self->auth_realm->str);
          z_proxy_leave(self);
          return FALSE;
        }
    }

  if (self->transparent_mode &&
      ((self->request_type == HTTP_REQTYPE_PROXY && !self->permit_proxy_requests) ||
       (self->request_type == HTTP_REQTYPE_SERVER && !self->permit_server_requests)))
    {
      /* */
      self->error_code = HTTP_MSG_POLICY_VIOLATION;
      g_string_sprintf(self->error_info, "%s requests not permitted in transparent mode.", 
		       self->request_type == HTTP_REQTYPE_SERVER ? "server" : "proxy");
      /*LOG
        This is a placeholder 2.
       */
      z_proxy_log(self, HTTP_POLICY, 2, 
                  "This request type is not permitted in transparent mode; request_type='%s'", 
                  self->request_type == HTTP_REQTYPE_SERVER ? "server" : "proxy");
      z_proxy_leave(self);
      return FALSE;
    }

  if (((self->request_type == HTTP_REQTYPE_SERVER) && 
       http_lookup_header(&self->headers[EP_CLIENT], "Connection", &h)) ||
      ((self->request_type == HTTP_REQTYPE_PROXY) && 
       http_lookup_header(&self->headers[EP_CLIENT], "Proxy-Connection", &h)))
    {
      if (strcasecmp(h->value->str, "Keep-Alive") == 0)
        {
          self->connection_mode = HTTP_CONNECTION_KEEPALIVE;
        }
      else if (strcasecmp(h->value->str, "Close") == 0)
        {
          self->connection_mode = HTTP_CONNECTION_CLOSE;
        }
    }


  if (http_lookup_header(&self->headers[EP_CLIENT], "Host", &h))
    {
      g_strlcpy(self->remote_server, h->value->str, sizeof(self->remote_server));
    }
  else
    {
      self->remote_server[0] = 0;
    }


  if (self->transparent_mode)
    {
      if (self->request_url->str[0] == '/')
	{
	  gchar buf[strlen(self->remote_server) + 32];
	  
	  /* no protocol description */
	  if (!self->remote_server[0])
	    {
	      if (!self->require_host_header)
	        {
	          /* no host header */
                   /*LOG
                     This is a placeholder 4.
                    */
	          z_proxy_log(self, HTTP_VIOLATION, 4, "No host header in transparent request, 'unknown' is used instead;");
	          strcpy(self->remote_server, "unknown");
	        }
	      else
	        {
                  self->error_code = HTTP_MSG_CLIENT_SYNTAX;
                  if (self->proto_version[EP_CLIENT] < 0x0100)
                    {
                      g_string_sprintf(self->error_info, "'Host:' header is required, and HTTP/0.9 can't transfer headers.");
                      z_proxy_log(self, HTTP_POLICY, 2, "'Host:' header is required, and HTTP/0.9 can't transfer headers;");
                    }
                  else
                    {
                      g_string_sprintf(self->error_info, "No 'Host:' header in request, and policy requires this.");
                      z_proxy_log(self, HTTP_POLICY, 2, "No 'Host:' header in request, and policy requires this;");
                    }
                   /*LOG
                     This is a placeholder 5.
                    */
	          z_proxy_leave(self);
	          return FALSE;
	        }
	    }
	  g_snprintf(buf, sizeof(buf), "http://%s", self->remote_server);
	  g_string_prepend(self->request_url, buf);
	}
    }

  if (!http_check_url(self->request_url->str, self->permit_unicode_url))
    {
      /* invalid url */
      self->error_code = HTTP_MSG_INVALID_URL;
      g_string_sprintf(self->error_info, "Invalid URL %s", self->request_url->str);
      /*LOG
        This is a placeholder 3.
       */
      z_proxy_log(self, HTTP_VIOLATION, 1, "Invalid URL; url='%s'", self->request_url->str);
      z_proxy_leave(self);
      return FALSE;
    }
  z_proxy_leave(self);
  return TRUE;  
} 

static void
http_assign_connection_hdr_value(HttpProxy *self, GString *value)
{
  z_proxy_enter(self);
  if (self->connection_mode == HTTP_CONNECTION_KEEPALIVE)
    g_string_assign(value, "keep-alive");
  else if (self->connection_mode == HTTP_CONNECTION_CLOSE)
    g_string_assign(value, "close");
  z_proxy_leave(self);
}

static guint 
http_request_filter_connection_header(HttpProxy *self, GString *name, GString *value)
{
  z_proxy_enter(self);

  switch (self->request_type)
    {
    case HTTP_REQTYPE_SERVER:
      /* if we have a parent proxy, Connection -> Proxy-Connection
       * otherwise leave it as is 
       */
      if (strcasecmp(name->str, "Connection") == 0)
        {
          if (self->parent_proxy->len)
	    g_string_assign(name, "Proxy-Connection");
	  http_assign_connection_hdr_value(self, value);
	}
      break;
    case HTTP_REQTYPE_PROXY:
      /* if we have a parent proxy leave it as is
       * otherwise Proxy-Connection -> Connection 
       */
      if (strcasecmp(name->str, "Proxy-Connection") == 0)
        {
          if (self->parent_proxy->len == 0)
	    g_string_assign(name, "Connection");
	  http_assign_connection_hdr_value(self, value);
	}
      break;
    }
  z_proxy_leave(self);
  return HTTP_HDR_ACCEPT;
}

static gboolean
http_filter_request(HttpProxy *self)
{
  ZPolicyObj *f;
  gint rc;

  z_proxy_enter(self);
  f = g_hash_table_lookup(self->request_method_policy, self->request_method);
  if (!f)
    f = g_hash_table_lookup(self->request_method_policy, "*");
  if (f)
    {
      ZPolicyObj *handler, *res;
      gchar *errmsg;
      guint filter_type;
      
      z_policy_lock(self->super.thread);  
      if (!http_hash_get_type(f, &filter_type))
        {
          /*LOG
            This is a placeholder 6.
           */
	  z_proxy_log(self, HTTP_POLICY, 1, "Invalid item in request hash; method='%s'", self->request_method);
	  z_policy_unlock(self->super.thread);
	  z_proxy_leave(self);
	  return FALSE;
          
        }
      z_policy_unlock(self->super.thread);
      g_string_sprintf(self->error_info, "Method %s denied by policy", self->request_method);

      switch (filter_type)
        {
        case HTTP_REQ_POLICY:
	  z_policy_lock(self->super.thread);
          if (!z_policy_var_parse(f, "(iO)", &filter_type, &handler))
            {
              /*LOG
                This is placeholder 7.
              */
              z_proxy_log(self, HTTP_POLICY, 1, "Error parsing HTTP_REQ_POLICY tuple in request hash; method='%s'", self->request_method);
              z_policy_unlock(self->super.thread);
              z_proxy_leave(self);
              return FALSE;
            }
          res = z_policy_call_object(handler, 
                                     z_policy_var_build("(sss)", 
                                                        self->request_method, self->request_url->str, self->request_version), 
                                     self->super.session_id);
          if (!res || !z_policy_var_parse(res, "i", &rc))
            {
	      rc = HTTP_REQ_REJECT;
              g_string_assign(self->error_info, "Error in policy handler, or returned value not integer;");
            }
          z_policy_var_unref(res);
	  z_policy_unlock(self->super.thread);
          break;
	case HTTP_REQ_REJECT:
	  errmsg = NULL;
          z_policy_lock(self->super.thread);
	  if (!z_policy_var_parse_tuple(f, "i|s", &filter_type, &errmsg))
	    {
              /*LOG
                This is placeholder 8.
              */
	      z_proxy_log(self, HTTP_POLICY, 1, "Error parsing HTTP_REQ_REJECT in request hash; req='%s'", self->request_method);
              z_policy_unlock(self->super.thread);
	      z_proxy_leave(self);
              return FALSE;
	    }
	  z_policy_unlock(self->super.thread);
          if (errmsg)
	    g_string_assign(self->error_info, errmsg);
          /* fallthrough */
	case HTTP_REQ_ACCEPT:
	case HTTP_REQ_DENY:
	case HTTP_REQ_ABORT:
	  /* dropped command */
	  rc = filter_type;
	  break;
	default:
          /*LOG
            This is placeholder 8.
          */
	  z_proxy_log(self, HTTP_POLICY, 1, "Unknown request hash item; req='%s'", self->request_method);
	  z_proxy_leave(self);
	  return FALSE;
        }
      switch (rc)
        {
        case HTTP_REQ_ACCEPT:
          g_string_truncate(self->error_info, 0);
          break;
        default:
	  self->error_code = HTTP_MSG_POLICY_VIOLATION;
	  z_proxy_leave(self);
          return FALSE;
        }
      if (self->proto_version[EP_CLIENT] >= 0x0100)
        {
          if (!http_filter_headers(self, EP_CLIENT, http_request_filter_connection_header))
    	    {
    	      z_proxy_leave(self);
    	      return FALSE;
    	    }
        }
      z_proxy_leave(self);
      return TRUE;
    }
  self->error_code = HTTP_MSG_POLICY_VIOLATION;
  g_string_sprintf(self->error_info, "Method %s denied by policy", self->request_method);
  z_proxy_leave(self);
  return FALSE;
}

static gboolean
http_connect_server(HttpProxy *self)
{
  z_proxy_enter(self);
  if (!self->super.endpoints[EP_SERVER] ||
      strcasecmp(self->remote_server, self->connected_server) != 0 ||
      self->remote_port != self->connected_port)
    {
      gboolean success = FALSE;
      
      if (self->super.endpoints[EP_SERVER])
	{
	  z_stream_shutdown(self->super.endpoints[EP_SERVER], SHUT_RDWR, NULL);
	  z_stream_close(self->super.endpoints[EP_SERVER], NULL);
	  z_stream_unref(self->super.endpoints[EP_SERVER]);
	  self->super.endpoints[EP_SERVER] = NULL;
	}
      
      g_string_sprintf(self->error_info, "Error establishing connection to %s", self->remote_server);
      if (self->transparent_mode)
        success = z_proxy_connect_server_event(&self->super, self->remote_server, self->remote_port);
      else
        {
          if (self->parent_proxy->len || z_port_enabled(self->target_port_range->str, self->remote_port))
            {
              success = z_proxy_connect_server_event(&self->super, self->remote_server, self->remote_port);
            }
          else
            {
              g_string_sprintf(self->error_info, "Connecting to port %d is prohibited by policy.", self->remote_port);
              success = FALSE;
            }
        }

      if (!success)
	{
	  /* error connecting to server */
	  self->error_code = HTTP_MSG_CONNECT_ERROR;
	  self->error_status = 502;
	  z_proxy_leave(self);
	  return FALSE;
	}

      strcpy(self->connected_server, self->remote_server);
      self->connected_port = self->remote_port;
      if (!http_server_stream_init(self))
        {
          /* should never happen */
          z_proxy_leave(self);
          return FALSE;
        }
    }
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_rewrite_host_header(HttpProxy *self, gchar *host, guint host_len, guint port)
{
  HttpHeader *h;
  z_proxy_enter(self);  
  
  if (self->rewrite_host_header && http_lookup_header(&self->headers[EP_CLIENT], "Host", &h))
    {
      /* NOTE: the constant 80 below is intentional, if
       * default_port is changed we still have to send
       * correct host header (containing port number)
       */
      if (port != 80 && port != 0)
	g_string_sprintf(h->value, "%.*s:%d", host_len, host, port);
      else
	g_string_sprintf(h->value, "%.*s", host_len, host);
    }
  http_log_headers(self, EP_CLIENT, "postfilter");
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_copy_request(HttpProxy *self)
{
  gchar *request;
  gchar *proto, *user, *passwd, *host, *file;
  guint proto_len, user_len, passwd_len, host_len, file_len;
  guint attempts = 0;
  
  z_proxy_enter(self);
  if (!http_split_url(self->request_url->str, 
		      &proto, &proto_len, 
		      &user, &user_len,
		      &passwd, &passwd_len, 
		      &host, &host_len,
		      &self->remote_port,
		      &file, &file_len))
    {
      /* invalid URL */
      self->error_code = HTTP_MSG_INVALID_URL;
      g_string_sprintf(self->error_info, "Badly formatted url: %s", self->request_url->str);
      z_proxy_leave(self);
      return FALSE;
    }
  if (self->remote_port == 0)
    self->remote_port = self->default_port;
  
  if (host_len > self->max_hostname_length)
    {
      self->error_code = HTTP_MSG_CLIENT_SYNTAX;
      g_string_sprintf(self->error_info, "Too long hostname in URL: %.*s", host_len, host);
      z_proxy_leave(self);
      return FALSE;
    }

  if (strncasecmp(proto, "http://", 7) != 0 &&
      strncasecmp(proto, "ftp://", 6) != 0 &&
      strncasecmp(proto, "cache_object://", 9) != 0)
    {
      /* unsupported protocol */
      self->error_code = HTTP_MSG_CLIENT_SYNTAX;
      g_string_sprintf(self->error_info, "Unsupported protocol: %.*s", (int) proto_len, proto);
      z_proxy_leave(self);
      return FALSE;
    }

  http_rewrite_host_header(self, host, host_len, self->remote_port);
  if (self->parent_proxy->len)
    {
      self->remote_port = self->parent_proxy_port;
      g_strlcpy(self->remote_server, self->parent_proxy->str, sizeof(self->remote_server));
    }
  else
    {
      g_strlcpy(self->remote_server, host, MIN(host_len + 1, sizeof(self->remote_server)));
    }


  while (attempts < 4)
    {
      attempts++;
      if (!http_connect_server(self))
        {
          z_proxy_leave(self);
	  return FALSE;
	}

      if (!http_data_transfer_setup(self, EP_CLIENT, EP_SERVER, FALSE))
        {
          z_proxy_leave(self);
          return FALSE;
        }

      if (self->proto_version[EP_CLIENT] >= 0x0100)
	{
	  http_flat_headers(&self->headers[EP_CLIENT]);

	  request = alloca(strlen(self->request_method) + self->request_url->len + strlen(self->request_version) + self->headers[EP_CLIENT].flat->len + 10);
	  if (self->parent_proxy->len)
	    sprintf(request, "%s %s %s\r\n%s\r\n", self->request_method, self->request_url->str, self->request_version, self->headers[EP_CLIENT].flat->str);
	  else
	    sprintf(request, "%s %.*s %s\r\n%s\r\n", self->request_method, file_len, file, self->request_version, self->headers[EP_CLIENT].flat->str);
	}
      else
	{
	  request = alloca(strlen(self->request_method) + self->request_url->len + 5);
	  sprintf(request, "%s %s\r\n", self->request_method, self->request_url->str);
	}
  
      if (http_write(self, EP_SERVER, request, strlen(request)) != G_IO_STATUS_NORMAL)
        {
	  if (attempts < 4)
	    continue;

 	  self->error_code = HTTP_MSG_CONNECT_ERROR;
 	  self->error_status = 502;
 	  g_string_sprintf(self->error_info, "Error establishing connection to %s", self->remote_server);
 	  
	  z_proxy_leave(self);
	  return FALSE;
	}
    
      if (http_data_transfer(self))
	break;
      else 
        {
          z_proxy_leave(self);
          return FALSE;
        }

    }

  z_proxy_leave(self);
  return TRUE;
}

static gboolean 
http_fetch_response(HttpProxy *self)
{
  gchar *line;
  guint line_length;
  guint res;
  gboolean reply09;

  z_proxy_enter(self);
  http_clear_headers(&self->headers[EP_SERVER]);

  self->response[0] = 0;
  self->response_code = -1;
  if (self->proto_version[EP_CLIENT] < 0x0100)
    {
      self->proto_version[EP_SERVER] = 0x0009;
      z_proxy_leave(self);
      return TRUE;
    }
  
  while (self->response_code == -1 || self->response_code == 100 || self->response_code == 102)
    {
      res = z_stream_line_get(self->super.endpoints[EP_SERVER], &line, &line_length, NULL);
      if (res != G_IO_STATUS_NORMAL)
        {
          self->error_code = HTTP_MSG_OK;
          z_proxy_leave(self);
	  return FALSE;
	}
      
      if (!http_split_response(self, line, line_length, &reply09))
        {
          if (reply09)
            {
              if (self->request_count)
                {
                  /* this is a kept-alive request -> HTTP/0.9 is not possible */
                  g_string_assign(self->error_info, "Server incorrectly falled back to HTTP/0.9");
                  z_proxy_log(self, HTTP_VIOLATION, 2, "HTTP 0.9 response in keep-alive mode which requires HTTP/1.0;");
                  z_proxy_leave(self);
                  return FALSE;
                }
              z_stream_line_unget_line(self->super.endpoints[EP_SERVER]);
	      self->proto_version[EP_SERVER] = 0x0009;
	      z_proxy_leave(self);
	      return TRUE;
            }
          g_string_assign(self->error_info, "Invalid HTTP response heading");
          z_proxy_leave(self);
	  return FALSE;
	}
      
      self->response_flags = http_proto_response_lookup(self->response);
      if (!http_parse_version(self, EP_SERVER, self->response_version))
        {
          /*LOG
            This is placeholder 9.
          */
          g_string_sprintf(self->error_info, "Invalid HTTP version in response (%.*s)", line_length, line);
          z_proxy_log(self, HTTP_VIOLATION, 1, "Error in parsing response; line='%.*s'", line_length, line);
          z_proxy_leave(self);
	  return FALSE;
	}
      
      if (!http_fetch_headers(self, EP_SERVER))
        {
          g_string_assign(self->error_info, "Invalid headers received in response");
          z_proxy_leave(self);
	  return FALSE;
	}
    }

  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_process_response(HttpProxy *self)
{
  HttpHeader *h;

  z_proxy_enter(self);


  /* previously set by http_parse_version */
  switch (self->proto_version[EP_SERVER])
    {
    case 0x0009:
      strncpy(self->response, "200", sizeof(self->response_code));
      self->response_code = 200;
      self->response_flags = 0;
      self->connection_mode = HTTP_CONNECTION_CLOSE;
      z_proxy_leave(self);
      return TRUE;
    case 0x0100:
      self->connection_mode = HTTP_CONNECTION_CLOSE;
      break;
    case 0x0101:
      /* leave connection mode as is, as was requested by
       * the browser */
      break;
    default:
      /*LOG
        This is placeholder 9.
       */
      z_proxy_log(self, HTTP_VIOLATION, 1, "Unsupported protocol version; version='0x%04x'", self->proto_version[EP_SERVER]);
      z_proxy_leave(self);
      return FALSE;
    }

  if ((self->request_type == HTTP_REQTYPE_SERVER && 
       http_lookup_header(&self->headers[EP_SERVER], "Connection", &h)) ||
      (self->request_type == HTTP_REQTYPE_PROXY && 
       http_lookup_header(&self->headers[EP_SERVER], "Proxy-Connection", &h)))
    {
      if (strcasecmp(h->value->str, "Keep-Alive") != 0)
        {
          self->connection_mode = HTTP_CONNECTION_CLOSE;
        }
    }
  else
    {
      /* there was no appropriate Connection header in the response, if it
       * is an 1.0 client the connection mode should be changed to
       * connection close
       */
      if (self->proto_version[EP_CLIENT] == 0x0100)
        {
          gchar *conn_hdr = self->request_type == HTTP_REQTYPE_SERVER ? "Connection" : "Proxy-Connection";
          
          self->connection_mode = HTTP_CONNECTION_CLOSE;
          http_add_header(&self->headers[EP_SERVER], conn_hdr, strlen(conn_hdr), "close", 5);
        }
    }

  if (self->response_flags & HTTP_RESP_FLG_STOP)
    {
      z_proxy_leave(self);
      return FALSE;  /* FIXME: not implemented */
    }

  z_proxy_leave(self);
  return TRUE;
}

static guint
http_response_filter_connection_header(HttpProxy *self, GString *name, GString *value)
{
  z_proxy_enter(self);

#if 0
  if (strcasecmp(name->str, "Content-Length") == 0)
    {
      gchar *end;
      
      self->content_length = strtoul(value->str, &end, 10);
      if (end - value->str != value->len)
	{
	  /* content-length not a number */
	  z_proxy_leave(self);
	  return HTTP_HDR_ABORT;
	}
      z_proxy_leave(self);
      return HTTP_HDR_DROP;
    }
  else if (strcasecmp(name->str, "Transfer-Encoding") == 0)
    {
      g_string_assign_len(self->transfer_encoding, value->str, value->len);
      z_proxy_leave(self);
      return HTTP_HDR_DROP;
    }
#endif

  switch (self->request_type)
    {
    case HTTP_REQTYPE_SERVER:
      /* if we have a parent proxy, Proxy-Connection -> Connection
       * otherwise leave it as is 
       */
      if (strcasecmp(name->str, "Proxy-Connection") == 0)
        {
          if (self->parent_proxy->len)
	    g_string_assign(name, "Connection");
	  http_assign_connection_hdr_value(self, value);
	}
      break;
    case HTTP_REQTYPE_PROXY:
      /* if we have a parent proxy leave it as is
       * otherwise Connection -> Proxy-Connection 
       */
      if (strcasecmp(name->str, "Connection") == 0)
        {
          if (self->parent_proxy->len == 0)
	    g_string_assign(name, "Proxy-Connection");
          http_assign_connection_hdr_value(self, value);
	}
      break;
    }
  z_proxy_leave(self);
  return HTTP_HDR_ACCEPT;
}

static ZPolicyObj *
http_response_policy_get(HttpProxy *self)
{
  gchar *response_keys[2];

  response_keys[0] = self->request_method;
  response_keys[1] = self->response;
  
  return z_dim_hash_table_search(self->response_policy, 2, response_keys);
}

static gboolean
http_filter_response(HttpProxy *self)
{
  z_proxy_enter(self);
  if (self->proto_version[EP_SERVER] >= 0x0100)
    {
      ZPolicyObj *f, *res;
      gint rc;
      
      f = http_response_policy_get(self);
      if (f)
        {
          ZPolicyObj *handler;
	  guint filter_type;
          gchar *errmsg;
      
          z_policy_lock(self->super.thread);  
          if (!http_hash_get_type(f, &filter_type))
            {
              /*LOG
                This is placeholder 10.
               */
	      z_proxy_log(self, HTTP_POLICY, 1, "Invalid response hash item; request='%s', response='%d'", self->request_method, self->response_code);
	      z_policy_unlock(self->super.thread);
	      z_proxy_leave(self);
	      return FALSE;
          
            }
          z_policy_unlock(self->super.thread);
          
          g_string_sprintf(self->error_info, "Response %d for %s denied by policy.", self->response_code, self->request_method);
          switch (filter_type)
            {
            case HTTP_RSP_POLICY:
	      z_policy_lock(self->super.thread);
              if (!z_policy_var_parse(f, "(iO)", &filter_type, &handler))
                {
                  /*LOG
                    This is placeholder 11.
                   */
                  z_proxy_log(self, HTTP_POLICY, 1, 
                              "Error parsing HTTP_RSP_POLICY in response hash; request='%s', response='%d'", 
                              self->request_method, self->response_code);
                  z_policy_unlock(self->super.thread);
                  z_proxy_leave(self);
                  return FALSE;
                }
              res = z_policy_call_object(handler, z_policy_var_build("(sssi)", self->request_method, self->request_url->str, self->request_version, self->response_code), self->super.session_id);
              if (!z_policy_var_parse(res, "i", &rc))
                {
		  g_string_assign(self->error_info, "Error in policy handler.");
                  rc = HTTP_RSP_REJECT;
                }
              z_policy_var_unref(res);
	      z_policy_unlock(self->super.thread);
	      break;
	    case HTTP_RSP_REJECT:
	      errmsg = NULL;
	      z_policy_lock(self->super.thread);
	      if (!z_policy_var_parse_tuple(f, "i|s", &filter_type, &errmsg))
		{
                  /*LOG
                    This is placeholder 12.
                   */
		  z_proxy_log(self, HTTP_POLICY, 1, 
		              "Error parsing HTTP_RSP_REJECT in response hash; request='%s', response='%d'", 
		              self->request_method, self->response_code);
		  z_policy_unlock(self->super.thread);
		  z_proxy_leave(self);
		  return FALSE;
		}
	      z_policy_unlock(self->super.thread);
	      if (errmsg)
		g_string_assign(self->error_info, errmsg);
              /* fallthrough */
	    case HTTP_RSP_ACCEPT:
	    case HTTP_RSP_DENY:
	    case HTTP_RSP_ABORT:
	      /* dropped command */
	      rc = filter_type;
	      break;
	    default:
              /*LOG
                This is placeholder 13.
               */
	      z_proxy_log(self, HTTP_POLICY, 1, 
	                  "Invalid response hash item; request='%s', response='%d'", 
	                  self->request_method, self->response_code);
	      z_proxy_leave(self);
	      return FALSE;
            }
          switch (rc)
            {
            case HTTP_RSP_ACCEPT:
              break;
            default:
	    case HTTP_RSP_REJECT:
	      self->error_code = HTTP_MSG_POLICY_VIOLATION;
	      z_proxy_leave(self);
              return FALSE;
            }

        }
      
      if (!http_filter_headers(self, EP_SERVER, http_response_filter_connection_header))
        {
          z_proxy_leave(self);
          return FALSE;
        }
      http_log_headers(self, EP_SERVER, "postfilter");
    }
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_copy_response(HttpProxy *self)
{
  z_proxy_enter(self);
  
  if (!http_data_transfer_setup(self, EP_SERVER, EP_CLIENT, TRUE))
    {
      z_proxy_leave(self);
      return FALSE;
    }
  
  if (self->proto_version[EP_SERVER] >= 0x0100)
    {
      gchar *response; 
      
      http_flat_headers(&self->headers[EP_SERVER]);
      response = alloca(strlen(self->response_version) + strlen(self->response) + strlen(self->response_msg) + self->headers[EP_SERVER].flat->len + 8);
      sprintf(response, "%s %s %s\r\n%s\r\n", self->response_version, self->response, self->response_msg, self->headers[EP_SERVER].flat->str);
      if (http_write(self, EP_CLIENT, response, strlen(response)) != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self);
          return FALSE;
        }
    }

  if (((self->request_flags & HTTP_REQ_FLG_HEAD) != 0) ||
      ((self->response_flags & HTTP_RESP_FLG_NODATA) != 0))
    {
      /* no need to transfer data */
      ;
    }
  else
    {
      if (!http_data_transfer(self))
        {
          z_proxy_leave(self);
	  return FALSE;
	}
    }
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_handle_connect(HttpProxy *self)
{
  /* connect is enabled here */
  guint i, rc; 
  ZPolicyObj *res;
  gboolean called;
  gchar *success, *remote_host;
  gchar *colon, *end;
  guint remote_host_len, remote_port;

  z_proxy_enter(self);

  colon = strchr(self->request_url->str, ':');
  if (colon)
    {
      remote_host = self->request_url->str;
      remote_host_len = colon - remote_host;
      remote_port = strtoul(colon + 1, &end, 10);
    }
  else
    {
      self->error_code = HTTP_MSG_CLIENT_SYNTAX;
      g_string_sprintf(self->error_info, "Invalid CONNECT request.");
      z_proxy_leave(self);
      return FALSE;
    }

  if (self->parent_proxy->len)
    {
      strncpy(self->remote_server, self->parent_proxy->str, sizeof(self->remote_server) - 1);
      self->remote_server[sizeof(self->remote_server)] = 0;
      self->remote_port = self->parent_proxy_port;
    }
  else
    {
      strncpy(self->remote_server, remote_host, MIN(sizeof(self->remote_server) - 1, remote_host_len));
      self->remote_server[sizeof(self->remote_server)] = 0;
      self->remote_port = remote_port;
    }
  if (!http_connect_server(self))
    {
      z_proxy_leave(self);
      return FALSE;
    }
    
  if (self->parent_proxy->len)
    {
      gchar buf[2048];
      
      snprintf(buf, sizeof(buf), "CONNECT %.*s:%d HTTP/1.0\r\n\r\n", remote_host_len, remote_host, remote_port);
      
      if (http_write(self, EP_SERVER, buf, strlen(buf)) != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self);
          return FALSE;
        }
      if (!http_fetch_response(self))
        {
          z_proxy_leave(self);
          return FALSE;
        }
      if (self->response_code != 200)
        {
          /*LOG
            This is placeholder 14.
           */
          z_proxy_log(self, HTTP_ERROR, 1, "Parent proxy refused our CONNECT request; host='%.*s', port='%d'", remote_host_len, remote_host, remote_port);
          z_proxy_leave(self);
          return FALSE;
        }
    }
  
  /*LOG
    This is placeholder 15.
   */
  z_proxy_log(self, HTTP_DEBUG, 6, "starting connect method;");
  
  success = "HTTP/1.0 200 Connection established\r\n\r\n";
  if (http_write(self, EP_CLIENT, success, strlen(success)) != G_IO_STATUS_NORMAL)
    {
      /* hmm... I/O error */
      z_proxy_leave(self);
      return FALSE;
    }
  
  z_policy_lock(self->super.thread);
  res = z_policy_call(self->super.handler, "connectMethod", z_policy_var_build("()"), &called, self->super.session_id);
  
  if (!z_policy_var_parse(res, "i", &rc))
    {
      self->error_code = HTTP_MSG_POLICY_SYNTAX;
      z_policy_var_unref(res);
      z_policy_unlock(self->super.thread);
      z_proxy_leave(self);
      return FALSE;
    }
  z_policy_var_unref(res);
  z_policy_unlock(self->super.thread);
  
  if (rc != HTTP_REQ_ACCEPT)
    {
      g_string_assign(self->error_info, "CONNECT request denied by policy.");
      z_proxy_leave(self);
      return FALSE;
    }

  /* release, but don't close fds */
  for (i = EP_CLIENT; i < EP_MAX; i++)
    {
      z_stream_unref(self->super.endpoints[i]);
      self->super.endpoints[i] = NULL;
    }

  z_proxy_leave(self);
  return TRUE;
}

static void
http_main(HttpProxy *self)
{
  self->request_count = 0;
  
  z_proxy_enter(self);
  /* loop for keep-alive */
  while (1)
    {
      self->error_code = -1;
      /*LOG
        This is placeholder 16.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "fetching request and headers;");
      /* fetch request */
      if (!http_fetch_request(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_CLIENT_SYNTAX;
	  break;
	}
      /*LOG
        This is placeholder 17.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "processing request and headers;");
      if (!http_process_request(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_CLIENT_SYNTAX;
	  break;
	}

      if (self->max_keepalive_requests != 0 && 
          self->request_count >= self->max_keepalive_requests - 1)
        {
          /*LOG
            This is placeholder 17.
           */
          z_proxy_log(self, HTTP_POLICY, 3, 
                      "max_keepalive_request reached, setting connection mode to close; count='%d', max='%d'", 
                      self->request_count, self->max_keepalive_requests);
          self->connection_mode = HTTP_CONNECTION_CLOSE;
        }

      /*LOG
        This is placeholder 18.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "filtering request and headers;");
      if (!http_filter_request(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_POLICY_SYNTAX;
	  break;
	}
      
      if ((self->request_flags & HTTP_REQ_FLG_CONNECT))
        {
          if (http_handle_connect(self))
            {
              /* connect method was successful, we can now safely quit */
              z_proxy_leave(self);
              return;
            }
          else
            break;
        }
      /*LOG
        This is placeholder 19.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "sending request and headers, copying request data;");
      if (!http_copy_request(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_CLIENT_SYNTAX;
	  break;
	}
      /*LOG
        This is placeholder 20.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "fetching response and headers;");
      if (!http_fetch_response(self))
        {
          if (self->error_code == -1)
            {
              self->error_code = HTTP_MSG_SERVER_SYNTAX;
            }
	  break;
	}
      /*LOG
        This is placeholder 21.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "processing response and headers;");
      if (!http_process_response(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_SERVER_SYNTAX;
          break;
        }
      /*LOG
        This is placeholder 22.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "filtering response and headers;");
      if (!http_filter_response(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_POLICY_SYNTAX;
          break;
        }
      /*LOG
        This is placeholder 23.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "copying response and headers, copying response data;");
      if (!http_copy_response(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_SERVER_SYNTAX;
          break;
        }
      if (self->connection_mode == HTTP_CONNECTION_CLOSE)
      	break;
	
      self->request_count++;
    }
  /*LOG
    This is placeholder 24.
   */
  z_proxy_log(self, HTTP_DEBUG, 6, "exiting keep-alive loop;");
  if (self->error_code != -1)
    {
      http_error_message(self, self->error_status, self->error_code, self->error_info);
    }
  z_proxy_leave(self);
}

static gpointer
http_thread(void *s)
{
  HttpProxy *self = (HttpProxy *) s;

  z_proxy_enter(self);
  http_config_set_defaults(self);
  http_register_vars(self);

  ZPROXY_SET_STATE(self, PS_CONFIG);
  if (z_proxy_config_event(&self->super))
    {
  
      http_config_init(self);
      http_client_stream_init(self);

      ZPROXY_SET_STATE(self, PS_STARTING_UP);
      if (z_proxy_startup_event(&self->super))
        {

          ZPROXY_SET_STATE(self, PS_WORKING);

          http_main(self);

          ZPROXY_SET_STATE(self, PS_SHUTTING_DOWN);
          z_proxy_shutdown_event(&self->super);
        }
    }
  z_proxy_destroy(&self->super);
  z_leave();
  return NULL;
}

static void
http_proxy_free(ZProxy *s)
{
  HttpProxy *self = (HttpProxy *) s;
  guint i;

  z_proxy_enter(s);
  for (i = EP_CLIENT; i < EP_MAX; i++)
    {
      http_destroy_headers(&self->headers[i]);
    }

  g_string_free(self->error_headers, TRUE);
  g_string_free(self->old_auth_header, TRUE);

  /* NOTE: hashes are freed up by pyvars */
  z_proxy_leave(s);
}

static ZProxy *
http_proxy_new(gchar *session_id, ZStream *client, ZPolicyObj *handler)
{
  HttpProxy *self = g_new0(HttpProxy, 1);

  z_enter();
  z_proxy_init(&self->super, session_id, client, handler);
  self->super.free = http_proxy_free;
  
  z_thread_new(session_id, http_thread, self);
  
  z_leave();
  return (&self->super);
}

gint
zorp_module_init(void)
{
  z_registry_add("http", ZR_PROXY, http_proxy_new);
  return TRUE;
}
