/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: attach.c,v 1.29 2004/01/26 16:55:34 bazsi Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/attach.h>
#include <zorp/connect.h>
#include <zorp/conntrack.h>
#include <zorp/log.h>
#include <zorp/streamfd.h>

#include <string.h>

/*
 * Attach - establish outgoing connection
 * 
 */

struct _ZAttach
{
  GStaticRecMutex lock; /* lock protecting ref_cnt */
  gint ref_cnt;
  gchar session_id[MAX_SESSION_ID];
  guint proto;
  ZSockAddr *bind_addr;
  ZSockAddr *local;
  ZSockAddr *remote;
  ZAttachParams params;
  ZAttachCallback callback;
  gpointer user_data;
  GDestroyNotify user_data_notify;
  
  GMutex *connected_lock;
  GCond *connected_cond;
  ZConnection *conn;
  
  union 
  {
    ZIOConnect *connect;
  } c;
};

#define not_connected_yet_mark ((ZConnection *) &z_attach_new)


/**
 * z_attach_free:
 * @self this
 *
 * Destructor, destroyes the connection, decrements the ref counters of sub-objects
 * 
 */
static void
z_attach_free(ZAttach *self)
{
  z_session_enter(self->session_id);
  
  if (self->conn && self->conn != not_connected_yet_mark)
    z_connection_destroy(self->conn, FALSE);
  
  z_sockaddr_unref(self->bind_addr);
  z_sockaddr_unref(self->local);
  z_sockaddr_unref(self->remote);
  switch (self->proto)
    {
    case ZD_PROTO_TCP:
      if (self->c.connect)
        z_io_connect_unref(self->c.connect);
      break;
    }
  if (self->connected_lock)
    {
      g_mutex_free(self->connected_lock);
      g_cond_free(self->connected_cond);
    }
  g_free(self);
  z_leave();
}


/**
 * z_attach_ref:
 * @self this
 *
 * Increments the reference counter of the instance
 *
 * Returns:
 * The instance
 */
ZAttach *
z_attach_ref(ZAttach *self)
{
  g_static_rec_mutex_lock(&self->lock);
  g_assert(self->ref_cnt);
  self->ref_cnt++;
  g_static_rec_mutex_unlock(&self->lock);
  return self;
}


/**
 * z_attach_unref:
 * @self this
 *
 * Decrements the reference counter of the instance
 *
 * Returns:
 * The instance
 */
void
z_attach_unref(ZAttach *self)
{
  g_static_rec_mutex_lock(&self->lock);
  g_assert(self->ref_cnt);
  if (--self->ref_cnt == 0)
    {
      g_static_rec_mutex_unlock(&self->lock);
      z_attach_free(self);
    }
  else
    g_static_rec_mutex_unlock(&self->lock);
}


/**
 * z_attach_callback:
 * @self this
 * @conn The connection to add
 *
 * Internal callback function, called when a connection is established.
 * Called from: z_attach_tcp_callback (tcp) or z_attach_start (udp).
 */
static void
z_attach_callback(ZAttach *self, ZConnection *conn)
{
  gchar buf[256];

  z_session_enter(self->session_id);
  /*LOG
    This message reports that the connection was successfully established.
  */
  z_log(NULL, CORE_DEBUG, 6, "Established connection; %s", z_connection_format(conn, buf, sizeof(buf)));
  if (self->callback)
    {
      self->callback(conn, self->user_data);
      if (self->user_data && self->user_data_notify)
        {
          self->user_data_notify(self->user_data);
        }
      self->user_data = NULL;
    }
  else
    {
      g_mutex_lock(self->connected_lock);
      self->conn = conn;
      g_cond_signal(self->connected_cond);
      g_mutex_unlock(self->connected_lock);
    }
  z_session_leave(self->session_id);
}


/**
 * z_attach_tcp_callback:
 * @fd socket descriptor
 * @error not used
 * @user_data this
 *
 * Internal callback function, called from z_attach_start/z_io_connect_new when
 * a new tcp connection is established.
 */
static void
z_attach_tcp_callback(gint fd, GError *error G_GNUC_UNUSED, gpointer user_data)
{
  ZAttach *self = (ZAttach *) user_data;
  ZConnection *conn;
  
  z_session_enter(self->session_id);

  if (fd != -1)
    {
      conn = z_connection_new();
      if (z_getsockname(fd, &conn->local, 0) != G_IO_STATUS_NORMAL ||
          z_getpeername(fd, &conn->remote, 0) != G_IO_STATUS_NORMAL)   
        {
          z_connection_destroy(conn, FALSE);
          close(fd);
          z_session_leave(self->session_id);
          return;   
        }
      conn->protocol = ZD_PROTO_TCP;
      conn->stream = z_stream_fd_new(fd, "");
      conn->dest = z_sockaddr_ref(conn->remote);
      conn->bound = z_sockaddr_ref(self->local);
    }
  else
    conn = NULL;
  
  z_attach_callback(self, conn);
  z_session_leave(self->session_id);
}


/**
 * z_attach_start:
 * @self this
 *
 * Initiate establishing a connection
 *
 * Returns:
 * TRUE on success
 */
gboolean
z_attach_start(ZAttach *self)
{
  gboolean res = FALSE;
  
  z_session_enter(self->session_id);
  switch (self->proto)
    {
    case ZD_PROTO_TCP:
      {
        self->c.connect = z_io_connect_new(self->session_id, self->bind_addr, self->remote, (self->params.tcp.loose ? ZSF_LOOSE_BIND : 0) | ZSF_MARK_TPROXY, self->params.tcp.tos, z_attach_tcp_callback, self);
        if (self->c.connect)
          {
            z_io_connect_set_timeout(self->c.connect, self->params.tcp.timeout);
            z_attach_ref(self);
            z_io_connect_set_destroy_notify(self->c.connect, (GDestroyNotify) z_attach_unref);
            if (self->callback)
              {
                self->local = z_io_connect_start(self->c.connect);
              }
            else
              {
                self->local = z_io_connect_start_block(self->c.connect);
              }
            if (!self->local)
              {
                z_io_connect_unref(self->c.connect);
                self->c.connect = NULL;
              }
            else
              res = TRUE;
          }
        break;
      }
#if ENABLE_CONNTRACK
    case ZD_PROTO_UDP:
      {
        ZDGramConnection *sock;
        ZStream *proxy_stream;
        ZConnection *conn;
        
        sock = z_dgram_connection_new(self->session_id, self->remote, self->bind_addr, ZCS_TO_SERVER, self->params.udp.tos, &proxy_stream);
        
        if (sock)
          {
            conn = z_connection_new();
            conn->protocol = ZD_PROTO_UDP;
            conn->stream = proxy_stream;
            conn->local = z_sockaddr_ref(sock->local_addr);
            conn->bound = z_sockaddr_ref(sock->local_addr);
            self->local = z_sockaddr_ref(sock->local_addr);
            conn->remote = z_sockaddr_ref(sock->remote_addr);
            conn->dest = z_sockaddr_ref(sock->remote_addr);
            z_attach_callback(self, conn);
            res = TRUE;
            
            /* start forwarding */
            z_dgram_connection_start(sock);
            z_dgram_connection_unref(sock);
          }
        break;
      }
#endif
    }
  z_session_leave(self->session_id);
  return res;
}


/**
 * z_attach_block:
 * @self this
 *
 * Wait until the connection is established
 *
 * Returns:
 * The established connection
 */
ZConnection *
z_attach_block(ZAttach *self)
{
  ZConnection *conn;
  
  z_session_enter(self->session_id);
  z_attach_ref(self);
  
  g_mutex_lock(self->connected_lock);
  while (self->conn == not_connected_yet_mark)
    g_cond_wait(self->connected_cond, self->connected_lock);
  conn = self->conn;
  self->conn = NULL;
  g_mutex_unlock(self->connected_lock);
  
  z_attach_unref(self);
  z_session_leave(self->session_id);
  return conn;
}


/**
 * z_attach_cancel:
 * @self this
 *
 * Cancel a connection, and call the notification callback if the connection
 * wasn't established (and so the callback wasn't called yet).
 */
void
z_attach_cancel(ZAttach *self)
{
  z_session_enter(self->session_id);
  switch (self->proto)
    {
    case ZD_PROTO_TCP:
      if (self->c.connect)
        z_io_connect_cancel(self->c.connect);
      break;
    case ZD_PROTO_UDP:
      break;
    }
    
  /* either our callback was called in which case user_data is already
   * NULLed or we _have_ to call user_data_notify here. */
  
  if (self->user_data && self->user_data_notify)
    {
      self->user_data_notify(self->user_data);
     }
  self->user_data = NULL;
  z_session_leave(self->session_id);
}


/**
 * z_attach_get_local:
 * @self this
 *
 * Get the address of the connection's local endpoint.
 *
 * Returns:
 * The address of the local endpoint
 */
ZSockAddr *
z_attach_get_local(ZAttach *self)
{
  return z_sockaddr_ref(self->local);
}


/**
 * z_attach_new:
 * @session_id The ID of the session
 * @bind_addr The address to bind to
 * @remote The address to connect to
 * @params The optional parameters for the connection
 * @callback Callback function to call when the connection is established
 * @notify Callback to call when the structure is destroyed
 *
 * Allocates and sets up a new instance of ZAttach.
 * (For the connection parameters see ZAttachTCPParams and ZAttachUDPParams.)
 *
 * Returns:
 * The new instance
 */
ZAttach *
z_attach_new(gchar *session_id, 
             guint proto, ZSockAddr *bind_addr, ZSockAddr *remote, 
             ZAttachParams *params,
             ZAttachCallback callback, gpointer user_data, GDestroyNotify notify)
{
  ZAttach *self = g_new0(ZAttach, 1);
  
  z_session_enter(session_id);
  g_strlcpy(self->session_id, session_id, sizeof(self->session_id));
  self->proto = proto;
  self->bind_addr = z_sockaddr_ref(bind_addr);
  self->remote = z_sockaddr_ref(remote);
  self->callback = callback;
  self->user_data = user_data;
  self->user_data_notify = notify;
  self->ref_cnt = 1;
  memcpy(&self->params, params, sizeof(self->params));
  if (!callback)
    {
      self->connected_lock = g_mutex_new();
      self->connected_cond = g_cond_new();
    }
  self->conn = not_connected_yet_mark;
  z_session_leave(session_id);
  return self;
}
