/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: proxy.c,v 1.132 2004/07/05 16:57:46 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor : kisza
 * Last audited version: 1.23
 * Notes:
 *
 ***************************************************************************/

#include <zorp/proxy.h>
#include <zorp/policy.h>
#include <zorp/proxyvars.h>
#include <zorp/streamfd.h>
#include <zorp/log.h>
#include <zorp/io.h>
#include <zorp/thread.h>

#include <zorp/pystream.h>
#include <zorp/pysockaddr.h>
#include <zorp/pyproxy.h>

#include <stdarg.h>
#include <sys/socket.h>
#include <unistd.h>

/*
 * References between child and parent proxies work as follows
 *
 * When a proxy wants to stack a child proxy a ZStackedProxy structure is
 * created which contains the stream objects towards the child, and also
 * contains references to both the proxy and its child. These references are
 * removed when the ZStackedProxy structure is freed using
 * z_stacked_proxy_destroy.
 *
 * When the child proxy starts up it adds a reference to its parent still in
 * the parent proxy's thread using z_proxy_add_child() in z_proxy_new(). 
 * This adds a reference from child to parent through its parent_proxy
 * field, and from parent to child through its child_proxies list. This is a
 * circular reference.
 *
 * 1) The child proxy exits first
 * 
 *    When the child proxy exits it calls z_proxy_destroy() which in turn
 *    calls z_proxy_set_parent(NULL) which drops the reference to its
 *    parent. The circular reference is now resolved. The parent will detect
 *    that its child exited (probably one of the streams indicates EOF) and
 *    calls z_stacked_proxy_destroy() for the ZStackedProxy structure
 *    associated with this child. It closes and frees streams, and removes
 *    the child proxy from the parent->child_proxies list.
 *
 * 2) The parent proxy exits first
 *
 *    It is assumed that ZStackedProxy structures associated with any child
 *    proxy is freed prior to calling z_proxy_destroy(). This assumption is
 *    valid as ZTransfer or proxy specific transfer code calls
 *    z_stacked_proxy_destroy(), thus the only remaining reference to the
 *    child proxy instance is though child_proxies.
 *    
 *    When the parent proxy exits, it calls z_proxy_destroy() which in turn
 *    frees every item on its child_proxies list.
 *
 *  3) The parent and child proxies exit at the same time
 * 
 *    In this case the exit is not synchronized by the EOF the parent reads
 *    from the child. Thus z_stacked_proxy_destroy() and z_proxy_destroy()
 *    might race on the following (possibly shared) data accesses:
 *     z_stacked_proxy_destroy:
 *       child is removed from parent->child_list
 *     z_proxy_destroy (parent):
 *       child is removed from parent->child_list (if present)
 *     z_proxy_destroy (child):
 *       child->parent is set to NULL
 *
 * Synchronization during reference counting
 *
 * The general rule is that every proxy instance modifies its own data
 * fields only in order to avoid locking.  The only locks used are the
 * recursive mutexes protecting the reference counts. The only exception to
 * this rule happens when the child proxy starts up and adds itself to its
 * parent's child_proxies list. The synchronization here is also simple as
 * this happens in the parent proxy's thread, thus no locks are necessary.
 * 
 */


/**
 * z_proxy_policy_call_event:
 * @self this #ZProxy instance
 * @event the called python event
 *
 * Thiis function call the @event event from current instance.
 *
 * Returns: a boolean value
 */
static gboolean
z_proxy_policy_call_event(ZProxy *self, gchar *event, gchar *old_event_name)
{
  ZPolicyObj *res;
  gboolean called;
  /*LOG
    This message reports that Zorp is about to call the proxy's %event() event.
   */
  z_proxy_log(self, CORE_DEBUG, 7, "calling %s() event;", event);
  res = z_policy_call(self->handler, event, NULL, &called, self->session_id);
  if (!called && old_event_name)
    {
      static gboolean obsolete_name_logged = FALSE;

      z_policy_var_unref(res);
      res = z_policy_call(self->handler, old_event_name, NULL, &called, self->session_id);
      
      if (!obsolete_name_logged && called)
        {
          obsolete_name_logged = TRUE;
          z_proxy_log(self, CORE_POLICY, 0, "Obsolete policy handler in Proxy definition; new_name='%s', old_name='%s'", event, old_event_name);
        }
    }
  if (res == NULL && called)
    {
      z_proxy_leave(self);
      return FALSE;
    }
  z_policy_var_unref(res);
  return TRUE;
}

/**
 * z_proxy_policy_call:
 * @self this #ZProxy instance
 * @event the called python event family
 *
 * This function call a python event family.
 * If event named to "event" it's first call
 * __pre_event__. If the call was success it's
 * call event and for last it's call __post_event__.
 *
 * Returns: TRUE if all call is success, FALSE otherwise
 */
static gboolean
z_proxy_policy_call(ZProxy *self, gchar *event, gchar *old_event_name)
{
  gchar event_string[512];
  
  z_proxy_enter(self);
  
  z_policy_thread_acquire(self->thread);

  g_snprintf(event_string, sizeof(event_string), "__pre_%s__", event);
  if (z_proxy_policy_call_event(self, event_string, NULL))
    {
      if (z_proxy_policy_call_event(self, event, old_event_name))
        {
          g_snprintf(event_string, sizeof(event_string), "__post_%s__", event);
          if (z_proxy_policy_call_event(self, event_string, NULL))
            {
              z_policy_thread_release(self->thread);
              return TRUE;
            }
        }
    }

  z_policy_thread_release(self->thread);

  z_proxy_leave(self);
  return FALSE;
}

/**
 * z_proxy_policy_config:
 * @self: this ZProxy instance
 *
 * Acquires the thread associated with this Proxy instance and calls
 * the __pre_config__, config and __post_config__ events.
 *
 **/
gboolean
z_proxy_policy_config(ZProxy *self)
{
  z_proxy_enter(self);
  
  z_proxy_set_state(self, ZPS_CONFIG);

  if (!z_proxy_policy_call(self, "config", NULL))
    {
      z_proxy_leave(self);
      return FALSE;
    }
  
  z_policy_thread_acquire(self->thread);
  z_proxy_vars_dump_values(self->vars, self);
  z_policy_thread_release(self->thread);

  z_proxy_leave(self);
  return TRUE;
}

/**
 * z_proxy_policy_startup:
 * @self: this ZProxy instance
 *
 * Acquires the thread associated with this TProxy instance and calls
 * the __pre_startup__, startup and __post_startup__ events.
 **/
gboolean
z_proxy_policy_startup(ZProxy *self)
{
  z_proxy_enter(self);
  z_proxy_set_state(self, ZPS_STARTING_UP);

  if (!z_proxy_policy_call(self, "startup", "startUp"))
    {
      z_proxy_leave(self);
      return FALSE;
    }

  z_proxy_set_state(self, ZPS_WORKING);

  z_proxy_leave(self);
  return TRUE;
}

/**
 * z_proxy_policy_shutdown:
 * @self: this ZProxy instance
 *
 * Acquires the thread associated with this TProxy instance and calls
 * the __pre_shutdown__, shutdown and __post_shutdown events.
 **/
void 
z_proxy_policy_shutdown(ZProxy *self)
{
  z_proxy_enter(self);
  z_proxy_set_state(self, ZPS_SHUTTING_DOWN);

  z_proxy_policy_call(self, "shutdown", "shutDown");

  z_proxy_leave(self);
}

/**
 * z_proxy_policy_destroy:
 * @self: this ZProxy instance
 *
 * Acquires the thread associated with this TProxy instance and calls
 * the __destroy__ event.
 **/
void 
z_proxy_policy_destroy(ZProxy *self)
{
  ZPolicyObj *res;
  gboolean called;

  z_proxy_enter(self);
  /*LOG
    This message reports that Zorp is about to call the proxy's __destroy__() event.
    This method handles the pre destroy tasks, like the shutdown of the server side connection.
   */
  z_proxy_log(self, CORE_DEBUG, 7, "calling __destroy__() event;");
  z_policy_thread_acquire(self->thread);
  res = z_policy_call(self->handler, "__destroy__", NULL, &called, self->session_id);
  z_policy_var_unref(res);
  z_policy_thread_release(self->thread);
  z_proxy_leave(self);
}

/**
 * z_proxy_connect_server:
 * @self: proxy instance
 * @host: host to connect to, used as a hint by the policy layer but may as well be ignored
 * @port: port in host to connect to
 * 
 * Send a connectServer event to the associated policy object.  Returns TRUE
 * if the server-side connection is established, otherwise the
 * connection to the client should be closed.
 **/
gboolean
z_proxy_connect_server(ZProxy *self, char *host, gint port)
{
  ZPolicyObj *res, *args;
  gint rc;
  gboolean called;
  
  z_proxy_enter(self);

  z_policy_thread_acquire(self->thread);
  self->endpoints[EP_SERVER] = NULL;
  if (host)
    {
      args = z_policy_var_build("(si)", host, port);
      res = z_policy_call(self->handler, "setServerAddress", args, &called, self->session_id);
      if (!res)
        {
          z_policy_thread_release(self->thread);
          z_proxy_leave(self);
          return FALSE;
        }
      if (!z_policy_var_parse(res, "i", &rc) || !rc)
        {
          z_policy_thread_release(self->thread);
          z_proxy_leave(self);
          return FALSE;
        }
      z_policy_var_unref(res);
    }

  res = z_policy_call(self->handler, "connectServer", NULL, &called, self->session_id);

  if (res && z_py_zorp_stream_check(res))
    {
      self->endpoints[EP_SERVER] = ((ZorpStream *) res)->stream;
      z_stream_ref(self->endpoints[EP_SERVER]);
      rc = 1;
    }
  else
    {
      rc = 0;
    }
  z_policy_var_unref(res);
  z_policy_thread_release(self->thread);
  z_proxy_leave(self);
  return rc;
}

/**
 * z_proxy_user_authenticated:
 * @self: proxy instance
 * @entity: the name of the authenticated entity
 *
 * This function is called by the proxy when it decides that the user is
 * authenticated by some inband authentication method.
 **/
gboolean
z_proxy_user_authenticated(ZProxy *self, gchar *entity)
{
  ZPolicyObj *res;
  gboolean called;
  gboolean rc = TRUE;
  
  z_proxy_enter(self);
  z_policy_thread_acquire(self->thread);

  res = z_policy_call(self->handler, "userAuthenticated", z_policy_var_build("(sOs)", entity, z_policy_none, "inband"), &called, self->session_id);
  if (!res)
    rc = FALSE;
  z_policy_var_unref(res);
  z_policy_thread_release(self->thread);
  z_proxy_leave(self);
  return rc;
}

/**
 * z_proxy_stack_prepare_streams:
 * @self: ZProxy instance
 * @downpair:
 * @uppair:
 *
 **/
static gboolean
z_proxy_stack_prepare_streams(ZProxy *self, gint *downpair, gint *uppair)
{
  z_proxy_enter(self);

  if (socketpair(AF_UNIX, SOCK_STREAM, 0, downpair) == -1)
    {
      /*LOG
        This message indicates that stacking a child proxy failed, because
        creating an AF_UNIX domain socketpair failed on the client side.
       */
      z_proxy_log(self, CORE_ERROR, 1, "Error creating client socketpair for stacked proxy; error='%s'", g_strerror(errno));
      z_proxy_leave(self);
      return FALSE;
    }
  else if (socketpair(AF_UNIX, SOCK_STREAM, 0, uppair) == -1)
    {
      close(downpair[0]);
      close(downpair[1]);
      /*LOG
        This message indicates that stacking a child proxy failed, because
        creating an AF_UNIX domain socketpair failed on the server side.
       */
      z_proxy_log(self, CORE_ERROR, 1, "Error creating server socketpair for stacked proxy; error='%s'", g_strerror(errno));
      z_proxy_leave(self);
      return FALSE;
    }
  z_proxy_leave(self);
  return TRUE;
}

/**
 * z_proxy_stack_proxy:
 * @self: proxy instance
 * @proxy_class: a Python class to be instantiated as the child proxy
 *
 * This function is called to start a child proxy.
 **/
ZStackedProxy *
z_proxy_stack_object(ZProxy *self, ZPolicyObj *proxy_class)
{
  int downpair[2], uppair[2];
  ZPolicyObj *res, *client_stream, *server_stream;
  ZStackedProxy *stacked;
  ZStream *tmpstream;
  ZStream *client_upstream, *server_upstream;
  
  z_proxy_enter(self);
  if (proxy_class == z_policy_none)
    { 
      z_policy_var_unref(proxy_class);
      z_proxy_leave(self);
      return NULL;
    }
  
  if (!z_proxy_stack_prepare_streams(self, downpair, uppair))
    {
      z_policy_var_unref(proxy_class);
      z_proxy_leave(self);
      return NULL;
    }
  
  /*LOG
    This message reports that Zorp is about to stack a proxy class
    with the given fds as communication channels.
   */
  z_proxy_log(self, CORE_DEBUG, 6, "Stacking subproxy; client='%d:%d', server='%d:%d'", downpair[0], downpair[1], uppair[0], uppair[1]);
  
  tmpstream = z_stream_fd_new(downpair[1], "");
  client_stream = z_py_stream_new(tmpstream);
  z_stream_unref(tmpstream);
  
  tmpstream = z_stream_fd_new(uppair[1], "");
  server_stream = z_py_stream_new(tmpstream);
  z_stream_unref(tmpstream);
  res = z_policy_call(self->handler, "stackProxy", z_policy_var_build("(OOO)", client_stream, server_stream, proxy_class), NULL, self->session_id);
  
  z_policy_var_unref(client_stream);
  z_policy_var_unref(server_stream);
  
  if (!res || res == z_policy_none)
    {
      close(downpair[0]);
      close(downpair[1]);
      close(uppair[0]);
      close(uppair[1]);
      z_policy_var_unref(res);
      z_proxy_leave(self);
      return NULL;
    }

  client_upstream = z_stream_fd_new(downpair[0], "");
  server_upstream = z_stream_fd_new(uppair[0], "");
  stacked = z_stacked_proxy_new(client_upstream, server_upstream, NULL, self, z_py_zorp_proxy_get_proxy(res));
  z_policy_var_unref(res);
  
  z_proxy_leave(self);
  return stacked;
}


/**
 * z_proxy_get_addresses:
 * @self: proxy instance
 * @protocol: the protocol number (ZD_PROTO_*) is returned here
 * @client_address: the remote address of the client is returned here
 * @client_local: the local address of the connection to the client is returned here
 * @server_address: the remote address of the server is returned here
 * @server_local: the local address of the connection to the server is returned here
 * @client_listen: the address of the listener which initiated this session is returned here
 *
 * This function is used to query the addresses used to connecting the proxy
 * to the client and server. The utilized application protocol is also
 * returned and the listener address which accepted the connection.
 **/
gboolean
z_proxy_get_addresses(ZProxy *self, 
                      guint *protocol,
                      ZSockAddr **client_address, ZSockAddr **client_local,
                      ZSockAddr **server_address, ZSockAddr **server_local,
                      ZSockAddr **client_listen)
{
  ZorpSockAddr *sockaddr;

  z_proxy_enter(self);
  
  z_policy_thread_acquire(self->thread);

  if (protocol)
    {
      ZPolicyObj *pyproto;
      
      pyproto = z_session_getattr(self->handler, "protocol");
      if (PyInt_Check(pyproto))
        *protocol = PyInt_AsLong(pyproto);
      else
        *protocol = ZD_PROTO_TCP;
      z_policy_var_unref(pyproto);
    }

  if (client_address)
    {
      sockaddr = (ZorpSockAddr *) z_session_getattr(self->handler, "client_address");
      if (z_py_zorp_sockaddr_check(sockaddr))
        *client_address = z_sockaddr_ref(sockaddr->sa);
      else 
        *client_address = NULL;
      z_policy_var_unref(sockaddr);
    }

  if (client_local)
    {
      sockaddr = (ZorpSockAddr *) z_session_getattr(self->handler, "client_local");
      if (z_py_zorp_sockaddr_check(sockaddr))
        *client_local = z_sockaddr_ref(sockaddr->sa);
      else 
        *client_local = NULL;
      z_policy_var_unref(sockaddr);
    }

  if (client_listen)
    {
      sockaddr = (ZorpSockAddr *) z_session_getattr(self->handler, "client_listen");
      if (z_py_zorp_sockaddr_check(sockaddr))
        *client_listen = z_sockaddr_ref(sockaddr->sa);
      else 
        *client_listen = NULL;
      z_policy_var_unref(sockaddr);
    }

  if (server_address)
    {
      sockaddr = (ZorpSockAddr *) z_session_getattr(self->handler, "server_address");
      if (z_py_zorp_sockaddr_check(sockaddr))
        *server_address = z_sockaddr_ref(sockaddr->sa);
      else
        *server_address = NULL;
      z_policy_var_unref(sockaddr);
    }

  if (server_local)
    {
      sockaddr = (ZorpSockAddr *) z_session_getattr(self->handler, "server_local");
      if (z_py_zorp_sockaddr_check(sockaddr))
        *server_local = z_sockaddr_ref(sockaddr->sa);
      else
        *server_local = NULL;
      z_policy_var_unref(sockaddr);
    }

  z_policy_thread_release(self->thread);
  z_proxy_leave(self);
  return TRUE;
}

/**
 * z_proxy_check_secondary:
 * @self: ZProxy instance
 * @secondary_mask: which parts of the address is taken 'important'
 * @pri_client: the client address of the primary connection
 * @pri_server: the server address of the primary connection
 * @sec_client: the client address of the secondary connection
 * @sec_server: the server address of the secondary connection
 *
 * Check if the specified secondary connection matches our secondary accept
 * policy (specified as a bitset of ZS_* constants).
 *
 * Return TRUE if the connection is matching and FALSE otherwise.
 **/
gboolean
z_proxy_check_secondary(ZProxy *self G_GNUC_UNUSED, guint32 secondary_mask, 
                        ZSockAddr *pri_client, ZSockAddr *pri_server,
                        ZSockAddr *sec_client, ZSockAddr *sec_server)
{
  struct sockaddr_in *master_client, *master_server, *client, *server;
  gboolean res = TRUE;
  
  z_proxy_enter(self);
  client = (struct sockaddr_in *) &sec_client->sa;
  server = (struct sockaddr_in *) &sec_server->sa;
  master_client = (struct sockaddr_in *) &pri_client->sa;
  master_server = (struct sockaddr_in *) &pri_server->sa;
  
  if ((secondary_mask & ZS_MATCH_SADDR) &&
      client->sin_addr.s_addr != master_client->sin_addr.s_addr)
    {
      res = FALSE;
    }
  else if ((secondary_mask & ZS_MATCH_SPORT) &&
      client->sin_port != master_client->sin_port)
    {
      res = FALSE;
    }
  else if ((secondary_mask & ZS_MATCH_DADDR) &&
      server->sin_addr.s_addr != master_server->sin_addr.s_addr)
    {
      res = FALSE;
    }
  else if ((secondary_mask & ZS_MATCH_DPORT) &&
      server->sin_port != master_server->sin_port)
    {
      res = FALSE;
    }

  z_proxy_leave(self);
  return res;
}

/**
 * z_proxy_set_parent:
 * @self: ZProxy instance referring to self
 * @parent: ZProxy instance referring to the parent proxy
 *
 * This function is called to change the reference to the parent proxy.
 * A value of NULL specifies to drop the reference, anything else
 * removes the earlier reference and assigns a new one. See the 
 * comment on locking at the beginning of this file for more details.
 **/
gboolean
z_proxy_set_parent(ZProxy *self, ZProxy *parent)
{
  ZProxy *old_parent;

  z_proxy_enter(self);  
  if (parent)
    {
      /* establish parent link */
      if (!self->parent_proxy)
        {
          z_proxy_ref(parent);
          self->parent_proxy = parent;
        }
      else
        {
          z_proxy_leave(self);
          return FALSE;
        }
    }
  else
    {
      /* remove parent link */
      if (self->parent_proxy)
        {
          old_parent = self->parent_proxy;
          self->parent_proxy = parent;
          z_proxy_unref(old_parent);
        }
      else
        {
          z_proxy_leave(self);
          return FALSE;
        }
    }
  z_proxy_leave(self);
  return TRUE;
}

/**
 * z_proxy_add_child:
 * @self: ZProxy instance referring to self
 * @child_proxy: ZProxy instance to be added to the child list
 *
 * This function adds the specified ZProxy instance to the child_proxies
 * linked list.
 **/
gboolean
z_proxy_add_child(ZProxy *self, ZProxy *child_proxy)
{
  z_enter();
  if (z_proxy_set_parent(child_proxy, self))
    {
      self->child_proxies = g_list_prepend(self->child_proxies, z_proxy_ref(child_proxy));
      z_leave();
      return TRUE;
    }
  z_leave();
  return FALSE;
}

/**
 * z_proxy_del_child:
 * @self: ZProxy instance referring to self
 * @child_proxy: ZProxy instance to be deleted from the child_proxies list
 *
 * This function removes @child_proxy from the child_proxies list in @self.
 **/
gboolean
z_proxy_del_child(ZProxy *self, ZProxy *child_proxy)
{
  z_proxy_enter(self);

  self->child_proxies = g_list_remove(self->child_proxies, child_proxy);
  z_proxy_unref(child_proxy);

  z_proxy_leave(self);
  return TRUE;
}

/**
 * z_proxy_add_iface:
 * @self: ZProxy instance
 * @iface: exported interface to add
 *
 * This function adds an exported function interface callable from
 * other proxies to the set of supported interface.
 **/
void
z_proxy_add_iface(ZProxy *self, ZProxyIface *iface)
{
  z_object_ref(&iface->super);
  self->interfaces = g_list_prepend(self->interfaces, iface);
}

/**
 * z_proxy_del_iface:
 * @self: ZProxy instance
 * @iface: exported interface to delete
 *
 * This function deletes the interface specified in @iface from the
 * set of supported interfaces.
 **/
void
z_proxy_del_iface(ZProxy *self, ZProxyIface *iface)
{
  self->interfaces = g_list_remove(self->interfaces, iface);
  z_object_unref(&iface->super);
}

/** 
 * z_proxy_find_iface:
 * @self: ZProxy instance
 * @compat: search for an interface compatible with this class
 *
 * This function iterates on the set of supported interfaces in @self and
 * returns the first compatible with the class specified in @compat.
 **/
ZProxyIface *
z_proxy_find_iface(ZProxy *self, ZClass *compat)
{
  GList *p;
  if (!z_object_is_subclass(Z_CLASS(ZProxyIface), compat))
    {
      /*LOG
	This message indicates an internal error, please contact your Zorp support for assistance.
       */
      z_proxy_log(self, CORE_ERROR, 3, "Internal error, trying to look up a non-ZProxyIface compatible interface;");
      return NULL;
    }
  for (p = self->interfaces; p; p = p->next)
    {
      ZObject *obj;
      
      obj = (ZObject *) self->interfaces->data;
      if (z_object_is_compatible(obj, compat))
        return (ZProxyIface *) z_object_ref(obj);
    }
  return NULL;
}

/* methods for the ZProxy class */

/**
 * z_proxy_config_method:
 * @self: ZProxy instance
 *
 * This function is referenced as the default config method for the ZProxy
 * class. It calls the "config" method in the policy.
 * Returns FALSE upon failure, and TRUE otherwise.
 **/
gboolean
z_proxy_config_method(ZProxy *self)
{
  return z_proxy_policy_config(self);
}

/**
 * z_proxy_startup_method:
 * @self: ZProxy instance
 *
 * This function is referenced as the default startup method for the ZProxy
 * class. It calls the "startup" method in the policy.
 * Returns FALSE upon failure, and TRUE otherwise.
 **/
gboolean
z_proxy_startup_method(ZProxy *self)
{
  return z_proxy_policy_startup(self);
}

/**
 * z_proxy_main_method:
 * @self: ZProxy instance
 *
 * This function is referenced as the default main method for the ZProxy
 * class. Currently it does nothing and should be overriden in descendant
 * classes.
 **/
void
z_proxy_main_method(ZProxy *self G_GNUC_UNUSED)
{
  ;
}

/**
 * z_proxy_shutdown_method:
 * @self: ZProxy instance
 *
 * This function is referenced as the default shutdown method for the ZProxy
 * class. It calls the "shutdown" method in the policy.
 * Returns FALSE upon failure, and TRUE otherwise.
 **/
void
z_proxy_shutdown_method(ZProxy *self)
{
  z_proxy_policy_shutdown(self);
}

/**
 * z_proxy_destroy_method:
 * @self: proxy instance
 *
 * This function is called from proxy implementation when the proxy is to
 * exit. Frees up associated resources, closes streams, etc. Note that the
 * ZProxy instance is not freed immediately as the reference from Python and
 * the caller still exists. z_proxy_destroy() ensures however that circular
 * references are resolved so the proxy will be freed as soon as those
 * references are dropped.
 **/
void
z_proxy_destroy_method(ZProxy *self)
{
  int i;
  ZPolicyObj *handler;
  ZPolicyThread *thread;
  ZProxyVars *vars;

  z_proxy_enter(self);
  z_proxy_policy_destroy(self);
  z_proxy_set_state(self, ZPS_DESTROYING);

  /* this also removes the link to parent */

  z_proxy_set_parent(self, NULL);
  while (self->child_proxies)
    {
      z_proxy_del_child(self, (ZProxy *) self->child_proxies->data);
    }

    
  while (self->interfaces)
    {
      z_proxy_del_iface(self, (ZProxyIface *) self->interfaces->data);
    }
    
  for (i = EP_CLIENT; i <= EP_SERVER; i++)
    {
      if (self->endpoints[i])
        {
          z_stream_shutdown(self->endpoints[i], SHUT_RDWR, NULL);
          z_stream_close(self->endpoints[i], NULL);
          z_stream_unref(self->endpoints[i]);
          self->endpoints[i] = NULL;
        }
    }
  
  z_policy_thread_acquire(self->thread);
  thread = self->thread;
  self->thread = NULL;
  
  vars = self->vars;
  self->vars = NULL;
  z_proxy_vars_destroy(vars);

  handler = self->handler;
  self->handler = NULL;
  z_policy_var_unref(handler);  

  z_policy_thread_release(thread);
  
  z_policy_thread_destroy(thread);
  
  z_proxy_leave(self);
}

/**
 * z_proxy_run_method:
 * @self: ZProxy instance
 *
 * This function is referenced as the default run method for the ZProxy
 * class. It is started by the proxy specific thread and calls the
 * appropriate policy functions (config, startup), then continues by
 * calling z_proxy_main().
 **/
void
z_proxy_run_method(ZProxy *self)
{
  z_proxy_enter(self);
  if (z_proxy_config(self) &&
      z_proxy_startup(self))
    {
      z_proxy_main(self);
      
    }
  z_proxy_shutdown(self);
  z_proxy_destroy(self);
  z_proxy_leave(self);
}

/**
 * z_proxy_thread_func:
 * @s: ZProxy instance as a general pointer
 * 
 * This is the default thread function for proxies. The thread is started
 * in z_proxy_start().
 **/
static gpointer
z_proxy_thread_func(gpointer s)
{
  ZProxy *self = Z_CAST(s, ZProxy);

  z_proxy_run(self);
  z_proxy_unref(self);
  return NULL;
}

/**
 * z_proxy_start:
 * @self: ZProxy instance
 *
 * Starts the proxy by creating the new proxy thread. This function
 * is usually called by proxy constructors.
 **/
gboolean
z_proxy_start(ZProxy *self)
{
  z_proxy_ref(self);
  if (!z_thread_new(self->session_id, z_proxy_thread_func, self))
    {
      /*LOG
	This message indicates that Zorp was unable to create a new thread for the new proxy instance.
	It is likely that Zorp reached a thread limit, or no enugh resource is available.
       */
      z_proxy_log(self, CORE_ERROR, 2, "Error creating proxy thread;");
      z_proxy_unref(self);
      return FALSE;
    }
  return TRUE;
}

/**
 * z_proxy_new:
 * @proxy_class: proxy class to instantiate
 * @params: ZProxyParams containing ZProxy parameters
 *
 * This function is to be called from proxy constructors to initialize
 * common fields in the ZProxy struct. 
 *
 * NOTE: unlike in previous versions, z_proxy_new is called with the Python
 * interpreter unlocked, thus it must grab the interpreter lock to create
 * thread specific Python state.
 *
 **/
ZProxy *
z_proxy_new(ZClass *proxy_class, ZProxyParams *params)
{
  ZProxy *self;
  ZProxyIface *iface;
  ZPolicyThread *policy_thread;
  
  z_enter();
  self = Z_NEW_COMPAT(proxy_class, ZProxy);
  
  if (params->client)
    {
      self->endpoints[EP_CLIENT] = params->client;
      z_stream_ref(params->client);
    }

  g_strlcpy(self->session_id, params->session_id, sizeof(self->session_id));
  
  self->vars = z_proxy_vars_new();
  
  iface = (ZProxyIface *) z_proxy_basic_iface_new(Z_CLASS(ZProxyBasicIface), self);
  z_proxy_add_iface(self, iface);
  z_object_unref(&iface->super);
  
  z_python_lock();
  self->handler = params->handler;
  z_policy_var_ref(params->handler);
  policy_thread = z_policy_thread_self();
  self->thread = z_policy_thread_new(policy_thread ? z_policy_thread_get_policy(policy_thread) : current_policy);
  z_python_unlock();

  z_proxy_add_child(params->parent, self);
  z_proxy_leave(self);
  return self;
}


/**
 * z_proxy_free_method:
 * @self: proxy instance
 *
 * Called when the proxy object is finally to be freed (when the Python layer
 * releases its reference). Calls the proxy specific free function and
 * frees self
 **/
void
z_proxy_free_method(ZObject *s)
{
  ZProxy *self = Z_CAST(s, ZProxy);
  z_enter();
  
  /* NOTE: z_proxy_fastpath_destroy is called here as the fastpath
   * information might not have been registered when z_proxy_destroy is
   * called. (fastpath info is set right after startup and the proxy might
   * exit while that code is running)
   */
  
  z_proxy_fastpath_destroy(&self->fastpath);
  z_object_free_method(s);
  z_leave();
}

static ZProxyFuncs z_proxy_funcs =
{
  {
    Z_FUNCS_COUNT(ZProxy),
    z_proxy_free_method,
  },
  z_proxy_config_method,
  z_proxy_startup_method,
  z_proxy_main_method,
  z_proxy_shutdown_method,
  z_proxy_destroy_method,
  z_proxy_run_method
};

ZClass ZProxy__class = 
{
  Z_CLASS_HEADER,
  &ZObject__class,
  "ZProxy",
  sizeof(ZProxy),
  &z_proxy_funcs.super,
};

/* stacked proxy */

/** 
 * z_stacked_proxy_new:
 * @client_stream: client side stream
 * @server_stream: server side stream
 * @control_stream: control stream
 * @proxy: ZProxy instance which initiated stacking
 * @child_proxy: ZProxy instance of the 'child' proxy
 * 
 * This function creates a new ZStackedProxy instance encapsulating
 * information about a stacked proxy instance. This information can be freed
 * by calling z_stacked_proxy_destroy().  It consumes the stream references
 * passed to it (client, server and control) but does not consume the proxy
 * references (@proxy and @child_proxy)
 **/
ZStackedProxy *
z_stacked_proxy_new(ZStream *client_stream, ZStream *server_stream, ZStream *control_stream, ZProxy *proxy, ZProxy *child_proxy)
{
  ZStackedProxy *self = g_new0(ZStackedProxy, 1);
  
  z_proxy_enter(proxy);
  
  if (client_stream)
    {
      z_stream_set_nonblock(client_stream, TRUE);

      g_snprintf(client_stream->name, sizeof(client_stream->name), "%s/client_downstream", proxy->session_id);
      self->downstreams[EP_CLIENT] = client_stream;
    }
  
  if (server_stream)
    {
      z_stream_set_nonblock(server_stream, TRUE);
  
      g_snprintf(server_stream->name, sizeof(server_stream->name), "%s/server_downstream", proxy->session_id);
      self->downstreams[EP_SERVER] = server_stream;
    }
  
  if (control_stream)
    {
      g_snprintf(control_stream->name, sizeof(control_stream->name), "%s/control", proxy->session_id);
      self->control_stream = control_stream;
    }

  self->proxy = z_proxy_ref(proxy);
  if (child_proxy)
    self->child_proxy = z_proxy_ref(child_proxy);
  
  z_proxy_leave(proxy);
  return self;
}

/**
 * z_stacked_proxy_destroy:
 * @self: ZStackedProxy instance
 *
 * This function frees all references associated with a stacked proxy.
 **/
void
z_stacked_proxy_destroy(ZStackedProxy *self)
{
  gint i;

  z_enter();
  for (i = 0; i < EP_MAX; i++)
    {
      if (self->downstreams[i])
        {
          z_stream_shutdown(self->downstreams[i], SHUT_RDWR, NULL);
          z_stream_close(self->downstreams[i], NULL);
          z_stream_unref(self->downstreams[i]);
        }
    }

  if (self->control_stream)
    {
      z_stream_detach_source(self->control_stream);
      z_stream_shutdown(self->control_stream, SHUT_RDWR, NULL);
      z_stream_close(self->control_stream, NULL);
      z_stream_unref(self->control_stream);
    }
  
  if (self->child_proxy)
    {
      z_proxy_del_child(self->proxy, self->child_proxy);
      z_proxy_unref(self->child_proxy);
      self->child_proxy = NULL;
    }
  if (self->proxy)
    {
      z_proxy_unref(self->proxy);
    }
  g_free(self);
  z_leave();
}

/* ZProxyIface */

/**
 * z_proxy_iface:
 * @class: derived class description
 * @proxy: proxy instance to be associated with this interface
 *
 * Constructor for ZProxyIface objects and derivates. A ZProxyIface
 * class encapsulates a function interface which permits inter-proxy
 * communication. 
 **/
ZProxyIface *
z_proxy_iface_new(ZClass *class, ZProxy *proxy)
{
  ZProxyIface *self;
  
  self = Z_NEW_COMPAT(class, ZProxyIface);
  self->owner = z_proxy_ref(proxy);
  return self;
}

/**
 * z_proxy_iface_free_method:
 * @s: ZProxyIface instance passed as a ZObject pointer
 *
 * Destructor for ZProxyIface objects, frees associated references.
 **/
void
z_proxy_iface_free_method(ZObject *s)
{
  ZProxyIface *self = Z_CAST(s, ZProxyIface);
  
  z_proxy_unref(self->owner);
  self->owner = NULL;
  z_object_free_method(s);
}

ZObjectFuncs z_proxy_iface_funcs =
{
  Z_FUNCS_COUNT(ZObject),
  z_proxy_iface_free_method,
};

ZClass ZProxyIface__class =
{
  Z_CLASS_HEADER,
  Z_CLASS(ZObject),
  "ZProxyIface",
  sizeof(ZProxyIface),
  &z_proxy_iface_funcs
};

/* ZProxyBasicIface */

/**
 * z_proxy_basic_iface_new:
 * @class: class description
 * @proxy: associated proxy
 *
 * Constructor for ZProxyBasicIface class, derived from ZProxyIface.
 **/
ZProxyBasicIface *
z_proxy_basic_iface_new(ZClass *class, ZProxy *proxy)
{
  ZProxyBasicIface *self;
  
  self = (ZProxyBasicIface *) z_proxy_iface_new(class, proxy);
  return self;
}

ZProxyBasicIfaceFuncs z_proxy_basic_iface_funcs =
{
  {
    Z_FUNCS_COUNT(ZObject),
    NULL
  }
};

ZClass ZProxyBasicIface__class =
{
  Z_CLASS_HEADER,
  Z_CLASS(ZProxyIface),
  "ZProxyBasicIface",
  sizeof(ZProxyBasicIface),
  &z_proxy_basic_iface_funcs.super
};


ZProxyResultIfaceFuncs z_proxy_result_iface_funcs =
{
  {
    Z_FUNCS_COUNT(ZObject),
    NULL
  },
  NULL,
  NULL
};

ZClass ZProxyResultIface__class =
{
  Z_CLASS_HEADER,
  Z_CLASS(ZProxyIface),
  "ZProxyResultIface",
  sizeof(ZProxyResultIface),
  &z_proxy_result_iface_funcs.super
};
