/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: streamblob.c,v 1.00 2004/11/29 13:58:32 fules Exp $
 *
 * Author  : fules
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/streamblob.h>

#include <zorp/log.h>
#include <zorp/error.h>

#include <string.h>
#include <sys/types.h>
#include <assert.h>

typedef struct _ZStreamBlob
{
  ZStream       super;

  size_t        pos;
  ZBlob         *blob;
  GIOCondition  poll_cond;
} ZStreamBlob;

extern ZClass ZStreamBlob__class;

static gboolean
z_stream_blob_watch_prepare(ZStream *s, GSource *src G_GNUC_UNUSED, gint *timeout)
{
  ZStreamBlob   *self = Z_CAST(s, ZStreamBlob);
  gboolean      res;

  z_enter();
  if (timeout)
    {
      *timeout = -1;
    }
  res = FALSE;

  self->poll_cond = 0;
  
  if (self->super.want_read)
    {
      self->poll_cond |= G_IO_IN;
      res = TRUE;
    }

  if (self->super.want_write)
    {
      self->poll_cond |= G_IO_OUT;
      res = TRUE;
    }

  z_leave();
  return res;
}

static gboolean
z_stream_blob_watch_check(ZStream *s, GSource *src)
{
  gboolean res;

  /* now (should be reconsidered!) it's the same az watch_prepare */
  z_enter();
  res = z_stream_blob_watch_prepare(s, src, NULL);
  z_leave();
  return res;
}

static gboolean
z_stream_blob_watch_dispatch(ZStream *s, GSource *src)
{
  ZStreamSource *self = (ZStreamSource *) src;
  ZStreamBlob *mystream = Z_CAST(s, ZStreamBlob);
  gboolean rc = TRUE;

  z_enter();

  if (mystream->super.want_read && (mystream->poll_cond & G_IO_IN))
    {
      if (mystream->super.read_cb)
        {
          rc = (*mystream->super.read_cb)(self->stream, mystream->poll_cond, mystream->super.user_data_read);
        }
      else
        {
          /*LOG
            This message indicates an internal error, read event occurred, but no read
            callback is set. Please report this event to the Balabit QA Team (devel@balabit.com).
            */
          z_log(mystream->super.name, CORE_ERROR, 3, "Internal error, no read callback is set;");
        }
    }

  if (mystream->super.want_write && (mystream->poll_cond & G_IO_OUT) && rc)
    {
      if (mystream->super.write_cb)
        {
          rc &= (*mystream->super.write_cb)(self->stream, mystream->poll_cond, mystream->super.user_data_write);
        }
      else
        {
          /*LOG
            This message indicates an internal error, write event occurred, but no write
            callback is set. Please report this event to the Balabit QA Team (devel@balabit.com).
            */
          z_log(mystream->super.name, CORE_ERROR, 3, "Internal error, no write callback is set;");
        }
    }

  z_leave();
  return rc;
}

static GIOStatus
z_stream_blob_read_method(ZStream *stream,
                        gchar *buf,
                        gsize count,
                        gsize *bytes_read,
                        GError **error)
{
  ZStreamBlob *self = Z_CAST(stream, ZStreamBlob);

  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  z_log(self->super.name, CORE_DUMP, 7, "Reading channel; blob='%p', count='%zd', pos='%zd', size='%zd'", self->blob, count,
        self->pos, self->blob->size);
  if (self->pos >= self->blob->size)
    {
      *bytes_read = 0;
      z_leave();
      return G_IO_STATUS_EOF;
    }
  
  *bytes_read = z_blob_get_copy(self->blob, self->pos, buf, count, self->super.timeout);
  if (*bytes_read == 0)
    {
      z_log(self->super.name, CORE_ERROR, 1, "Channel read timed out; blob='%p'", self->blob);
      g_set_error (error, G_IO_CHANNEL_ERROR,
                   G_IO_CHANNEL_ERROR_FAILED,
                   "Channel read timed out");
      z_leave();
      return G_IO_STATUS_ERROR;
    }

  self->pos += *bytes_read;
  self->super.bytes_recvd += *bytes_read;
  z_log(self->super.name, CORE_DUMP, 7, "Reading channel; blob='%p', count='%zd'", self->blob, *bytes_read);
  if (z_log_enabled(CORE_DUMP, 9))
      z_data_dump(self->super.name, buf, *bytes_read);
  
  z_leave();
  return G_IO_STATUS_NORMAL;
}

static GIOStatus
z_stream_blob_write_method(ZStream *stream,
                         const gchar *buf,
                         gsize count,
                         gsize *bytes_written,
                         GError **error)
{
  ZStreamBlob *self = Z_CAST(stream, ZStreamBlob);

  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  z_log(self->super.name, CORE_DUMP, 7, "Writing channel; blob='%p', count='%zd'", self->blob, count);
  
  *bytes_written = z_blob_add_copy(self->blob, self->pos, buf, count, self->super.timeout);
  if (*bytes_written == 0)
    {
      z_log(self->super.name, CORE_ERROR, 1, "Channel write timed out; blob='%p'", self->blob);
      g_set_error (error, G_IO_CHANNEL_ERROR,
                   G_IO_CHANNEL_ERROR_FAILED,
                   "Channel write timed out");
      z_leave();
      return G_IO_STATUS_ERROR;
    }
  
  self->pos += *bytes_written;
  self->super.bytes_sent += *bytes_written;
  if (z_log_enabled(CORE_DEBUG, 9))
      z_data_dump(self->super.name, buf, *bytes_written);

  z_leave();
  return G_IO_STATUS_NORMAL;
}


static GIOStatus
z_stream_blob_close_method(ZStream *stream, GError **error G_GNUC_UNUSED)
{
  ZStreamBlob *self = Z_CAST(stream, ZStreamBlob);

  z_enter();
  /*LOG
    This message indicates that the given fd is to be closed.
    */
  z_log(self->super.name, CORE_DEBUG, 6, "Closing channel; blob='%p'", self->blob);
  z_leave();
  return G_IO_STATUS_NORMAL;
}

static gboolean
z_stream_blob_ctrl_method(ZStream *s, guint function, gpointer value, guint vlen)
{
  ZStreamBlob *self = Z_CAST(s, ZStreamBlob);

  z_enter();

  switch (ZST_CTRL_MSG(function))
    {
      case ZST_CTRL_SET_NONBLOCK:
        if (vlen == sizeof(gboolean))
          {
            self->super.timeout = *((gboolean *)value) ? 0 : -1;
            z_leave();
            return TRUE;
          }
        else
            /*LOG
              This message indicates that an internal error occurred, during setting NONBLOCK mode
              on a stream, because the size of the parameter is wrong. Please report this event to
              the Balabit QA Team (devel@balabit.com).
              */
            z_log(NULL, CORE_ERROR, 4, "Internal error, bad parameter is given for setting NONBLOCK mode; size='%d'", vlen);
      break;
      
      case ZST_CTRL_GET_NONBLOCK:
        if (vlen == sizeof(gboolean))
          {
            *((gboolean *) value) = (self->super.timeout ==0);
            z_leave();
            return TRUE;
          }
        else
            /*LOG
              This message indicates that an internal error occurred, during getting NONBLOCK mode status
              on a stream, because the size of the parameter is wrong. Please report this event to
              the Balabit QA Team (devel@balabit.com).
              */
            z_log(NULL, CORE_ERROR, 4, "Internal error, bad parameter is given for getting the NONBLOCK mode; size='%d'", vlen);
      break;
      
      default:
        if (z_stream_ctrl_method(s, function, value, vlen))
          {
            z_leave();
            return TRUE;
          }
        /*LOG
          This message indicates that an internal error occurred, because an invalid control was called.
          Please report this event to the Balabit QA Team (devel@balabit.com).
          */
        z_log(NULL, CORE_ERROR, 4, "Internal error, unknown stream ctrl; ctrl='%d'", ZST_CTRL_MSG(function));
      break;
    }
  z_leave();
  return FALSE;
}



static void
z_stream_blob_attach_source_method(ZStream *stream, GMainContext *context)
{
  /*ZStreamBlob *self = Z_CAST(stream, ZStreamBlob); */
  
  z_enter();
  if (!stream->source)
    {
      stream->source = z_stream_source_new(stream);
      g_source_attach(stream->source, context);
    }

  z_leave();
  return;
}

static void
z_stream_blob_detach_source_method(ZStream *stream)
{
  GSource *source;

  z_enter();
  if (stream->source)
    {
      source = stream->source;
      stream->source = NULL;
      g_source_destroy(source);
      g_source_unref(source);
    }

  z_leave();
}

/* destructor */
static void
z_stream_blob_free_method(ZObject *s)
{
  ZStreamBlob *self = Z_CAST(s, ZStreamBlob);
  time_t time_close;

  z_enter();
  time_close = time(NULL);

  /*LOG
    This message contains accounting information on the given channel. It
    reports the number of seconds the fd was open and the number of
    bytes sent/received on this channel.
    */
  z_log(self->super.name, CORE_ACCOUNTING, 4,
        "accounting info; type='stream', duration='%d', sent='%" G_GUINT64_FORMAT "', received='%" G_GUINT64_FORMAT "'", 
        (int) difftime(time_close, self->super.time_open),
        self->super.bytes_sent,
        self->super.bytes_recvd);

  z_blob_unref(self->blob);
  z_stream_free_method(s);
  z_leave();
}

static ZStreamFuncs 
z_stream_blob_funcs =
{
    {
      Z_FUNCS_COUNT(ZStream),
      z_stream_blob_free_method   /* ok */
    },
    z_stream_blob_read_method,   /* ok */
    z_stream_blob_write_method,   /* ok */
    NULL,
    NULL,
    NULL,
    z_stream_blob_close_method,   /* ok */
    z_stream_blob_ctrl_method,   /* ok */

    z_stream_blob_attach_source_method, /* TODO */
    z_stream_blob_detach_source_method, /* ok */
    z_stream_blob_watch_prepare,
    z_stream_blob_watch_check,
    z_stream_blob_watch_dispatch,
    NULL,

    NULL,
    NULL,
    NULL
};

ZClass ZStreamBlob__class =
{
  Z_CLASS_HEADER,
  &ZStream__class,
  "ZStreamBlob",
  sizeof(ZStreamBlob),
  &z_stream_blob_funcs.super
};


/**
 * z_stream_blob_new:
 * @blob: blob to create the stream around
 * @name: name to identify the stream in the logs
 *
 * Allocate and initialize a ZStreamBlob instance with the given blob and name.
 *
 * Returns:
 * The new stream instance
 */
ZStream *
z_stream_blob_new(ZBlob *blob, gchar *name)
{
  ZStreamBlob *self;

  z_enter();
  self = Z_CAST(z_stream_new(Z_CLASS(ZStreamBlob), name, NULL, Z_STREAM_FLAG_READ|Z_STREAM_FLAG_WRITE), ZStreamBlob);
  self->blob = z_blob_ref(blob);
  self->pos = 0;
  self->poll_cond = 0;
  z_leave();
  return &self->super;
}
