/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: code.h,v 1.2 2004/04/15 11:13:36 bazsi Exp $
 *
 ***************************************************************************/
       
#ifndef ZORP_CODE_H_INCLUDED
#define ZORP_CODE_H_INCLUDED
       
#define Z_CODE_NULL   0
#define Z_CODE_BASE64 1
#define Z_CODE_QUOTED 2

#include <zorp/zorp.h>

struct _ZCode;

/*
 * ZCode is an abstract class for implementing any arbitrary coding, it contains
 * a start/code/finish triplet for both encoding and decoding.
 * The 'start' method initialises the coder, the 'code' method transcribes its
 * from/fromlen input buffer into the to/tolen output buffer, the 'finish'
 * method finally extracts the results to its to/tolen arguments.
 *
 * This scheme applies to both transcribing codings that convert data (like
 * compressions) by their 'code' method and digesting codings (like checksums)
 * that produce the result by their 'finish' method after processing the input.
 */

typedef struct _ZCode
{
  guint type;
  gboolean (*decode_start)(struct _ZCode *code);
  gboolean (*decode)(struct _ZCode *code, const gchar *from, guint fromlen, gchar *to, guint *tolen);
  gboolean (*decode_finish)(struct _ZCode *code, gchar *to, guint *tolen);
  gboolean (*encode_start)(struct _ZCode *code);
  gboolean (*encode)(struct _ZCode *code, const gchar *from, guint fromlen, gchar *to, guint *tolen);
  gboolean (*encode_finish)(struct _ZCode *code, gchar *to, guint *tolen);
  gboolean (*free)(struct _ZCode *code);
} ZCode;


#define z_code_decode_start(c) (c)->decode_start(c)
#define z_code_decode(c, fr, fl, to, tl) (c)->decode(c, fr, fl, to, tl)
#define z_code_decode_finish(c, to, tl) (c)->decode_finish(c, to, tl)

#define z_code_encode_start(c) (c)->encode_start(c)
#define z_code_encode(c, fr, fl, to, tl) (c)->encode(c, fr, fl, to, tl)
#define z_code_encode_finish(c, to, tl) (c)->encode_finish(c, to, tl)

static inline void
z_code_free(ZCode *self)
{
  if (self->free)
    self->free(self);
  else
    g_free(self);
}

#endif
