/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: http.c,v 1.175 2004/07/26 11:45:57 bazsi Exp $
 * 
 * Author: Balazs Scheidler <bazsi@balabit.hu>
 * Auditor: 
 * Last audited version: 
 * Notes:
 *   Slightly based on the code by: Viktor Peter Kovacs <vps__@freemail.hu>
 *   
 ***************************************************************************/

#include "http.h"

#include <zorp/thread.h>
#include <zorp/registry.h>
#include <zorp/log.h>
#include <zorp/policy.h>
#include <zorp/authprovider.h>
#include <zorp/misc.h>
#include <zorp/policy.h>
#include <zorp/proxy/base64.h>

#include <ctype.h>
#include <sys/socket.h>
#include <unistd.h>
#include <fcntl.h>

/**
 * http_set_stacked_result:
 * @self: ZProxyResultIface instance
 * @decision: verdict sent by the child proxy
 * @description: additional information about @decision
 *
 * This function is the virtual set_result method in the ZProxyResultIface
 * interface callable by child proxies. It simply stores the verdict sent
 * by the child proxy in self->stack_decision. 
 * 
 * NOTE: this function runs in the thread of the child proxy and there is no
 * synchronization between this and the main HTTP thread except for the fact
 * that the child proxy should call this function before he sends anything
 * and HTTP will make use of this value when it first receives something,
 * thus there must be at least one context switch in between, and the HTTP
 * is possibly sleeping when this function is called. Nevertheless a memory
 * barrier function would need to be called to be absolutely certain that
 * these values are stored before the other thread refers to them. However
 * there are no memory barrier functions currently in GLib, they will be
 * added in GLib 2.4.
 **/
static void
http_set_stacked_result(ZProxyResultIface *self, guint decision, gchar *description)
{
  HttpProxy *proxy = Z_CAST(self->owner, HttpProxy);
  
  g_string_assign(proxy->stack_info, description);
  proxy->stack_decision = decision;
}

/**
 * http_set_content_length:
 * @self: ZProxyResultIface instance
 * @content_length: the length of the content to be transferred
 *
 * This function is the virtual set_content_length method in the
 * ZProxyResultIface interface callable by child proxies. It simply stores
 * the future size of the transferred data blob. This information will then
 * be used by the transfer code to decide whether to include a
 * content-length header. Calling this function is not mandatory by the
 * child proxy, if it is not called chunked mode transfer-encoding will be
 * used.
 */
static void
http_set_content_length(ZProxyResultIface *self, gsize content_length)
{
  HttpProxy *proxy = Z_CAST(self->owner, HttpProxy);
  
  proxy->content_length_hint = content_length;
  proxy->content_length_hint_set = TRUE;
}

ZProxyResultIfaceFuncs http_proxy_result_iface_funcs = 
{
  { 
    Z_FUNCS_COUNT(ZProxyResultIface),
    NULL,
  },
  http_set_stacked_result,
  http_set_content_length
};

ZClass HttpProxyResultIface__class =
{
  Z_CLASS_HEADER,
  Z_CLASS(ZProxyResultIface),
  "HttpProxyResultIface",
  sizeof(ZProxyResultIface),
  &http_proxy_result_iface_funcs.super,
};

/**
 * http_filter_hash_compare:
 * @a: first item
 * @b: second item
 * 
 * This function is the hash compare function for request header hashes.
 **/
static gint
http_filter_hash_compare(gconstpointer a, gconstpointer b)
{
  z_enter();
  if (strcasecmp((char *) a, (char *) b) == 0)
    {
      z_leave();
      return 1;
    }
  z_leave();
  return 0;
}

/**
 * http_filter_hash_bucket:
 * @a: item to calculate hash value for
 * 
 * This function is the hash calculation function for request header hashes.
 **/
static gint
http_filter_hash_bucket(gconstpointer a)
{
  int sum = 0;
  char *s = (char *) a;
  
  z_enter();
  while (*s != 0)
    {
      sum += toupper(*s);     
      s++;
    }

  z_leave();
  return (sum % 16);          
}

/**
 * http_config_set_defaults:
 * @self: HttpProxy instance
 *
 * This function initializes various attributes exported to the Python layer
 * for possible modification.
 **/
static void
http_config_set_defaults(HttpProxy *self)
{
  z_proxy_enter(self);
  
  self->connection_mode = HTTP_CONNECTION_CLOSE;
  self->server_connection_mode = HTTP_CONNECTION_CLOSE;
  self->force_reconnect = FALSE;
  self->transparent_mode = TRUE;
  self->permit_server_requests = TRUE;
  self->permit_proxy_requests = FALSE;
  self->permit_unicode_url = FALSE;
  self->permit_http09_responses = TRUE;

  self->rewrite_host_header = TRUE;
  self->require_host_header = TRUE;
  self->strict_header_checking = TRUE;
  self->permit_null_response = TRUE;
  self->max_line_length = 4096;
  self->max_url_length = 4096;
  self->max_header_lines = 50;
  self->max_hostname_length = 256;
  self->max_chunk_length = 262144;
  self->timeout_request = 10000;
  self->timeout = 300000;
  self->default_http_port = 80;
  self->default_ftp_port = 21;
  self->use_default_port_in_transparent_mode = TRUE;
  self->max_body_length = 0;
  self->buffer_size = 1500;

  http_init_headers(&self->headers[EP_CLIENT]);
  http_init_headers(&self->headers[EP_SERVER]);

  http_init_url(&self->request_url_parts);
  self->request_url = g_string_sized_new(128);
  
  self->current_header_name = g_string_sized_new(16);
  self->current_header_value = g_string_sized_new(32);

  self->parent_proxy = g_string_sized_new(0);
  self->parent_proxy_port = 3128;
  
  self->target_port_range = g_string_new("80,443");
  self->auth_header_value = g_string_sized_new(32);
  
  self->remote_server = g_string_sized_new(32);
  self->connected_server = g_string_sized_new(32);
  self->request_method = g_string_sized_new(16);
  self->response_msg = g_string_sized_new(32);

  self->request_method_policy =
    g_hash_table_new((GHashFunc) http_filter_hash_bucket,
                     (GCompareFunc) http_filter_hash_compare);
  self->request_header_policy =
    g_hash_table_new((GHashFunc) http_filter_hash_bucket,
                     (GCompareFunc) http_filter_hash_compare);

  self->response_policy = 
    z_dim_hash_table_new(1, 2, DIMHASH_WILDCARD, DIMHASH_CONSUME);
  self->response_header_policy =
    g_hash_table_new((GHashFunc) http_filter_hash_bucket,
                     (GCompareFunc) http_filter_hash_compare);


  self->error_info = g_string_sized_new(0);
  self->error_msg = g_string_sized_new(0);
  self->error_headers = g_string_sized_new(0);
  self->error_code = -1;
  self->error_status = 500;
  self->error_files_directory = g_string_new(ZORP_DATADIR "/http");

  self->error_silent = FALSE;
  
  self->stack_decision = 0;
  self->stack_info = g_string_sized_new(32);
  
  self->auth_inband_supported = 1;
  self->auth_realm = g_string_new("Zorp HTTP auth");
  self->old_auth_header = g_string_sized_new(0);
  z_proxy_leave(self);
}

/**
 * http_config_init:
 * @self: HttpProxy instance
 *
 * This function is called right after the config() method to initialize
 * settings the Python layer specified for us. 
 **/
static void
http_config_init(HttpProxy *self)
{
  z_proxy_enter(self);
  if (self->max_line_length > HTTP_MAX_LINE)
    self->max_line_length = HTTP_MAX_LINE;
  self->super.endpoints[EP_CLIENT]->timeout = self->timeout_request;
  self->poll = z_poll_new();
  z_proxy_leave(self);
}

/**
 * http_query_request_url:
 * @self: HttpProxy instance
 * @name: name of requested variable
 * @value: unused
 *
 * This function is registered as a Z_VAR_TYPE_CUSTOM get handler, e.g. it
 * is called whenever one of the request_url_* attributes are requested from
 * Python. Instead of presetting those attributes before calling into Python
 * we calculate their value dynamically.
 **/
static ZPolicyObj *
http_query_request_url(HttpProxy *self, gchar *name, gpointer value G_GNUC_UNUSED)
{
  ZPolicyObj *res = NULL;

  z_proxy_enter(self);

  if (strcmp(name, "request_url") == 0)
    {
      res = z_policy_var_build("s#", self->request_url->str, self->request_url->len);
    }
  if (strcmp(name, "request_url_proto") == 0 || strcmp(name, "request_url_scheme") == 0)
    {
      res = z_policy_var_build("s#", self->request_url_parts.scheme->str, self->request_url_parts.scheme->len);
    }
  else if (strcmp(name, "request_url_username") == 0)
    {
      res = z_policy_var_build("s#", self->request_url_parts.user->str, self->request_url_parts.user->len);
    }
  else if (strcmp(name, "request_url_passwd") == 0)
    {
      res = z_policy_var_build("s#", self->request_url_parts.passwd->str, self->request_url_parts.passwd->len);
    }
  else if (strcmp(name, "request_url_host") == 0)
    {
      res = z_policy_var_build("s#", self->request_url_parts.host->str, self->request_url_parts.host->len);
    }
  else if (strcmp(name, "request_url_port") == 0)
    {
      res = z_policy_var_build("i", self->request_url_parts.port ? self->request_url_parts.port : self->default_http_port);
    }
  else if (strcmp(name, "request_url_file") == 0)
    {
      res = z_policy_var_build("s#", self->request_url_parts.file->str, self->request_url_parts.file->len);
    }
  else if (strcmp(name, "request_url_query") == 0)
    {
      res = z_policy_var_build("s#", self->request_url_parts.query->str, self->request_url_parts.query->len);
    }
  else
    {
      PyErr_SetString(PyExc_AttributeError, "Unknown attribute");
    }
  z_proxy_leave(self);
  return res;
}

/**
 * http_set_request_url:
 * @self: HttpProxy instance
 * @name: name of requested variable
 * @value: unused
 * @new: new value for attribute
 *
 * This function is registered as a Z_VAR_TYPE_CUSTOM set handler, e.g. it
 * is called whenever the request_url attribute are changed from Python.
 * We need to reparse the URL in these cases.
 **/
static gint
http_set_request_url(HttpProxy *self, gchar *name G_GNUC_UNUSED, gpointer value G_GNUC_UNUSED, PyObject *new)
{
  z_proxy_enter(self);
  if (strcmp(name, "request_url") == 0)
    {
      gchar *str;
      const gchar *reason;
      
      if (!PyArg_Parse(new, "s", &str))
        {
          z_proxy_leave(self);
          return 0;
        }
      g_string_assign(self->request_url, str);
      if (!http_parse_url(&self->request_url_parts, self->permit_unicode_url, self->permit_invalid_hex_escape, self->request_url->str, &reason))
        {
          z_proxy_log(self, HTTP_ERROR, 2, "Policy tried to force an invalid URL; url='%s', reason='%s'", self->request_url->str, reason);
          z_policy_raise_exception_obj(z_policy_exc_value_error, "Invalid URL.");
          z_proxy_leave(self);
          return 0;
        }
      z_proxy_leave(self);
      return 1;
    }
  z_proxy_leave(self);
  return 0;
}

/**
 * http_query_mime_type:
 * @self: HttpProxy instance
 * @name: name of requested variable
 * @value: unused
 *
 * This function is registered as a Z_VAR_TYPE_CUSTOM get handler, e.g. it
 * is called whenever one of the request_mime_type or response_mime_type
 * attributes are requested from Python. Instead of presetting those
 * attributes before calling into Python we calculate their value
 * dynamically.
 **/
static ZPolicyObj *
http_query_mime_type(HttpProxy *self, gchar *name, gpointer value G_GNUC_UNUSED)
{
  ZPolicyObj *res = NULL;
  HttpHeader *hdr;
  gboolean success;

  z_proxy_enter(self);
  if (strcmp(name, "request_mime_type") == 0)
    {
      success = http_lookup_header(&self->headers[EP_CLIENT], "Content-Type", &hdr);
    }
  else if (strcmp(name, "response_mime_type") == 0)
    {
      success = http_lookup_header(&self->headers[EP_SERVER], "Content-Type", &hdr);
    }
  else
    {
      PyErr_SetString(PyExc_AttributeError, "Unknown attribute");
      z_proxy_leave(self);
      return NULL;
    }
  
  if (!success || !hdr)
    {
      res = PyString_FromString("");
    }
  else
    {
      gchar *start, *end;
      
      start = hdr->value->str;
      while (*start == ' ')
        start++;
      end = strchr(hdr->value->str, ';');
      if (end)
        {
          end--;
          while (end > start && *end == ' ')
            end--;
        }
      if (end)
        res = PyString_FromStringAndSize(hdr->value->str, (end - start + 1));
      else
        res = PyString_FromString(hdr->value->str);
    }
  z_proxy_leave(self);
  return res;
}

static ZPolicyObj *
http_policy_header_manip(HttpProxy *self, ZPolicyObj *args)
{
  gint action, side;
  gchar *header, *new_value = NULL;
  HttpHeader *p = NULL;
  ZPolicyObj *res = NULL;
  
  z_proxy_enter(self);
  if (!z_policy_var_parse_tuple(args, "iis|s", &action, &side, &header, &new_value))
    {
      goto error;
    }
    
  side &= 1;
    
  switch (action)
    {
    case 0:
      /* get */
      if (http_lookup_header(&self->headers[side], header, &p))
        {
          res = z_policy_var_build("s", p->value->str);
        }
      else
        {
          z_policy_var_ref(z_policy_none);
          res = z_policy_none;
        }
      break;
    case 1:
      /* set */
      if (!new_value)
        goto error_set_exc;
        
      if (!http_lookup_header(&self->headers[side], header, &p))
        {
          p = http_add_header(&self->headers[side], header, strlen(header), new_value, strlen(new_value));
        }
      g_string_assign(p->value, new_value);
      p->present = TRUE;
      z_policy_var_ref(z_policy_none);
      res = z_policy_none;
      break;
    default:
      goto error_set_exc;
    }
    
  z_proxy_leave(self);
  return res;
 error_set_exc:
  z_policy_raise_exception_obj(z_policy_exc_value_error, "Invalid arguments.");
 error:
  z_proxy_leave(self);
  return NULL;
  
}

/**
 * http_register_vars:
 * @self: HttpProxy instance
 *
 * This function is called upon startup to export Python attributes.
 **/
static void
http_register_vars(HttpProxy *self)
{
  z_proxy_enter(self);
  
  z_proxy_var_new(&self->super, "transparent_mode",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->transparent_mode);

  z_proxy_var_new(&self->super, "permit_server_requests",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_server_requests);

  z_proxy_var_new(&self->super, "permit_null_response",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_null_response);

  z_proxy_var_new(&self->super, "permit_proxy_requests",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_proxy_requests);

  z_proxy_var_new(&self->super, "permit_unicode_url",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_unicode_url);

  z_proxy_var_new(&self->super, "permit_invalid_hex_escape",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_invalid_hex_escape);

  z_proxy_var_new(&self->super, "permit_http09_responses",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_http09_responses);

  z_proxy_var_new(&self->super, "permit_both_connection_headers",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_both_connection_headers);

  z_proxy_var_new(&self->super, "permit_ftp_over_http",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->permit_ftp_over_http);

  /* close or keep-alive */
  z_proxy_var_new(&self->super, "connection_mode", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET,
                  &self->connection_mode);

  z_proxy_var_new(&self->super, "keep_persistent", 
		  Z_VAR_TYPE_INT | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG | Z_VAR_GET,
                  &self->keep_persistent);

  /* string containing parent proxy */
  z_proxy_var_new(&self->super, "parent_proxy",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  self->parent_proxy);

  /* parent proxy port */
  z_proxy_var_new(&self->super, "parent_proxy_port",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->parent_proxy_port);

  /* default port if portnumber is not specified in urls */
  z_proxy_var_new(&self->super, "default_port",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  "default_http_port");

  z_proxy_var_new(&self->super, "use_default_port_in_transparent_mode",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->use_default_port_in_transparent_mode);
		  
  z_proxy_var_new(&self->super, "default_http_port",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->default_http_port);

  z_proxy_var_new(&self->super, "default_ftp_port",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->default_ftp_port);

  /* rewrite host header when redirecting */
  z_proxy_var_new(&self->super, "rewrite_host_header",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->rewrite_host_header);

  /* require host header */
  z_proxy_var_new(&self->super, "require_host_header",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->require_host_header);

  /* enable strict header checking */
  z_proxy_var_new(&self->super, "strict_header_checking",
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
		  &self->strict_header_checking);

  /* integer */
  z_proxy_var_new(&self->super, "max_line_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_line_length);

  /* integer */
  z_proxy_var_new(&self->super, "max_url_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_url_length);

  /* integer */
  z_proxy_var_new(&self->super, "max_hostname_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_hostname_length);

  /* integer */
  z_proxy_var_new(&self->super, "max_header_lines",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_header_lines);

  /* integer */
  z_proxy_var_new(&self->super, "max_keepalive_requests",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_keepalive_requests);

  /* integer */
  z_proxy_var_new(&self->super, "max_body_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_body_length);

  /* integer */
  z_proxy_var_new(&self->super, "max_chunk_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->max_chunk_length);

  /* integer */
  z_proxy_var_new(&self->super, "request_count",
                  Z_VAR_TYPE_INT | Z_VAR_GET,
                  &self->request_count);

  /* timeout value in milliseconds */
  z_proxy_var_new(&self->super, "timeout", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->timeout);

  /* timeout value in milliseconds */
  z_proxy_var_new(&self->super, "buffer_size", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->buffer_size);

  /* timeout value in milliseconds */
  z_proxy_var_new(&self->super, "timeout_request", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG,
                  &self->timeout_request);

  /* hash indexed by request method */
  z_proxy_var_new(&self->super, "request",
                  Z_VAR_TYPE_HASH | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  self->request_method_policy);

  /* hash indexed by header name */
  z_proxy_var_new(&self->super, "request_header",
                  Z_VAR_TYPE_HASH | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  self->request_header_policy);

  /* hash indexed by response code */
  z_proxy_var_new(&self->super, "response",
                  Z_VAR_TYPE_DIMHASH | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  self->response_policy);

  /* hash indexed by header name */
  z_proxy_var_new(&self->super, "response_header",
                  Z_VAR_TYPE_HASH | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  self->response_header_policy);

  /* header manipulation */
  z_proxy_var_new(&self->super, "_AbstractHttpProxy__headerManip",
                  Z_VAR_TYPE_METHOD | Z_VAR_GET,
                  self, http_policy_header_manip);



  /* string containing current url proto */
  z_proxy_var_new(&self->super, "request_method",
		  Z_VAR_TYPE_STRING | Z_VAR_GET,
		  self->request_method);

  /* string containing current url */
  z_proxy_var_new(&self->super, "request_url",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET | Z_VAR_SET,
		  NULL, http_query_request_url, http_set_request_url, NULL);

  /* string containing current url proto */
  z_proxy_var_new(&self->super, "request_url_proto",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_request_url, NULL, NULL);

  /* string containing current url proto */
  z_proxy_var_new(&self->super, "request_url_scheme",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_request_url, NULL, NULL);

  /* string containing current url username */
  z_proxy_var_new(&self->super, "request_url_username",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_request_url, NULL, NULL);

  /* string containing current url passwd */
  z_proxy_var_new(&self->super, "request_url_passwd",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_request_url, NULL, NULL);

  /* string containing current url hostname */
  z_proxy_var_new(&self->super, "request_url_host",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_request_url, NULL, NULL);

  /* string containing current url port */
  z_proxy_var_new(&self->super, "request_url_port",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_request_url, NULL, NULL);

  /* string containing current url file */
  z_proxy_var_new(&self->super, "request_url_file",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_request_url, NULL, NULL);

  /* string containing current url query */
  z_proxy_var_new(&self->super, "request_url_query",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_request_url, NULL, NULL);

  /* string containing current url fragment */
  z_proxy_var_new(&self->super, "request_url_fragment",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_request_url, NULL, NULL);

  /* string containing request mime type */
  z_proxy_var_new(&self->super, "request_mime_type",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_mime_type, NULL, NULL);

  /* string containing response mime type */
  z_proxy_var_new(&self->super, "response_mime_type",
		  Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
		  NULL, http_query_mime_type, NULL, NULL);

  /* string containing current header name */
  z_proxy_var_new(&self->super, "current_header_name",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
		  self->current_header_name);

  /* string containing current header value */
  z_proxy_var_new(&self->super, "current_header_value",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
		  self->current_header_value);

  /* error response */
  z_proxy_var_new(&self->super, "error_status", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  &self->error_status);

  /* error silence */
  z_proxy_var_new(&self->super, "error_silent", 
		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG, 
                  &self->error_silent);

  /* string inserted into error messages */
  z_proxy_var_new(&self->super, "error_info",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
		  self->error_info);

  z_proxy_var_new(&self->super, "error_msg",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
		  self->error_msg);

  z_proxy_var_new(&self->super, "error_headers",
		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
		  self->error_headers);

  /* inband authentication support */
  z_proxy_var_new(&self->super, "auth_inband_supported",
  		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_GET_CONFIG,
  		  &self->auth_inband_supported);

  z_proxy_var_new(&self->super, "auth_forward",
  		  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG,
  		  &self->auth_forward);

  z_proxy_var_new(&self->super, "auth",
  		  Z_VAR_TYPE_OBJECT | Z_VAR_GET | Z_VAR_SET_CONFIG,
  		  &self->auth);

  z_proxy_var_new(&self->super, "auth_realm",
  		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET_CONFIG,
  		  self->auth_realm);

  z_proxy_var_new(&self->super, "target_port_range", 
                  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG, 
                  self->target_port_range);

  z_proxy_var_new(&self->super, "error_files_directory",
  		  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG,
  		  self->error_files_directory);

  /* compatibility with Zorp 0.8.x */
  z_proxy_var_new(&self->super, "transparent_server_requests",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  "permit_server_requests");  

  z_proxy_var_new(&self->super, "transparent_proxy_requests",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_SET | Z_VAR_SET_CONFIG | Z_VAR_GET_CONFIG,
                  "permit_proxy_requests");  

  z_proxy_var_new(&self->super, "request_timeout",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_SET | Z_VAR_GET_CONFIG | Z_VAR_SET_CONFIG,
		  "timeout_request");

  z_proxy_var_new(&self->super, "request_headers",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  "request_header");

  z_proxy_var_new(&self->super, "response_headers",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  "response_header");
  
  z_proxy_var_new(&self->super, "url_proto",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_proto");

  z_proxy_var_new(&self->super, "url_username",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_username");

  z_proxy_var_new(&self->super, "url_passwd",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_passwd");

  z_proxy_var_new(&self->super, "url_host",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_host");

  z_proxy_var_new(&self->super, "url_port",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_port");

  z_proxy_var_new(&self->super, "url_file",
		  Z_VAR_TYPE_ALIAS | Z_VAR_GET,
		  "request_url_file");

  z_proxy_var_new(&self->super, "error_response",
                  Z_VAR_TYPE_ALIAS | Z_VAR_GET | Z_VAR_SET,
                  "error_status");

  z_proxy_leave(self);
}

/**
 * http_error_message:
 * @self: HttpProxy instance
 * @response_code: HTTP status code to return
 * @message_code: HTTP message code (one of HTTP_MSG_*)
 * @infomsg: additional information added into error messages
 *
 * This function formats and returns an error page to the client when its
 * request cannot be fulfilled. It switches to non-persistent mode and
 * prepares the proxy for shutdown.
 **/
static gboolean
http_error_message(HttpProxy *self, gint response_code, guint message_code, GString *infomsg)
{
  gchar *messages[] = 
    { 
      NULL, 
      "clientsyntax.html", 
      "serversyntax.html", 
      "policysyntax.html", 
      "policyviolation.html", 
      "invalidurl.html", 
      "connecterror.html", 
      "ioerror.html", 
      "auth.html",
      "clienttimeout.html",
      "servertimeout.html",
      "badcontent.html",
      "ftperror.html",
      "redirect.html"
    };
  gchar response[256], filename[256];
  int fd;
 
  z_proxy_enter(self); 
  if (message_code >= (sizeof(messages) / sizeof(char *)))
    {
      /*LOG
        This message indicates that Zorp caught an invalid error code
	internally. Please report this event to the Zorp QA team (at
	devel@balabit.com).
       */
      z_proxy_log(self, HTTP_ERROR, 2, "Internal error, error code out of range; error_code='%d'", message_code);
      z_proxy_leave(self);
      return FALSE;
    }
    
  if (message_code == 0)
    {
      z_proxy_leave(self);
      return TRUE;
    }
    
  if (self->proto_version[EP_CLIENT] >= 0x0100)
    {
      g_snprintf(response, sizeof(response), "HTTP/1.0 %d %s\r\n", response_code, self->error_msg->len > 0 ? self->error_msg->str : "Error encountered");
      
      if (http_write(self, EP_CLIENT, response, strlen(response)) != G_IO_STATUS_NORMAL)
        {
	  /* error writing error message is already sent by http_write */
	  z_proxy_leave(self);
          return FALSE;
        }
        
      /* FIXME: we should not use self->headers[EP_SERVER] for this purpose */
      g_string_truncate(self->headers[EP_SERVER].flat, 0);
      if (!self->transparent_mode)
	g_string_append(self->headers[EP_SERVER].flat, "Proxy-Connection: close\r\n");
      else
	g_string_append(self->headers[EP_SERVER].flat, "Connection: close\r\n");
      
      g_string_append(self->headers[EP_SERVER].flat, self->error_headers->str);
      g_string_append(self->headers[EP_SERVER].flat, "Content-Type: text/html\r\n\r\n");
      if (http_write(self, EP_CLIENT, self->headers[EP_SERVER].flat->str, self->headers[EP_SERVER].flat->len) != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self);
          return FALSE;
        }
    }
  g_snprintf(filename, sizeof(filename), "%s/%s", self->error_files_directory->str, messages[message_code]);

  if (self->error_silent)
    {
      /*LOG
        This message reports that Zorp would send the given error page to
        the browser, if silent mode would not be enabled. It is likely that
        some protocol/configuration/proxy error occurred.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "An error occurred, would serve error file, but silent mode is enabled; filename='%s'", filename);
      z_proxy_leave(self);
      return FALSE;
    }

  /*LOG
    This message reports that Zorp is sending the given error page to the
    clients browser. It is likely that some protocol/configuration/proxy
    error occurred.
   */
  z_proxy_log(self, HTTP_DEBUG, 6, "An error occurred, serving error file; filename='%s'", filename);
  fd = open(filename, O_RDONLY);
  if (fd == -1)
    {
      /*LOG
        This message indicates that Zorp was unable to open the error file
	for the given reason. It is likely that the file does not exist, or
	has too restrictive permissions.
       */
      z_proxy_log(self, HTTP_ERROR, 3, "I/O error opening error file; filename='%s', error='%s'", filename, g_strerror(errno));
      http_write(self, EP_CLIENT, "Proxy error encountered, error file not found.\r\n", 49);
    }
  else
    {
      GString *new_contents = g_string_sized_new(4096);
      gchar contents[4096], *src;
      gint count;
      
      count = read(fd, contents, sizeof(contents) - 1);
      close(fd);
      if (count == -1)
	count = 0;
      contents[count] = 0;
      src = contents;
      while (*src)
        {
          if (*src == '@')
            {
              if (strncmp(src, "@INFO@", 6) == 0)
                {
                  gchar *p;
                  /* add info */
                  src += 5;
                  
                  for (p = infomsg->str; *p; p++)
                    {
                      if (*p == '<')
                        g_string_append(new_contents, "&lt;");
                      else if (*p == '>')
                        g_string_append(new_contents, "&gt;");
                      else if (*p == '"')
                        g_string_append(new_contents, "&quot;");
                      else if (*p == '&')
                        g_string_append(new_contents, "&amp;");
                      else
                        g_string_append_c(new_contents, *p);
                    }
                }
              else if (strncmp(src, "@VERSION@", 9) == 0)
                {
                  src += 8;
		  g_string_append(new_contents, VERSION);
                }
              else if (strncmp(src, "@DATE@", 6) == 0)
		{
		  time_t t;
		  gchar timebuf[64];
		  struct tm tm;

		  src += 5;
		  t = time(NULL);
		  localtime_r(&t, &tm);
		  strftime(timebuf, sizeof(timebuf), "%a %b %e %H:%M:%S %Z %Y", &tm);
		  g_string_append(new_contents, timebuf);
		}
	      else if (strncmp(src, "@HOST@", 6) == 0)
		{
		  gchar hostname[256];

		  src += 5;
		  if (gethostname(hostname, sizeof(hostname)) == 0)
		    g_string_append(new_contents, hostname);
		}
	      else
		g_string_append_c(new_contents, *src);
	    }
	  else
	    g_string_append_c(new_contents, *src);
            
	  src++;
	}
      http_write(self, EP_CLIENT, new_contents->str, new_contents->len);
      g_string_free(new_contents, TRUE);
    }
  z_proxy_leave(self);
  return TRUE;
}

/**
 * http_client_stream_init:
 * @self: HttpProxy instance
 * 
 * This function is called upon startup to initialize our client stream.
 **/
static gboolean
http_client_stream_init(HttpProxy *self)
{
  ZStream *tmpstream;
  z_proxy_enter(self);
  
  tmpstream = self->super.endpoints[EP_CLIENT];
  self->super.endpoints[EP_CLIENT] = z_stream_line_new(tmpstream, self->max_line_length, ZRL_EOL_CRLF | ZRL_PARTIAL_READ);
  z_stream_unref(tmpstream);
  /* timeout is initialized after the config event */
  z_proxy_leave(self);
  return TRUE;
}

/**
 * http_server_stream_init:
 * @self: HttpProxy instance
 * 
 * This function is called upon startup to initialize our server stream.
 **/
static gboolean
http_server_stream_init(HttpProxy *self)
{
  ZStream *tmpstream;
  z_proxy_enter(self);
  
  tmpstream = self->super.endpoints[EP_SERVER];
  self->super.endpoints[EP_SERVER] = z_stream_line_new(tmpstream, self->max_line_length, ZRL_EOL_CRLF | ZRL_PARTIAL_READ);
  z_stream_unref(tmpstream);
  self->super.endpoints[EP_SERVER]->timeout = self->timeout;
  z_proxy_leave(self);
  return TRUE;
}

GIOStatus
http_write(HttpProxy *self, guint side, gchar *buf, size_t buflen)
{
  GIOStatus res;
  gsize bytes_written;
  
  z_proxy_enter(self);
  if (!self->super.endpoints[side])
    {
      /*LOG
        This message reports that Zorp was about to write to an invalid
	stream. Please report this event to the Zorp QA team (at
	devel@balabit.com).
       */
      z_proxy_log(self, HTTP_ERROR, 1, "Error writing stream, stream is NULL; side='%s'", side == EP_CLIENT ? "client" : "server");
      z_proxy_leave(self);
      return G_IO_STATUS_ERROR;
    }
  res = z_stream_write(self->super.endpoints[side], buf, buflen, &bytes_written, NULL);
  if (res != G_IO_STATUS_NORMAL || buflen != bytes_written)
    {
      /* FIXME: move this to a separate function */
      /*LOG
       	This message reports that Zorp was unable to write to the given
	stream. It is likely that the peer closed the connection
	unexpectedly.
       */
      z_proxy_log(self, HTTP_ERROR, 1, "Error writing stream; side='%s', res='%x', error='%s'", side == EP_CLIENT ? "client" : "server", res, g_strerror(errno));

      self->error_code = HTTP_MSG_IO_ERROR;
      self->error_status = 502;
      g_string_sprintf(self->error_info, "Error writing to %s (%s)", side == EP_CLIENT ? "client" : "server", g_strerror(errno));
      z_proxy_leave(self);
      return G_IO_STATUS_ERROR;
    }

  z_proxy_leave(self);
  return res;
}   

static int
http_parse_connection_hdr_value(HttpProxy *self G_GNUC_UNUSED, HttpHeader *hdr)
{
  z_proxy_enter(self);
  if (strcasecmp(hdr->value->str, "keep-alive") == 0)
    return HTTP_CONNECTION_KEEPALIVE;
  else 
    return HTTP_CONNECTION_CLOSE;
  z_proxy_leave(self);
}

static void
http_assign_connection_hdr_value(HttpProxy *self, GString *value)
{
  z_proxy_enter(self);
  if (self->connection_mode == HTTP_CONNECTION_KEEPALIVE)
    g_string_assign(value, "keep-alive");
  else if (self->connection_mode == HTTP_CONNECTION_CLOSE)
    g_string_assign(value, "close");
  z_proxy_leave(self);
}

static inline gboolean
http_parent_proxy_enabled(HttpProxy *self)
{
  return !!self->parent_proxy->len;
}

static gboolean
http_process_base64(gchar *dst, guint dstlen, gchar *src, guint srclen)
{
  ZCode *auth_line;
  
  auth_line = z_code_base64_new();

  auth_line->decode_start((auth_line));

  if (auth_line->decode(auth_line, src, srclen, dst, &dstlen) == FALSE)
    {
      /* caller is logging */
      z_leave();
      return FALSE;
    }

  dst[dstlen] = 0;

  g_free(auth_line);

  return TRUE;
}

/* FIXME: optimize header processing a bit (no need to copy hdr into a
   buffer) */
static gboolean
http_process_auth_info(HttpProxy *self, HttpHeader *h)
{
  gchar userpass[128];
  gchar *p;
  gchar **up;

  z_proxy_enter(self);

  if (self->old_auth_header->len &&
      strcmp(h->value->str, self->old_auth_header->str) == 0)
    {
      z_proxy_leave(self);
      return TRUE;
    }
  
  if (strncmp(h->value->str, "Basic", 5) != 0)
    {
      /*LOG
        This message indicates that the client tried to use the given
	unsupported HTTP authentication. Currently only Basic HTTP
	authentication is supported by Zorp.
       */
      z_proxy_log(self, HTTP_ERROR, 3, "Only Basic authentication is supported; authentication='%s'", h->value->str);
      /* not basic auth */
      z_proxy_leave(self);
      return FALSE;
    }
    
  p = h->value->str + 5;
  while (*p == ' ') 
    {
      p++;
    }
  if (!http_process_base64(userpass, sizeof(userpass), p, strlen(p)))
    {
      /*LOG
       	This message indicates that the client sent a malformed
	username:password field, during the authentication phase.
       */
      z_proxy_log(self, HTTP_VIOLATION, 1, "Invalid base64 encoded username:password pair;");
      z_proxy_leave(self);
      return FALSE;
    }
  
  up = g_strsplit(userpass, ":", 2);
  if (up)
    {
      gboolean res;
      
      z_policy_lock(self->super.thread);
      res = z_auth_provider_check_passwd(self->auth, self->super.session_id, up[0], up[1], &self->super);
      z_policy_unlock(self->super.thread);
      if (res)
        res = z_proxy_user_authenticated(&self->super, up[0]);
      g_strfreev(up);
      z_proxy_leave(self);
      return res;
    }
  /*LOG
    This message indicates that the username:password field received
    during authentication was malformed.
   */
  z_proxy_log(self, HTTP_VIOLATION, 2, "No colon is found in the decoded username:password pair;");
  z_proxy_leave(self);
  return FALSE;
}

static gboolean
http_rewrite_host_header(HttpProxy *self, gchar *host, gint host_len, guint port)
{
  HttpHeader *h;
  
  z_proxy_enter(self);  
  
  if (self->rewrite_host_header && http_lookup_header(&self->headers[EP_CLIENT], "Host", &h))
    {
      /* NOTE: the constant 80 below is intentional, if
       * default_port is changed we still have to send
       * correct host header (containing port number)
       */
      if (port != 80 && port != 0)
	g_string_sprintf(h->value, "%.*s:%d", host_len, host, port);
      else
	g_string_sprintf(h->value, "%.*s", host_len, host);
    }
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_fetch_request(HttpProxy *self)
{
  gchar *line;
  gsize line_length;
  gint empty_lines = 0;
  guint res;

  z_proxy_enter(self); 

  /* FIXME: this can probably be removed as http_fetch_header does this */
  http_clear_headers(&self->headers[EP_CLIENT]);
  
  while (empty_lines < HTTP_MAX_EMPTY_REQUESTS) 
    {
      self->super.endpoints[EP_CLIENT]->timeout = self->timeout_request;
      res = z_stream_line_get(self->super.endpoints[EP_CLIENT], &line, &line_length, NULL);
      self->super.endpoints[EP_CLIENT]->timeout = self->timeout;
      if (res == G_IO_STATUS_EOF)
        {
          self->error_code = HTTP_MSG_OK;
          z_proxy_leave(self);
          return FALSE;
        }
      else if (res != G_IO_STATUS_NORMAL)
        {
          self->error_code = HTTP_MSG_OK; 
          if (errno == ETIME)
            {
              if (self->request_count == 0)
                {
	          self->error_code = HTTP_MSG_CLIENT_TIMEOUT;
                  self->error_status = 408;
                }
            }
          z_proxy_leave(self);
	  return FALSE;
	}
      if (line_length != 0)
	break;
      empty_lines++;
    }
  
  if (!http_split_request(self, line, line_length))
    {
      g_string_assign(self->error_info, "Invalid request line.");
      /*LOG
        This message indicates that the client sent an invalid HTTP request
        to the proxy.
       */
      z_proxy_log(self, HTTP_VIOLATION, 2, "Invalid HTTP request received; line='%.*s'", line_length, line);
      z_proxy_leave(self);
      return FALSE;
    }
  self->request_flags = http_proto_request_lookup(self->request_method->str);

  if (!http_parse_version(self, EP_CLIENT, self->request_version))
    {
      /* parse version already logged */
      z_proxy_leave(self);
      return FALSE;
    }

  if (!http_fetch_headers(self, EP_CLIENT))
    {
      /* fetch headers already logged */
      z_proxy_leave(self);
      return FALSE;
    }

  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_process_request(HttpProxy *self)
{
  HttpHeader *h;
  const gchar *reason;

  z_proxy_enter(self);  
  if (self->proto_version[EP_CLIENT] > 0x0100)
    self->connection_mode = HTTP_CONNECTION_KEEPALIVE;
  else    
    self->connection_mode = HTTP_CONNECTION_CLOSE;
    
  if (self->auth)
    {
      if (self->proto_version[EP_CLIENT] >= 0x0100)
        {
          /* authentication is required */
          g_string_truncate(self->auth_header_value, 0);
          if (self->transparent_mode)
            {
              h = NULL;
              if (!http_lookup_header(&self->headers[EP_CLIENT], "Authorization", &h) ||
                  !http_process_auth_info(self, h))
                {
                  self->error_code = HTTP_MSG_AUTH_REQUIRED;
                  self->error_status = 401;
                  g_string_sprintf(self->error_msg, "Authentication is required.");
                  g_string_sprintfa(self->error_headers, "WWW-Authenticate: Basic realm=\"%s\"\r\n", self->auth_realm->str);
                  z_proxy_leave(self);
                  return FALSE;
                }
            }
          else
            {
              if (!http_lookup_header(&self->headers[EP_CLIENT], "Proxy-Authorization", &h) ||
                  !http_process_auth_info(self, h))
                {
                  self->error_code = HTTP_MSG_AUTH_REQUIRED;
                  self->error_status = 407;
                  g_string_sprintf(self->error_msg, "Authentication is required.");
                  g_string_sprintfa(self->error_headers, "Proxy-Authenticate: Basic realm=\"%s\"\r\n", self->auth_realm->str);
                  z_proxy_leave(self);
                  return FALSE;
                }
            } 
          g_string_assign(self->auth_header_value, h->value->str);
        }
      else
        {
          z_proxy_log(self, HTTP_POLICY, 2, "Authentication required, but client requested HTTP/0.9 which does not support authentication;");
          z_proxy_leave(self);
          return FALSE;
        }
    }
    
  if (self->request_flags & HTTP_REQ_FLG_CONNECT)
    {
      if (self->proto_version[EP_CLIENT] >= 0x0100)
        {
          self->request_type = HTTP_REQTYPE_PROXY;
          z_proxy_leave(self);
          return TRUE;
        }
      else
        {
          self->error_code = HTTP_MSG_CLIENT_SYNTAX;
          g_string_sprintf(self->error_info, "CONNECT method requires HTTP/1.0 or later");
          /*LOG
	    This message indicates that the client sent a CONNECT method
	    request, but it is only supported for HTTP/1.0 or later.
           */
          z_proxy_log(self, HTTP_VIOLATION, 1, "CONNECT method without version specification;");
          z_proxy_leave(self);
          return FALSE;
        }
    }
    
  /* detect request type and set connection header in self */
  self->connection_hdr = NULL;
    
  if (self->proto_version[EP_CLIENT] < 0x0100)
    {
      /* no proxy protocol for version 0.9 */
      self->request_type = HTTP_REQTYPE_SERVER;
    }
  else 
    {
      HttpHeader *pconn_hdr = NULL, *conn_hdr = NULL;
      
      http_lookup_header(&self->headers[EP_CLIENT], "Proxy-Connection", &pconn_hdr);
      http_lookup_header(&self->headers[EP_CLIENT], "Connection", &conn_hdr);
      if (pconn_hdr && conn_hdr)
	{
	  if (!self->permit_both_connection_headers)
	    {
              /* both Proxy-Connection & Connection headers ... */
              self->error_code = HTTP_MSG_CLIENT_SYNTAX;
              g_string_sprintf(self->error_info, "Both Proxy-Connection and Connection headers exist.");
              /*LOG
                This message indicates that the client sent both Connection and
                Proxy-Connection headers, but these are mutually exclusive. It
                is likely sent by a buggy proxy/browser on the client side.
               */
              z_proxy_log(self, HTTP_VIOLATION, 1, "Both Proxy-Connection and Connection headers exist;");
              z_proxy_leave(self);
              return FALSE;
            }
          else
            {
              if (self->request_url->str[0] == '/')
                {
                  self->request_type = HTTP_REQTYPE_SERVER;
                }
              else
                {
                  self->request_type = HTTP_REQTYPE_PROXY;
                }
            }
	}
      else if (pconn_hdr)
        {
          self->request_type = HTTP_REQTYPE_PROXY;
        }
      else if (conn_hdr)
        {
          self->request_type = HTTP_REQTYPE_SERVER;
        }
      else if (self->request_url->str[0] != '/')
        {
          /* neither Connection nor Proxy-Connection header exists, and URI
             doesn't seem to be a simple filename */
          self->request_type = HTTP_REQTYPE_PROXY;
        }
      else
        {
          /* default */
          self->request_type = HTTP_REQTYPE_SERVER;
        }
      
      if (self->request_type == HTTP_REQTYPE_SERVER)
        {
          if (pconn_hdr)
            pconn_hdr->present = FALSE;
          self->connection_hdr = conn_hdr;
        }
      if (self->request_type == HTTP_REQTYPE_PROXY)
        {
          if (conn_hdr)
            conn_hdr->present = FALSE;
          self->connection_hdr = pconn_hdr;
        }
    }

  if (self->connection_hdr)
    {
      /* connection_mode overridden by connection header */
      self->connection_mode = http_parse_connection_hdr_value(self, self->connection_hdr);
    }


  if (self->transparent_mode &&
      ((self->request_type == HTTP_REQTYPE_PROXY && !self->permit_proxy_requests) ||
       (self->request_type == HTTP_REQTYPE_SERVER && !self->permit_server_requests)))
    {
      /* */
      self->error_code = HTTP_MSG_POLICY_VIOLATION;
      g_string_sprintf(self->error_info, "%s requests not permitted in transparent mode.", 
		       self->request_type == HTTP_REQTYPE_SERVER ? "server" : "proxy");
      /*LOG
       	This message indicates that the client sent the given type request,
	which is not permitted by the policy. Check the
	permit_proxy_requests and the permit_server_requests attributes.
       */
      z_proxy_log(self, HTTP_POLICY, 2, 
                  "This request type is not permitted in transparent mode; request_type='%s'", 
                  self->request_type == HTTP_REQTYPE_SERVER ? "server" : "proxy");
      z_proxy_leave(self);
      return FALSE;
    }


  if (http_lookup_header(&self->headers[EP_CLIENT], "Host", &h))
    {
      g_string_assign(self->remote_server, h->value->str);
    }
  else
    {
      g_string_truncate(self->remote_server, 0);
    }

  if (self->transparent_mode)
    {
      if (self->request_url->str[0] == '/')
	{
	  gchar buf[self->remote_server->len + 32];
	  
	  /* no protocol description */
	  if (!self->remote_server->len)
	    {
	      if (!self->require_host_header)
	        {
	          /* no host header */
                   /*LOG
		     This message indicates that no Host header was sent by
		     the client. As the content of the host header is used
		     to reconstruct the requested URL, the request_url
		     attribute will refer to a host named 'unknown'.
                    */
	          z_proxy_log(self, HTTP_VIOLATION, 4, "No host header in transparent request, 'unknown' is used instead;");
	          g_string_assign(self->remote_server, "unknown");
	        }
	      else
	        {
                  self->error_code = HTTP_MSG_CLIENT_SYNTAX;
                  if (self->proto_version[EP_CLIENT] < 0x0100)
                    {
                      g_string_sprintf(self->error_info, "'Host:' header is required, and HTTP/0.9 can't transfer headers.");
		      /*LOG
		       	This message indicates that an HTTP/0.9 request was
			sent by the client, and Host header is required by
			the policy, but HTTP/0.9 does not support headers.
			Check the require_host_header attribute.
		       */
                      z_proxy_log(self, HTTP_POLICY, 2, "'Host:' header is required, and HTTP/0.9 can't transfer headers;");
                    }
                  else
                    {
                      g_string_sprintf(self->error_info, "No 'Host:' header in request, and policy requires this.");
		      /*LOG
		       	This message indicates that no Host header was sent
			by the client, but it was required by the policy.
			Check the require_host_header attribute.
		       */
                      z_proxy_log(self, HTTP_POLICY, 2, "No 'Host:' header in request, and policy requires this;");
                    }

	          z_proxy_leave(self);
	          return FALSE;
	        }
	    }
	  g_snprintf(buf, sizeof(buf), "http://%s", self->remote_server->str);
	  g_string_prepend(self->request_url, buf);
	}
    }
  
  if (!http_parse_url(&self->request_url_parts, self->permit_unicode_url, self->permit_invalid_hex_escape, self->request_url->str, &reason))
    {
      /* invalid URL */
      self->error_code = HTTP_MSG_INVALID_URL;
      g_string_sprintf(self->error_info, "Badly formatted url: %s", self->request_url->str);
      /*LOG
        This message indicates that there was an error parsing an already
        canonicalized URL.  Please report this event to the Zorp QA team (at
        devel@balabit.com)
       */
      z_proxy_log(self, HTTP_ERROR, 1, "Error parsing URL; url='%s', reason='%s'", self->request_url->str, reason);
      z_proxy_leave(self);
      return FALSE;
    }
  if (!http_format_url(&self->request_url_parts, self->request_url, TRUE, self->permit_unicode_url, &reason))
    {
      self->error_code = HTTP_MSG_INVALID_URL;
      g_string_sprintf(self->error_info, "Error canonicalizing url (%s): %s", reason, self->request_url->str);
      /*LOG
        This message indicates that there was an error parsing an already
        canonicalized URL.  Please report this event to the Zorp QA team (at
        devel@balabit.com)
       */
      z_proxy_log(self, HTTP_ERROR, 1, "Error parsing URL; url='%s', reason='%s'", self->request_url->str, reason);
      z_proxy_leave(self);
      return FALSE;
    }
  self->remote_port = self->request_url_parts.port;
  z_proxy_leave(self);
  
  return TRUE;
}

static gboolean
http_process_filtered_request(HttpProxy *self)
{
  if ((self->request_flags & HTTP_REQ_FLG_CONNECT))
    {
      self->server_protocol = HTTP_PROTO_HTTP;
    }
  else if (http_parent_proxy_enabled(self) &&
           (strcasecmp(self->request_url_parts.scheme->str, "http") == 0 ||
            strcasecmp(self->request_url_parts.scheme->str, "ftp") == 0 ||
            strcasecmp(self->request_url_parts.scheme->str, "cache_object") == 0))
    {
      self->server_protocol = HTTP_PROTO_HTTP;
    }
  else if (!http_parent_proxy_enabled(self) && strcasecmp(self->request_url_parts.scheme->str, "ftp") == 0)
    {
      if (self->permit_ftp_over_http)
        {
          self->server_protocol = HTTP_PROTO_FTP;
        }
      else
        {
          /*LOG
            This message indicates that a client tried to use FTP over HTTP
            which is not allowed by default. Either set a parent proxy or
            enable the permit_ftp_over_http attribute.
           */
          z_proxy_log(self, HTTP_POLICY, 2, "Client attempted to use FTP over HTTP, which is currently disabled;");
          self->error_code = HTTP_MSG_CLIENT_SYNTAX;
          z_proxy_leave(self);
          return FALSE;
        }
    }
  else if (!http_parent_proxy_enabled(self) && strcasecmp(self->request_url_parts.scheme->str, "http") == 0)
    {
      self->server_protocol = HTTP_PROTO_HTTP;
    }
  else
    {
      /* unsupported protocol */
      self->error_code = HTTP_MSG_CLIENT_SYNTAX;
      g_string_sprintf(self->error_info, "Unsupported scheme: %s", self->request_url_parts.scheme->str);
      /*LOG
        This message indicates that the requested URL refers to an
        unsupported protocol scheme. Zorp currently knows about http and
        cache_object protocols, and can support the ftp protocol
        if a parent_proxy supporting ftp over http tunneling is present.
       */
      z_proxy_log(self, HTTP_ERROR, 3, "Unsupported scheme in URL; proto='%s'", self->request_url_parts.scheme->str);
      z_proxy_leave(self);
      return FALSE;
    }

  if (self->request_url_parts.host->len > self->max_hostname_length)
    {
      self->error_code = HTTP_MSG_CLIENT_SYNTAX;
      g_string_sprintf(self->error_info, "Too long hostname in URL: %s", self->request_url_parts.host->str);
      /*LOG
        This message indicates that the HTTP request was rejected because
        the hostname part in the URL was too long. You can increase the permitted
        limit by changing the max_hostname_length attribute. 
       */
      z_proxy_log(self, HTTP_POLICY, 2, "Too long hostname in URL; host='%s'", self->request_url_parts.host->str);
      z_proxy_leave(self);
      return FALSE;
    }

  if (self->remote_port == 0)
    {
      if (self->server_protocol == HTTP_PROTO_HTTP)
				{
					self->remote_port = self->default_http_port;
				}
      else
				{
					self->remote_port = self->default_ftp_port;
				}
    }
  
  if (!(self->request_flags & HTTP_REQ_FLG_CONNECT))
    http_rewrite_host_header(self, self->request_url_parts.host->str, self->request_url_parts.host->len, self->remote_port);
  if (http_parent_proxy_enabled(self))
    {
      self->remote_port = self->parent_proxy_port;
      g_string_assign(self->remote_server, self->parent_proxy->str);
    }
  else
    {
      g_string_assign(self->remote_server, self->request_url_parts.host->str);
      
    }

  /*LOG
    This is an accounting message that reports the requested method and URL.
   */
  z_proxy_log(self, HTTP_ACCOUNTING, 4, "Accounting; command='%s', url='%s'", self->request_method->str, self->request_url->str);

  z_proxy_leave(self);
  return TRUE;  
} 

/* FIXME: this code should be converted into an explicit referral to the
 * modified headers, e.g. instead of looping over all the headers, lookup
 * the necessary headers using http_lookup_header and modify the returned
 * values */
static guint 
http_request_filter_headers(HttpProxy *self, GString *name, GString *value)
{
  gint res = HTTP_HDR_ACCEPT;
  z_proxy_enter(self);

  switch (self->request_type)
    {
    case HTTP_REQTYPE_SERVER:
      /* if we have a parent proxy, Connection -> Proxy-Connection
       * otherwise leave it as is 
       */
      if (strcasecmp(name->str, "Connection") == 0)
        {
          if (http_parent_proxy_enabled(self))
	    g_string_assign(name, "Proxy-Connection");
	  http_assign_connection_hdr_value(self, value);
	}
      else if (strcasecmp(name->str, "Authorization") == 0)
        {
          if (self->auth)
            {
              /* if inband authentication was performed, drop the
               * authentication header unless forwarding was explicitly
               * requested */
              if (self->auth_forward)
                {
                  g_string_assign(value, self->auth_header_value->str);
                  
                  /* if the upstream is a proxy, forward it as a
                   * Proxy-Authorization header */
                   
                  if (http_parent_proxy_enabled(self))
                    g_string_assign(name, "Proxy-Authorization");
                }
              else
                res = HTTP_HDR_DROP;
            }
        }
      break;
    case HTTP_REQTYPE_PROXY:
      /* if we have a parent proxy leave it as is
       * otherwise Proxy-Connection -> Connection 
       */
      if (strcasecmp(name->str, "Proxy-Connection") == 0)
        {
          if (http_parent_proxy_enabled(self) == 0)
	    g_string_assign(name, "Connection");
	  http_assign_connection_hdr_value(self, value);
	}
      else if (strcasecmp(name->str, "Proxy-Authorization") == 0)
        {
          if (self->auth)
            {
              /* if inband authentication was performed, drop the
               * authentication header unless forwarding was explicitly
               * requested */
              if (self->auth_forward)
                {
                  g_string_assign(value, self->auth_header_value->str);
                  
                  /* if the upstream is not a proxy, forward it as a
                   * Authorization header */
                   
                  if (!http_parent_proxy_enabled(self))
                    g_string_assign(name, "Authorization");
                }
              else
                res = HTTP_HDR_DROP;
            }
        }
      break;
    }
  z_proxy_leave(self);
  return res;
}

static gboolean
http_filter_request(HttpProxy *self)
{
  ZPolicyObj *f;
  gint rc;

  z_proxy_enter(self);
  f = g_hash_table_lookup(self->request_method_policy, self->request_method->str);
  if (!f)
    f = g_hash_table_lookup(self->request_method_policy, "*");
  if (f)
    {
      ZPolicyObj *handler, *res;
      gchar *errmsg;
      guint filter_type;
      
      z_policy_lock(self->super.thread);  
      if (!z_policy_tuple_get_verdict(f, &filter_type))
        {
          /*LOG
	    This message indicates that the request hash contains an invalid
	    item for the given request method. Check your Zorp
	    configuration.
           */
	  z_proxy_log(self, HTTP_POLICY, 1, "Invalid item in request hash; method='%s'", self->request_method->str);
	  z_policy_unlock(self->super.thread);
	  z_proxy_leave(self);
	  return FALSE;
          
        }
      z_policy_unlock(self->super.thread);
      g_string_sprintf(self->error_info, "Method %s denied by policy", self->request_method->str);

      switch (filter_type)
        {
        case HTTP_REQ_POLICY:
	  z_policy_lock(self->super.thread);
          if (!z_policy_var_parse(f, "(iO)", &filter_type, &handler))
            {
              /*LOG
	       	This message indicates that the request hash contains an
		invalid POLICY tuple for the given request method.  It
		should contain a valid call-back function in the tuple.
              */
              z_proxy_log(self, HTTP_POLICY, 1, "Error parsing HTTP_REQ_POLICY tuple in request hash; method='%s'", self->request_method->str);
              z_policy_unlock(self->super.thread);
              z_proxy_leave(self);
              return FALSE;
            }
          res = z_policy_call_object(handler, 
                                     z_policy_var_build("(sss)", 
                                                        self->request_method->str, self->request_url->str, self->request_version), 
                                     self->super.session_id);
          if (!res || !z_policy_var_parse(res, "i", &rc))
            {
	      rc = HTTP_REQ_REJECT;
              g_string_assign(self->error_info, "Error in policy handler, or returned value not integer;");
            }
          z_policy_var_unref(res);
	  z_policy_unlock(self->super.thread);
          break;
	case HTTP_REQ_REJECT:
	  errmsg = NULL;
          z_policy_lock(self->super.thread);
	  if (!z_policy_var_parse_tuple(f, "i|s", &filter_type, &errmsg))
	    {
              /*LOG
	       	This message indicates that the request hash contains an
		invalid REJECT tuple for the given request method.  It
		should contain an error message, which is sent back to the
		client.
              */
	      z_proxy_log(self, HTTP_POLICY, 1, "Error parsing HTTP_REQ_REJECT in request hash; req='%s'", self->request_method->str);
              z_policy_unlock(self->super.thread);
	      z_proxy_leave(self);
              return FALSE;
	    }
	  z_policy_unlock(self->super.thread);
          if (errmsg)
	    g_string_assign(self->error_info, errmsg);
          /* fallthrough */
	case HTTP_REQ_ACCEPT:
	case HTTP_REQ_DENY:
	case HTTP_REQ_ABORT:
	  /* dropped command */
	  rc = filter_type;
	  break;
	default:
          /*LOG
	    This message indicates that the request hash contains an invalid
	    action for the given request method.  Check your Zorp
	    configuration.
          */
	  z_proxy_log(self, HTTP_POLICY, 1, "Unknown request hash item; req='%s'", self->request_method->str);
	  z_proxy_leave(self);
	  return FALSE;
        }
      switch (rc)
        {
        case HTTP_REQ_ACCEPT:
          g_string_truncate(self->error_info, 0);
          break;
        default:
          /*LOG
            This log message indicates that the specified HTTP request was not permitted by your policy.
           */
          z_proxy_log(self, HTTP_POLICY, 2, "Request not permitted by policy; req='%s'", self->request_method->str);          
	  self->error_code = HTTP_MSG_POLICY_VIOLATION;
	  z_proxy_leave(self);
          return FALSE;
        }
      if (self->proto_version[EP_CLIENT] >= 0x0100)
        {
          if (!http_filter_headers(self, EP_CLIENT, http_request_filter_headers))
    	    {
    	      z_proxy_leave(self);
    	      return FALSE;
    	    }
        }
      z_proxy_leave(self);
      return TRUE;
    }
  self->error_code = HTTP_MSG_POLICY_VIOLATION;
  g_string_sprintf(self->error_info, "Method %s denied by policy", self->request_method->str);
  /*LOG
    This log message indicates that the specified HTTP request was not permitted by your policy.
   */
  z_proxy_log(self, HTTP_POLICY, 2, "Request not permitted by policy; req='%s'", self->request_method->str);
  z_proxy_leave(self);
  return FALSE;
}

gboolean
http_connect_server(HttpProxy *self)
{
  z_proxy_enter(self);
  if (!self->super.endpoints[EP_SERVER] ||
      (!self->transparent_mode && 
        (strcasecmp(self->remote_server->str, self->connected_server->str) != 0 ||
         self->remote_port != self->connected_port)) ||
      self->force_reconnect)
    {
      gboolean success = FALSE;
      
      self->force_reconnect = FALSE;
      
      if (self->super.endpoints[EP_SERVER])
	{
	  z_stream_shutdown(self->super.endpoints[EP_SERVER], SHUT_RDWR, NULL);
	  z_stream_close(self->super.endpoints[EP_SERVER], NULL);
	  z_stream_unref(self->super.endpoints[EP_SERVER]);
	  self->super.endpoints[EP_SERVER] = NULL;
	}
      
      g_string_sprintf(self->error_info, "Error establishing connection to %s", self->remote_server->str);
      if (http_parent_proxy_enabled(self))
        {
          success = z_proxy_connect_server(&self->super, self->parent_proxy->str, self->parent_proxy_port);
        }
      else if (self->transparent_mode && self->use_default_port_in_transparent_mode)
        {
          success = z_proxy_connect_server(&self->super, self->remote_server->str, 
                                           self->server_protocol == HTTP_PROTO_HTTP ? self->default_http_port : self->default_ftp_port);
        }
      else if (z_port_enabled(self->target_port_range->str, self->remote_port))
        {
          success = z_proxy_connect_server(&self->super, self->remote_server->str, self->remote_port);
        }
      else
        {
          /*LOG
            This message indicates that the proxy did not allow
            addressing the specified port as the target_port_range
            attribute does not allow it.
            */
          z_proxy_log(self, HTTP_VIOLATION, 2, "Connecting to this port is prohibited by policy; port='%d'", self->remote_port);
          g_string_sprintf(self->error_info, "Connecting to port %d is prohibited by policy.", self->remote_port);
          success = FALSE;
        }

      if (!success)
	{
	  /* error connecting to server */
	  self->error_code = HTTP_MSG_CONNECT_ERROR;
	  self->error_status = 502;
	  /* connect_server already logged */
	  z_proxy_leave(self);
	  return FALSE;
	}

      g_string_assign(self->connected_server, self->remote_server->str);
      self->connected_port = self->remote_port;
      if (!http_server_stream_init(self))
        {
          /* should never happen */
          /*LOG
            This message indicates that initializing the server stream
            failed. Please report this event to the Zorp QA team (at
            devel@balabit.com).
           */
          z_proxy_log(self, HTTP_ERROR, 1, "Internal error initializing server stream;");
          z_proxy_leave(self);
          return FALSE;
        }
    }
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_format_request(HttpProxy *self, gboolean stacked, GString *request)
{
  gboolean res = TRUE;
  const gchar *reason;
  GString *url = g_string_sized_new(self->request_url->len);
  
  z_proxy_enter(self);

  if (self->proto_version[EP_CLIENT] >= 0x0100)
    {
      if (self->request_flags & HTTP_REQ_FLG_CONNECT)
        {
          g_assert(http_parent_proxy_enabled(self));
          
          http_flat_headers(&self->headers[EP_CLIENT]);
          g_string_sprintf(request, "%s %s %s\r\n%s\r\n",  
                           self->request_method->str, self->request_url->str, self->request_version,
                           self->headers[EP_CLIENT].flat->str);
        }
      else
        {
          if (!stacked)
            {
              http_flat_headers(&self->headers[EP_CLIENT]);
              
              if (!http_format_url(&self->request_url_parts, url, http_parent_proxy_enabled(self), self->permit_unicode_url, &reason))
                res = FALSE;
              g_string_sprintf(request, "%s %s %s\r\n%s\r\n", 
                      self->request_method->str, url->str, self->request_version, 
                      self->headers[EP_CLIENT].flat->str);
            }
          else
            {
              http_flat_headers_into(&self->headers[EP_CLIENT], request);
            }
        }
    }
  else
    {
      if (!http_format_url(&self->request_url_parts, url, FALSE, self->permit_unicode_url, &reason))
        res = FALSE;
      else
        g_string_sprintf(request, "%s %s\r\n", self->request_method->str, url->str);
    }
  if (!res)
    {
      z_proxy_log(self, HTTP_ERROR, 3, "Error reformatting requested URL; url='%s', reason='%s'", self->request_url->str, reason);
    }
  g_string_free(url, TRUE);
  z_proxy_leave(self);
  return res;
}

static gboolean
http_copy_request(HttpProxy *self)
{
  gint attempts = 0;
  
  z_proxy_enter(self);

  while (attempts < 4)
    {
      attempts++;
      if (!http_connect_server(self))
        {
          /* connect_server already logs */
          z_proxy_leave(self);
	  return FALSE;
	}

      self->reattempt_connection = FALSE;
      if (http_data_transfer(self, EP_CLIENT, self->super.endpoints[EP_CLIENT], EP_SERVER, self->super.endpoints[EP_SERVER], FALSE, FALSE, http_format_request))
        {
	  break;
	}
      else 
        {
          /* FIXME: this is a hack, a better solution would be to propagate
           * GError here */
          
          if (self->reattempt_connection)
            continue;
            
          /* http_data_transfer already logs */
          z_proxy_leave(self);
          return FALSE;
        }

    }

  z_proxy_leave(self);
  return TRUE;
}

static gboolean 
http_fetch_response(HttpProxy *self)
{
  gchar *line;
  gsize line_length;
  guint res;
  gchar status[4];
  gsize br;

  z_proxy_enter(self);
  /* FIXME: this can probably be removed as http_fetch_header does this */
  http_clear_headers(&self->headers[EP_SERVER]);

  self->response[0] = 0;
  self->response_code = -1;
  if (self->proto_version[EP_CLIENT] < 0x0100)
    {
      self->proto_version[EP_SERVER] = 0x0009;
      z_proxy_leave(self);
      return TRUE;
    }
  
  while (self->response_code == -1 || self->response_code == 100 || self->response_code == 102)
    {
      res = z_stream_read(self->super.endpoints[EP_SERVER], status, sizeof(status), &br, NULL);
      if (res != G_IO_STATUS_NORMAL)
        {
          /* the server closed our connection */
          self->error_code = HTTP_MSG_OK;
          z_proxy_leave(self);
          return FALSE;
        }
      if (!z_stream_line_unget(self->super.endpoints[EP_SERVER], status, 4))
        {
          /* error falling back to 0.9 */
	  /*LOG
	    This message indicates that Zorp was unable to enable HTTP/0.9
	    compatibility mode, due to the full buffer.  If you experience
	    this problem many times, please contact your Zorp support.
	   */
          z_proxy_log(self, HTTP_ERROR, 2, "Error in HTTP/0.9 compatibility code, line buffer full;");
          z_proxy_leave(self);
          return FALSE;
        }
      if (memcmp(status, "HTTP", 4) == 0)
        {
          res = z_stream_line_get(self->super.endpoints[EP_SERVER], &line, &line_length, NULL);
          if (res != G_IO_STATUS_NORMAL)
            {
              self->error_code = HTTP_MSG_OK;
              z_proxy_leave(self);
	      return FALSE;
	    }
	}
      else if (self->permit_http09_responses)
        {
	  self->proto_version[EP_SERVER] = 0x0009;
	  z_proxy_leave(self);
	  return TRUE;
        }
      else
        {
          /* HTTP/0.9 is not permitted by policy */
          g_string_assign(self->error_info, "Server falled back to HTTP/0.9 which is prohibited by policy.");
	  /*LOG
	    This message indicates that the server sent back HTTP/0.9
	    response, which is prohibited by the policy.  It is likely a
	    buggy or old server. Check the permit_http09_responses
	    attribute.
	   */ 
          z_proxy_log(self, HTTP_POLICY, 2, "Server falled back to HTTP/0.9 which is prohibited by policy;");
          z_proxy_leave(self);
          return FALSE;
        }
      if (!http_split_response(self, line, line_length))
        {
          /*LOG
            This message indicates the the HTTP status line returned by the
            server was invalid.
           */
          z_proxy_log(self, HTTP_VIOLATION, 1, "Invalid HTTP response heading; line='%.*s'", line_length, line);
          g_string_assign(self->error_info, "Invalid HTTP response heading");
          z_proxy_leave(self);
	  return FALSE;
	}
      
      self->response_flags = http_proto_response_lookup(self->response);
      if (!http_parse_version(self, EP_SERVER, self->response_version))
        {
          g_string_sprintf(self->error_info, "Invalid HTTP version in response (%.*s)", (gint) line_length, line);
          /*LOG
            This message indicates that the server sent the response with an
	    invalid HTTP version.  It is likely that the server is buggy.
          */
          z_proxy_log(self, HTTP_VIOLATION, 1, "Error parsing response version; line='%.*s'", (gint) line_length, line);
          z_proxy_leave(self);
	  return FALSE;
	}
      
      if (!http_fetch_headers(self, EP_SERVER))
        {
          g_string_assign(self->error_info, "Invalid headers received in response");
          z_proxy_leave(self);
	  return FALSE;
	}
    }

  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_process_response(HttpProxy *self)
{
  HttpHeader *pconn_hdr = NULL, *conn_hdr = NULL;

  z_proxy_enter(self);

  /* previously set by http_parse_version */
  switch (self->proto_version[EP_SERVER])
    {
    case 0x0009:
      g_strlcpy(self->response, "200", sizeof(self->response_code));
      self->response_code = 200;
      self->response_flags = 0;
      self->connection_mode = HTTP_CONNECTION_CLOSE;
      self->server_connection_mode = HTTP_CONNECTION_CLOSE;
      z_proxy_leave(self);
      return TRUE;
    case 0x0100:
      self->server_connection_mode = HTTP_CONNECTION_CLOSE;
      break;
    case 0x0101:
      self->server_connection_mode = HTTP_CONNECTION_KEEPALIVE;
      break;
    default:
      /*LOG
        This message indicates that the server sent an unsupported protocol
	version. It is likely that the server is buggy.
       */
      z_proxy_log(self, HTTP_VIOLATION, 1, "Unsupported protocol version; version='0x%04x'", self->proto_version[EP_SERVER]);
      z_proxy_leave(self);
      return FALSE;
    }
 
  /* process connection header */
 
  self->connection_hdr = NULL;

  http_lookup_header(&self->headers[EP_SERVER], "Proxy-Connection", &pconn_hdr);
  http_lookup_header(&self->headers[EP_SERVER], "Connection", &conn_hdr);
  if ((http_parent_proxy_enabled(self) && pconn_hdr) || conn_hdr)
    {
      /* override default */
      if (http_parent_proxy_enabled(self) && pconn_hdr)
        {
          self->connection_hdr = pconn_hdr;
          if (conn_hdr)
            conn_hdr->present = FALSE;
        }
      else
        {
          self->connection_hdr = conn_hdr;
          if (pconn_hdr)
            pconn_hdr->present = FALSE;
        }
        
      if (self->connection_mode == HTTP_CONNECTION_KEEPALIVE && 
          http_parse_connection_hdr_value(self, self->connection_hdr) == HTTP_CONNECTION_KEEPALIVE)
        {
          self->server_connection_mode = HTTP_CONNECTION_KEEPALIVE;
        }
      else
        {
          self->server_connection_mode = HTTP_CONNECTION_CLOSE;
        }
    }
  else
    {
      /* NOTE: there was no appropriate Connection header in the response, if it
       * is an 1.0 client the connection mode should be changed to
       * connection close regardless what the client specified in its
       * connection header.
       */
       
      gchar *conn_hdr_str = http_parent_proxy_enabled(self) ? "Proxy-Connection" : "Connection";
      
      /* we add an appriopriate connection header just as if we received one
       * (e.g. it might be rewritten later by
       * http_response_filter_connection_header), we default to not sending
       * this new header, it will be made visible when we want to ensure
       * that our connection mode must be enforced. */
      
      self->connection_hdr = http_add_header(&self->headers[EP_SERVER], conn_hdr_str, strlen(conn_hdr_str), "close", 5);
      self->connection_hdr->present = FALSE;
      
      if (self->proto_version[EP_CLIENT] == 0x0100)
        {
          if (self->connection_mode == HTTP_CONNECTION_KEEPALIVE)
            {
              /* it is somewhat uncertain what happens when an 1.0 client
               * requests keepalive and an 1.1 server responds without a
               * connection header, resolve this uncertainity by adding a 
               * connection header */

              self->connection_hdr->present = TRUE;
            }
          self->server_connection_mode = HTTP_CONNECTION_CLOSE;
        }
    }
  if (self->request_flags & HTTP_REQ_FLG_CONNECT && self->response_code == 200)
    self->server_connection_mode = HTTP_CONNECTION_CLOSE;
  
  if (!self->keep_persistent && self->server_connection_mode == HTTP_CONNECTION_CLOSE)
    self->connection_mode = HTTP_CONNECTION_CLOSE;
  
  if (self->response_flags & HTTP_RESP_FLG_STOP)
    {
      z_proxy_leave(self);
      return FALSE;  /* FIXME: not implemented */
    }

  z_proxy_leave(self);
  return TRUE;
}

/**
 * http_fetch_buffered_data:
 * @self: HttpProxy instance
 *
 * Read all buffered data from the client up to maximum ten 4096 chunks.
 * This is needed to avoid sending RSTs to connections where the client sent
 * some unprocessed bytes (like IE 5.0 in the case of POST requests).
 **/
static void
http_fetch_buffered_data(HttpProxy *self)
{
  gchar buf[4096];
  gsize br;
  gint count = 0;
  
  z_stream_set_nonblock(self->super.endpoints[EP_CLIENT], TRUE);
  while (count < 10 && z_stream_read(self->super.endpoints[EP_CLIENT], buf, sizeof(buf), &br, NULL) == G_IO_STATUS_NORMAL)
    {
      count++;
    }
  z_stream_set_nonblock(self->super.endpoints[EP_CLIENT], FALSE);
}

static ZPolicyObj *
http_response_policy_get(HttpProxy *self)
{
  gchar *response_keys[2];

  response_keys[0] = self->request_method->str;
  response_keys[1] = self->response;
  
  return z_dim_hash_table_search(self->response_policy, 2, response_keys);
}

static gboolean
http_filter_response(HttpProxy *self)
{
  z_proxy_enter(self);
  if (self->proto_version[EP_SERVER] >= 0x0100)
    {
      ZPolicyObj *f, *res;
      gint rc;
      
      f = http_response_policy_get(self);
      if (f)
        {
          ZPolicyObj *handler;
	  guint filter_type;
          gchar *errmsg;
      
          z_policy_lock(self->super.thread);  
          if (!z_policy_tuple_get_verdict(f, &filter_type))
            {
              /*LOG
	        This message indicates that the response hash contains an
	        invalid item for the given response. Check your Zorp
	        configuration.
               */
	      z_proxy_log(self, HTTP_POLICY, 1, "Invalid response hash item; request='%s', response='%d'", self->request_method->str, self->response_code);
	      z_policy_unlock(self->super.thread);
	      z_proxy_leave(self);
	      return FALSE;
          
            }
          z_policy_unlock(self->super.thread);
          
          g_string_sprintf(self->error_info, "Response %d for %s denied by policy.", self->response_code, self->request_method->str);
          switch (filter_type)
            {
            case HTTP_RSP_POLICY:
	      z_policy_lock(self->super.thread);
              if (!z_policy_var_parse(f, "(iO)", &filter_type, &handler))
                {
                  /*LOG
	       	    This message indicates that the response hash contains
		    an invalid POLICY tuple for the given response.  It
		    should contain a valid call-back function in the tuple.
                   */
                  z_proxy_log(self, HTTP_POLICY, 1, 
                              "Error parsing HTTP_RSP_POLICY in response hash; request='%s', response='%d'", 
                              self->request_method->str, self->response_code);
                  z_policy_unlock(self->super.thread);
                  z_proxy_leave(self);
                  return FALSE;
                }
              res = z_policy_call_object(handler, z_policy_var_build("(sssi)", self->request_method->str, self->request_url->str, self->request_version, self->response_code), self->super.session_id);
              if (!z_policy_var_parse(res, "i", &rc))
                {
		  g_string_assign(self->error_info, "Error in policy handler.");
                  rc = HTTP_RSP_REJECT;
                }
              z_policy_var_unref(res);
	      z_policy_unlock(self->super.thread);
	      break;
	    case HTTP_RSP_REJECT:
	      errmsg = NULL;
	      z_policy_lock(self->super.thread);
	      if (!z_policy_var_parse_tuple(f, "i|s", &filter_type, &errmsg))
		{
                  /*LOG
	       	    This message indicates that the response hash contains
		    an invalid REJECT tuple for the given response.
		    It should contain an error message, which is sent back to
		    the client.
                   */
		  z_proxy_log(self, HTTP_POLICY, 1, 
		              "Error parsing HTTP_RSP_REJECT in response hash; request='%s', response='%d'", 
		              self->request_method->str, self->response_code);
		  z_policy_unlock(self->super.thread);
		  z_proxy_leave(self);
		  return FALSE;
		}
	      z_policy_unlock(self->super.thread);
	      if (errmsg)
		g_string_assign(self->error_info, errmsg);
              /* fallthrough */
	    case HTTP_RSP_ACCEPT:
	    case HTTP_RSP_DENY:
	    case HTTP_RSP_ABORT:
	      /* dropped command */
	      rc = filter_type;
	      break;
	    default:
              /*LOG
	        This message indicates that the response hash contains
	        an invalid action for the given response.
	        Check your Zorp configuration.
               */
	      z_proxy_log(self, HTTP_POLICY, 1, 
	                  "Invalid response hash item; request='%s', response='%d'", 
	                  self->request_method->str, self->response_code);
	      z_proxy_leave(self);
	      return FALSE;
            }
          switch (rc)
            {
            case HTTP_RSP_ACCEPT:
              break;
            default:
	    case HTTP_RSP_REJECT:
	      /*LOG
	        This message indicates that the status code returned by the server is
	        not a permitted response code for this request.
	       */
              z_proxy_log(self, HTTP_POLICY, 2, "Response not permitted by policy; req='%s', rsp='%d'", self->request_method->str, self->response_code);
	      self->error_code = HTTP_MSG_POLICY_VIOLATION;
	      z_proxy_leave(self);
              return FALSE;
            }

        }
      
      if (!http_filter_headers(self, EP_SERVER, NULL))
        {
          z_proxy_leave(self);
          return FALSE;
        }
    }
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_format_response(HttpProxy *self, gboolean stacked, GString *response)
{
  if (self->proto_version[EP_SERVER] >= 0x0100)
    {
      if (!stacked)
        {
          http_flat_headers(&self->headers[EP_SERVER]);
          g_string_sprintf(response, "%s %s %s\r\n%s\r\n", 
                  self->response_version, self->response, self->response_msg->str, 
                  self->headers[EP_SERVER].flat->str);
        }
      else
        {
          http_flat_headers_into(&self->headers[EP_SERVER], response);
        }
    }
  else
    g_string_truncate(response, 0);
  return TRUE;
}

static gboolean
http_copy_response(HttpProxy *self)
{
  gboolean suppress_data;

  z_proxy_enter(self);
  
  /* self->connection_hdr must never be NULL for server->client direction when at least HTTP/1.0 is used */

  if (self->proto_version[EP_SERVER] >= 0x0100)
    {  
      g_assert(self->connection_hdr);
      
      if (self->connection_mode != self->server_connection_mode)
        self->connection_hdr->present = TRUE;
        
      if (self->connection_hdr->present)
        {
          g_string_assign(self->connection_hdr->name, self->request_type == HTTP_REQTYPE_SERVER ? "Connection" : "Proxy-Connection");
          http_assign_connection_hdr_value(self, self->connection_hdr->value);
        }
    }    
  suppress_data = (self->request_flags & HTTP_REQ_FLG_HEAD) != 0 ||
                  ((self->request_flags & HTTP_REQ_FLG_CONNECT) != 0 && self->response_code == 200) ||
                  (self->response_flags & HTTP_RESP_FLG_NODATA) != 0;
  if (!http_data_transfer(self, EP_SERVER, self->super.endpoints[EP_SERVER], EP_CLIENT, self->super.endpoints[EP_CLIENT], TRUE, suppress_data, http_format_response))
    {
      /* http_data_transfer already logs */
      z_proxy_leave(self);
      return FALSE;
    }
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
http_handle_connect(HttpProxy *self)
{
  /* connect is enabled here */
  guint i, rc; 
  ZPolicyObj *res;
  gboolean called;
  gchar *success, *remote_host;
  gchar *colon, *end;
  gint remote_host_len, remote_port;

  z_proxy_enter(self);

  self->connection_mode = HTTP_CONNECTION_CLOSE;
  colon = strchr(self->request_url->str, ':');
  if (colon)
    {
      remote_host = self->request_url->str;
      remote_host_len = colon - remote_host;
      remote_port = strtoul(colon + 1, &end, 10);
    }
  else
    {
      self->error_code = HTTP_MSG_CLIENT_SYNTAX;
      g_string_sprintf(self->error_info, "Invalid CONNECT request.");
      /*LOG
        This message indicates that the received CONNECT request did not
        include a port number to connect to.
       */
      z_proxy_log(self, HTTP_VIOLATION, 1, "Missing port number in CONNECT request; url='%s'", self->request_url->str);
      z_proxy_leave(self);
      return FALSE;
    }

  if (http_parent_proxy_enabled(self))
    {
      g_string_assign(self->remote_server, self->parent_proxy->str);
      self->remote_port = self->parent_proxy_port;
    }
  else
    {
      /* From buffer is longer than we want to copy. */
      g_string_assign_len(self->remote_server, remote_host, remote_host_len);
      self->remote_port = remote_port;
    }
  if (!http_connect_server(self))
    {
      z_proxy_leave(self);
      return FALSE;
    }
    
  if (http_parent_proxy_enabled(self))
    {
      GString *request = g_string_sized_new(64);
      
      http_format_request(self, FALSE, request);
      
      if (http_write(self, EP_SERVER, request->str, request->len) != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self);
          return FALSE;
        }
      g_string_free(request, TRUE);
      
      if (!http_fetch_response(self))
        {
          z_proxy_leave(self);
          return FALSE;
        }
      if (self->response_code != 200)
        {
          /*LOG
	    This message indicates that the our parent proxy
	    refused our CONNECT request. It is likely that the
	    parent proxy does not permit CONNECT method.
           */
          z_proxy_log(self, HTTP_ERROR, 1, "Parent proxy refused our CONNECT request; host='%.*s', port='%d'", remote_host_len, remote_host, remote_port);
          
          z_proxy_log(self, HTTP_DEBUG, 6, "Processing response and headers (CONNECT);");
          if (!http_process_response(self) || 
              !http_filter_response(self) || 
              !http_copy_response(self))
            {
              z_proxy_log(self, HTTP_ERROR, 2, "Error copying CONNECT response, or CONNECT response rejected by policy; request='%s', response='%d'", self->request_method->str, self->response_code);
            }
          z_proxy_leave(self);
          return FALSE;
        }
    }
  
  /*LOG
    This message reports that CONNECT method is in use, and
    CONNECT method was accepted by out parent proxy.
   */
  z_proxy_log(self, HTTP_DEBUG, 6, "Starting connect method;");
  
  if (!http_parent_proxy_enabled(self))
    {
      success = "HTTP/1.0 200 Connection established\r\n\r\n";
      if (http_write(self, EP_CLIENT, success, strlen(success)) != G_IO_STATUS_NORMAL)
        {
          /* hmm... I/O error */
          z_proxy_leave(self);
          return FALSE;
        }
    }
  else
    {
      if (!http_process_response(self) ||
          !http_filter_response(self) ||
          !http_copy_response(self))
        {
          z_proxy_log(self, HTTP_ERROR, 2, "Error copying CONNECT response, or CONNECT response rejected by policy; request='%s', response='%d'", self->request_method->str, self->response_code);
          z_proxy_leave(self);
          return FALSE;
        }
    }
    
  /* FIXME: flush already buffered data, before starting plug */
  
  z_policy_lock(self->super.thread);
  res = z_policy_call(self->super.handler, "connectMethod", z_policy_var_build("()"), &called, self->super.session_id);
  
  if (!z_policy_var_parse(res, "i", &rc))
    {
      /*LOG
        This message indicates that an internal error occurred, the
        connectMethod function did not return an integer. Please report this
        event to the Zorp QA team (at devel@balabit.com).
       */
      z_proxy_log(self, HTTP_ERROR, 1, "Internal error, connectMethod is expected to return an integer;");
      self->error_code = HTTP_MSG_POLICY_SYNTAX;
      z_policy_var_unref(res);
      z_policy_unlock(self->super.thread);
      z_proxy_leave(self);
      return FALSE;
    }
  z_policy_var_unref(res);
  z_policy_unlock(self->super.thread);
  
  if (rc != HTTP_REQ_ACCEPT)
    {
      g_string_assign(self->error_info, "CONNECT request denied by policy.");
      /*LOG
        This log message indicates that the specified HTTP request was not permitted by your policy.
       */
      z_proxy_log(self, CORE_POLICY, 2, "Request not permitted by policy; req='%s'", self->request_method->str);
      z_proxy_leave(self);
      return FALSE;
    }

  /* release, but don't close fds */
  for (i = EP_CLIENT; i < EP_MAX; i++)
    {
      z_stream_unref(self->super.endpoints[i]);
      self->super.endpoints[i] = NULL;
    }

  z_proxy_leave(self);
  return TRUE;
}

static void
http_main(ZProxy *s)
{
  HttpProxy *self = Z_CAST(s, HttpProxy);
  self->request_count = 0;
  
  z_proxy_enter(self);
  http_client_stream_init(self);

  /* loop for keep-alive */
  while (1)
    {
      self->error_code = -1;
      /*LOG
        This message reports that Zorp is fetching the request and the
	headers from the client.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "Fetching request and headers;");
      /* fetch request */
      if (!http_fetch_request(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_CLIENT_SYNTAX;
	  break;
	}
      /*LOG
        This message reports that Zorp is processing the fetched request and
	the headers.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "processing request and headers;");
      if (!http_process_request(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_CLIENT_SYNTAX;
	  break;
	}

      if (self->max_keepalive_requests != 0 && 
          self->request_count >= self->max_keepalive_requests - 1)
        {
          /*LOG
	    This message reports that the maximum number of requests in a keep-alive loop is
	    reached, and Zorp is closing after this request. Check the max_keepalive_request
	    attribute.
           */
          z_proxy_log(self, HTTP_POLICY, 3, 
                      "Maximum keep-alive request reached, setting connection mode to close; count='%d', max='%d'", 
                      self->request_count, self->max_keepalive_requests);
          self->connection_mode = HTTP_CONNECTION_CLOSE;
        }

      /*LOG
        This message reports that Zorp is filtering the processed
        request and the headers.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "Filtering request and headers;");
      if (!http_filter_request(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_POLICY_SYNTAX;
          break;
        }
        
      /*LOG 
        This message indicates that Zorp is recechecking the HTTP request
        after possible changes performed by the policy layer.
       */
      z_proxy_log(self, HTTP_DEBUG, 6, "Reprocessing filtered request;");
      if (!http_process_filtered_request(self))
        {
          if (self->error_code == -1)
            self->error_code = HTTP_MSG_CLIENT_SYNTAX;
          break;
        }

      if (self->server_protocol == HTTP_PROTO_HTTP)
        {
          
          if ((self->request_flags & HTTP_REQ_FLG_CONNECT))
            {
              if (http_handle_connect(self))
                {
                  /* connect method was successful, we can now safely quit */
                  z_proxy_leave(self);
                  return;
                }
              else
                break;
            }
          /*LOG
            This message reports that Zorp is sending the filtered request and
            headers, and copies the requests data to the server.
           */
          z_proxy_log(self, HTTP_DEBUG, 6, "Sending request and headers, copying request data;");
          if (!http_copy_request(self))
            {
              if (self->error_code == -1)
                self->error_code = HTTP_MSG_CLIENT_SYNTAX;
              break;
            }
          /*LOG
            This message reports that Zorp is fetching the response and headers
            from the server.
           */
          z_proxy_log(self, HTTP_DEBUG, 6, "Fetching response and headers;");
          if (!http_fetch_response(self))
            {
              if (self->error_code == -1)
                {
                  self->error_code = HTTP_MSG_SERVER_SYNTAX;
                }
              break;
            }
          /*LOG
            This message reports that Zorp is processing the fetched response
            and the headers.
           */
          z_proxy_log(self, HTTP_DEBUG, 6, "Processing response and headers;");
          if (!http_process_response(self))
            {
              if (self->error_code == -1)
                self->error_code = HTTP_MSG_SERVER_SYNTAX;
              break;
            }
          /*LOG
            This message reports that Zorp is filtering the processed
            response and the headers.
           */
          z_proxy_log(self, HTTP_DEBUG, 6, "Filtering response and headers;");
          if (!http_filter_response(self))
            {
              if (self->error_code == -1)
                self->error_code = HTTP_MSG_POLICY_SYNTAX;
              break;
            }
          /*LOG
            This message reports that Zorp is sending the filtered
            response and headers, and copies the response data to the client.
           */
          z_proxy_log(self, HTTP_DEBUG, 6, "Copying response and headers, copying response data;");
          if (!http_copy_response(self))
            {
              if (self->error_code == -1)
                self->error_code = HTTP_MSG_SERVER_SYNTAX;
              break;
            }
        }
      else if (self->server_protocol == HTTP_PROTO_FTP)
        {
          if (!http_handle_ftp_request(self))
            {
              if (self->error_code == -1)
                self->error_code = HTTP_MSG_FTP_ERROR;
            }
          self->connection_mode = HTTP_CONNECTION_CLOSE;
        }
      else
        {
          /*LOG
            This message indicates an internal error in HTTP proxy. Please
            report this event to the Zorp QA team (at devel@balabit.com).
           */
          z_proxy_log(self, CORE_ERROR, 1, "Internal error, invalid server_protocol; server_protocol='%d'", self->server_protocol);
        }
      if (self->connection_mode == HTTP_CONNECTION_CLOSE)
      	break;
      
      if (self->server_connection_mode == HTTP_CONNECTION_CLOSE)
        {
          /* close the server connection, but keep the client connection in-tact */
          self->force_reconnect = TRUE;
        }
	
      self->request_count++;
    }
  /*LOG
    This message reports that Zorp is exiting the keep-alive loop and
    closing the connection.
   */
  z_proxy_log(self, HTTP_DEBUG, 6, "exiting keep-alive loop;");
  if (self->error_code != -1)
    {
      http_error_message(self, self->error_status, self->error_code, self->error_info);
    }
  /* in some cases the client might still have some data already queued to us, 
   * fetch and ignore that data to avoid the RST sent in these cases 
   */
  http_fetch_buffered_data(self);
  z_proxy_leave(self);
}

static gboolean
http_config(ZProxy *s)
{
  HttpProxy *self = Z_CAST(s, HttpProxy);
  
  http_config_set_defaults(self);
  http_register_vars(self);
  if (Z_SUPER(s, ZProxy)->config(s))
    {
      http_config_init(self);
      return TRUE;
    }
  return FALSE;
}

static void
http_proxy_free(ZObject *s)
{
  HttpProxy *self = Z_CAST(s, HttpProxy);
  guint i;

  z_proxy_enter(s);
  for (i = EP_CLIENT; i < EP_MAX; i++)
    {
      http_destroy_headers(&self->headers[i]);
    }

  g_string_free(self->old_auth_header, TRUE);
  g_string_free(self->stack_info, TRUE);
  g_string_free(self->auth_header_value, TRUE);
  g_string_free(self->response_msg, TRUE);
  g_string_free(self->connected_server, TRUE);
  g_string_free(self->remote_server, TRUE);
  
  g_string_free(self->request_url, TRUE);
  http_destroy_url(&self->request_url_parts);
  
  /* NOTE: hashes are freed up by pyvars */
  z_poll_unref(self->poll);
  z_proxy_leave(s);
  z_proxy_free_method(s);
}

static ZProxy *
http_proxy_new(ZProxyParams *params)
{
  HttpProxy *self;
  ZProxyResultIface *iface;

  z_enter();
  self = Z_CAST(z_proxy_new(Z_CLASS(HttpProxy), params), HttpProxy);

  iface = z_proxy_iface_new(Z_CLASS(HttpProxyResultIface), &self->super);
  z_proxy_add_iface(&self->super, iface);
  z_object_unref(&iface->super);
  
  z_proxy_start(&self->super);
  
  z_leave();
  return (&self->super);
}

static void http_proxy_free(ZObject *s);

ZProxyFuncs http_proxy_funcs =
{
  { 
    Z_FUNCS_COUNT(ZProxy),
    http_proxy_free,
  },
  http_config,
  NULL,
  http_main,
  NULL,
  NULL,
  NULL
};

ZClass HttpProxy__class = 
{
  Z_CLASS_HEADER,
  &ZProxy__class,
  "HttpProxy",
  sizeof(HttpProxy),
  &http_proxy_funcs.super
};



gint
zorp_module_init(void)
{
  z_registry_add("http", ZR_PROXY, http_proxy_new);
  return TRUE;
}
