/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: httpfltr.c,v 1.83 2004/07/26 11:45:57 bazsi Exp $
 *
 * Author: Balazs Scheidler <bazsi@balabit.hu>
 * Auditor: 
 * Last audited version: 
 * Notes:
 *
 ***************************************************************************/
 
#include "http.h"

#include <zorp/log.h>
#include <zorp/io.h>
#include <zorp/transfer.h>

/* transfer states */

#define HTTP_SR_INITIAL           0
#define HTTP_SR_PUSH_HEADERS	  10
#define HTTP_SR_PUSH_HEADERS_EOL  11
#define HTTP_SR_PUSH_HEADERS_MAX  11

#define HTTP_SR_READ_INITIAL      20
#define HTTP_SR_READ_CHUNK_LENGTH 21
#define HTTP_SR_READ_CHUNK        22
#define HTTP_SR_READ_FOOTER       23
#define HTTP_SR_READ_ENTITY       24     /* reading content-length or EOF terminated entity */

#define HTTP_DW_INITIAL            0

#define HTTP_DW_POP_HEADERS        10
#define HTTP_DW_POP_HEADERS_CR     11
#define HTTP_DW_POP_HEADERS_LF     12
#define HTTP_DW_POP_HEADERS_MAX    12


#define HTTP_DW_WRITE_PREAMBLE     20
#define HTTP_DW_WRITE_INITIAL      21
#define HTTP_DW_FORMAT_CHUNK_LENGTH 22
#define HTTP_DW_WRITE_CHUNK_LENGTH 23
#define HTTP_DW_WRITE_CHUNK        24
#define HTTP_DW_WRITE_FOOTER       25

#define ZTS_HTTP_TRANSFER_DELAYED  0x10000

struct _HttpTransfer
{
  ZTransfer super;
  
  GString *preamble;
  guint preamble_ofs;
  
  ZPolicyObj *stack_object;
  gint stack_type;
  
  /* transfer endpoints */
  gint transfer_from, transfer_to;
  
  /* the headers to send to the downstream proxy */
  GString *stacked_preamble;
  /* offset within mime_headers if writing blocked */
  guint stacked_preamble_ofs;
  guint stacked_preamble_read_bytes;
  
  /* function used to format the preamble to stacked proxy/peer */
  HttpTransferPreambleFunc format_preamble_func;
  
  /* whether to send preamble on shutdown */
  gboolean flush_on_shutdown;
  
  /* whether to push mime headers to the downstream proxy */
  gboolean push_mime_headers;
  
  /* whether to force the end of the connection */
  gboolean force_nonpersistent_mode;
  /* we can stay persisent, but only if we receive a content-length hint from downstream proxy */
  gboolean persistent_with_cl_hint_only;
  
  HttpHeader *transfer_encoding_hdr, *content_length_hdr;
  
  /* used while stripping off MIME headers returned by the downstream proxy */
  gint dst_eol_count;
  
  /* complete content_length, -2 if no entity, -1 if length unknown, otherwise the exact length */
  gint content_length;
  
  /* indicates whether source is chunked */
  gboolean src_chunked;
  
  /* source read state */
  guint src_read_state;
  
  /* indicates that the current chunk is an EOF chunk */
  gboolean src_last_chunk; 
  
  /* indicates that this is the last chunk and that it was truncated 
   * because the body was over max_body_length */
  gboolean src_chunk_truncated;
  
  /* the number of bytes waiting to be read in the current chunk */
  guint src_chunk_left;
  
  /* the total number of bytes read during this transfer */
  guint src_whole_length;
  
  gboolean dst_chunked;
  guint dst_write_state;
  
  /* the number of bytes still to be written to the destination */
  guint dst_chunk_left;
  
  gchar dst_chunk_length_buf[32];
  guint dst_chunk_length_ofs;
  guint dst_chunk_length_end;
  guint dst_footer_ofs;
  
  /* the total number of bytes in the chunk being written */
  guint dst_chunk_length;
  
  /* the total number of transferred bytes on the write side */
  guint dst_whole_length;
  
};

extern ZClass HttpTransfer__class;


/* HttpTransfer implementation */

static GIOStatus 
http_transfer_src_read(ZTransfer *s, ZStream *stream, gchar *buf, gsize count, gsize *bytes_read, GError **err)
{
  HttpTransfer *self = Z_CAST(s, HttpTransfer);
  HttpProxy *owner = (HttpProxy *) s->owner;
  GError *local_error = NULL;
  gsize br;
  GIOStatus res = G_IO_STATUS_NORMAL;

  if (self->src_read_state == HTTP_SR_INITIAL)
    self->src_read_state = HTTP_SR_PUSH_HEADERS;
  if (self->src_read_state >= HTTP_SR_PUSH_HEADERS && self->src_read_state <= HTTP_SR_PUSH_HEADERS_MAX)
    {
      if (self->push_mime_headers)
        {
          *bytes_read = 0;
          switch (self->src_read_state)
            {
            case HTTP_SR_PUSH_HEADERS:
              {
                gint move = MIN(count, self->stacked_preamble->len - self->stacked_preamble_ofs);
            
                memmove(buf, self->stacked_preamble->str + self->stacked_preamble_ofs, move);
                self->stacked_preamble_ofs += move;
                *bytes_read = move;
                if (self->stacked_preamble_ofs == self->stacked_preamble->len)
                  {
                    self->src_read_state = HTTP_SR_PUSH_HEADERS_EOL;
                  }
                else
                  return G_IO_STATUS_NORMAL;
                  
                /* fallthrough if the whole headers were copied */
              }
            case HTTP_SR_PUSH_HEADERS_EOL:
              /* append CRLF as self->stacked_preamble is not CRLF terminated */
              if (*bytes_read + 2 <= count)
                {
                  buf[*bytes_read] = '\r';
                  buf[*bytes_read+1] = '\n';
                  *bytes_read += 2;
                  self->src_read_state = HTTP_SR_READ_INITIAL;
                }
              return G_IO_STATUS_NORMAL;
            }
        }
      else
        {
          self->src_read_state = HTTP_SR_READ_INITIAL;
        }
    }
  *bytes_read = 0;
  if (self->src_chunked)
    {
      /* read as a chunked stream */
      switch (self->src_read_state)
        {
        case HTTP_SR_READ_INITIAL:
          self->src_whole_length = 0;
          self->src_read_state = HTTP_SR_READ_CHUNK_LENGTH;
          /* fallthrough */
        case HTTP_SR_READ_CHUNK_LENGTH:
          {
            gchar line_buf[32], *line;
            gsize line_length;
            guint32 chunk_length;
            gchar *end;
            
            z_stream_line_set_poll_partial(stream, FALSE);
            res = z_stream_line_get(stream, &line, &line_length, NULL);
            z_stream_line_set_poll_partial(stream, TRUE);
            if (res == G_IO_STATUS_NORMAL)
              {
                /* a complete line was read, check if it is a valid chunk length */
                if (line_length > sizeof(line_buf) - 3)
                  {
		    /*LOG
		      This message indicates that the chunk length line is too long.
		      It is likely caused by a buggy client or server.
		     */
                    z_proxy_log(self->super.owner, HTTP_VIOLATION, 1, "Chunk length line too long; line='%.*s'", (gint) line_length, line);
                    res = G_IO_STATUS_ERROR;
                    break;
                  }
                g_strlcpy(line_buf, line, MIN(line_length + 1, sizeof(line_buf)));
                chunk_length = strtoul(line_buf, &end, 16);
                
                if (end == line_buf)
                  {
                    /* hmm... invalid chunk length */
		    /*LOG
		      This message indicates that the chunk length is invalid.
		      It is likely caused by a buggy client or server.
		     */
                    z_proxy_log(self->super.owner, HTTP_VIOLATION, 1, "Invalid chunk length; line='%s'", line_buf);
                    res = G_IO_STATUS_ERROR;
                    break;
                  }
                
                /* 
                 * NOTE: strlcpy ensures that line_buf, and thus the string
                 * pointed by end is NUL terminated, thus we will not overflow
                 * our buffer
                 */
                
                while (*end == ' ')
                  end++;
                if (*end == ';')
                  {
                    /* ignore and strip chunk extensions */
                    *end = 0;
                  }
                if (*end)
                  {
                    /* hmm... invalid chunk length */
		    /*LOG
		      This message indicates that the chunk length is invalid.
		      It is likely caused by a buggy client or server.
		     */
                    z_proxy_log(self->super.owner, HTTP_VIOLATION, 1, "Invalid chunk length; line='%s'", line_buf);
                    res = G_IO_STATUS_ERROR;
                    break;
                  }
                
                if ((owner->max_chunk_length && chunk_length > owner->max_chunk_length) ||
                    (chunk_length & 0x80000000))
                  {
		    /*LOG
		      This message indicates that the length of the chunk is larger than allowed 
		      or is a negative number. Check the 'max_chunk_length' attribute.
		     */
                    z_proxy_log(self->super.owner, HTTP_POLICY, 2, "Chunk too large; length='%d', max_chunk_length='%d'", chunk_length, owner->max_chunk_length);
                    res = G_IO_STATUS_ERROR;
                    break;
                  }
                  
                if (owner->max_body_length && (guint) self->src_whole_length + chunk_length > owner->max_body_length)
                  {
                    /* this chunk would be over body_length limit */
                    
                    chunk_length = owner->max_body_length - self->src_whole_length;
                    self->src_chunk_left = chunk_length;
                    self->force_nonpersistent_mode = TRUE;
                    self->src_chunk_truncated = TRUE;
                  }
                self->src_chunk_left = chunk_length;
                self->src_last_chunk = chunk_length == 0;
                self->src_read_state = HTTP_SR_READ_CHUNK;
                /* fall through */
              }
            else
              break;
          }
        case HTTP_SR_READ_CHUNK:
          if (!self->src_last_chunk)
            {
              res = z_stream_read(stream, buf, MIN(self->src_chunk_left, count), &br, &local_error);
              if (res == G_IO_STATUS_NORMAL)
                {
                  self->src_whole_length += br;
                  self->src_chunk_left -= br;
                  *bytes_read = br;
                }
              else if (res == G_IO_STATUS_EOF)
                {
                  /* unexpected eof */
		  /*LOG
		    This message indicates that Zorp unexpectedly got EOF during
		    chunk encoded data transfer. It is likely a caused by a buggy client
		    or server.
		   */
                  z_proxy_log(self->super.owner, HTTP_VIOLATION, 1, "Unexpected EOF while dechunking stream;");
                  res = G_IO_STATUS_ERROR;
                  break;
                }
              if (self->src_chunk_left == 0)
                {
                  self->src_read_state = HTTP_SR_READ_FOOTER;
                }
              break;
            }
          else
            {
              self->src_read_state = HTTP_SR_READ_FOOTER;
              /* fallthrough */
            }
        case HTTP_SR_READ_FOOTER:
          {
            gchar *line;
            gsize line_length;
            
            if (!self->src_chunk_truncated)
              {
                z_stream_line_set_poll_partial(stream, FALSE);
                res = z_stream_line_get(stream, &line, &line_length, NULL);
                z_stream_line_set_poll_partial(stream, TRUE);
              }
            else
              {
                res = G_IO_STATUS_EOF;
              }
            if (res == G_IO_STATUS_NORMAL)
              {
                if (line_length != 0)
                  {
		    /*LOG
		      This message indicates that the chunk footer contains data.
		      It is likely caused by a buggy client or server.
		     */
                    z_proxy_log(self->super.owner, HTTP_VIOLATION, 1, "Chunk footer is not an empty line;");
                    res = G_IO_STATUS_ERROR;
                    break;
                  }
                if (self->src_last_chunk)
                  {
                    res = G_IO_STATUS_EOF;
                  }
                else
                  {
                    self->src_read_state = HTTP_SR_READ_CHUNK_LENGTH;
                    /* come back later */
                    res = G_IO_STATUS_AGAIN;
                  }
                break;
              }
            break;
          }
        }
    }
  else
    {
      /* copy until EOF or self->content_length bytes */
      if (self->content_length == HTTP_LENGTH_NONE)
        {
          res = G_IO_STATUS_EOF;
        }
      else
        {
          if (self->src_read_state == HTTP_SR_INITIAL)
            {
              self->src_whole_length = 0;
              self->src_read_state = HTTP_SR_READ_ENTITY;
            }
          
          if (self->content_length == HTTP_LENGTH_UNKNOWN)
            {
              if (owner->max_body_length && self->src_whole_length + count >= owner->max_body_length)
                {
                  count = owner->max_body_length - self->src_whole_length;
                }
              if (count == 0)
                {
                  self->force_nonpersistent_mode = TRUE;
                  res = G_IO_STATUS_EOF;
                }
              else
                res = z_stream_read(stream, buf, count, &br, &local_error);
            }
          else
            {
              /* for specified content-length, max_body_length has already
                 been processed, and content_length contains the number of
                 bytes to be transferred, but maximum max_body_length */                 
              if ((guint) self->content_length == self->src_whole_length)
                res = G_IO_STATUS_EOF;
              else
                res = z_stream_read(stream, buf, MIN(count, self->content_length - self->src_whole_length), &br, &local_error);
            }
          
          if (res == G_IO_STATUS_NORMAL)
            {
              self->src_whole_length += br;
              *bytes_read = br;
            }
        }
    }
  if (local_error)
    g_propagate_error(err, local_error);
  return res;
}

static GIOStatus
http_transfer_src_shutdown(ZTransfer *self G_GNUC_UNUSED, ZStream *s G_GNUC_UNUSED, gint shutdown_mode G_GNUC_UNUSED, GError **err G_GNUC_UNUSED)
{
  /* do nothing */
  return G_IO_STATUS_NORMAL;
}

static GIOStatus
http_transfer_dst_write_preamble(HttpTransfer *self, ZStream *stream, GError **err)
{
  GIOStatus res = G_IO_STATUS_NORMAL;
  GError *local_error = NULL;
  gsize bw;

  http_log_headers((HttpProxy *) self->super.owner, self->transfer_from, "postfilter");
  res = z_stream_write(stream, &self->preamble->str[self->preamble_ofs], self->preamble->len - self->preamble_ofs, &bw, &local_error);
  if (res == G_IO_STATUS_NORMAL)
    {
      self->preamble_ofs += bw;
      if (self->preamble_ofs == self->preamble->len)
        {
          self->dst_write_state = HTTP_DW_POP_HEADERS;
        }
      else
        {
          res = G_IO_STATUS_AGAIN;
        }
    }
  else if (g_error_matches(local_error, G_IO_CHANNEL_ERROR, G_IO_CHANNEL_ERROR_PIPE))
    {
      /* FIXME: this is a hack. A better solution would be to propagate
       * GError to the proxy so this case can be handled there cleanly */
       
      ((HttpProxy *) self->super.owner)->reattempt_connection = TRUE;
    }
  if (local_error)
    g_propagate_error(err, local_error);
  return res;
}

/* NOTE: this function assumes that ZTransfer does not change the buffer
 * between successive calls when an G_IO_STATUS_AGAIN is returned. This is
 * required for performance reasons, as otherwise this function would have
 * to copy the buffer contents to a private buffer and return
 * G_IO_STATUS_AGAIN while it is being flushed.*/
static GIOStatus 
http_transfer_dst_write(ZTransfer *s, ZStream *stream, const gchar *buf, gsize count, gsize *bytes_written, GError **err)
{
  HttpTransfer *self = Z_CAST(s, HttpTransfer);
  HttpProxy *owner = (HttpProxy *) self->super.owner;
  GError *local_error = NULL;
  gsize bw;
  GIOStatus res = G_IO_STATUS_NORMAL;
  
  *bytes_written = 0;
  
  if (self->dst_write_state == HTTP_DW_INITIAL)
    self->dst_write_state = HTTP_DW_POP_HEADERS;

  if (self->dst_write_state >= HTTP_DW_POP_HEADERS && self->dst_write_state <= HTTP_DW_POP_HEADERS_MAX)
    {
      if (self->push_mime_headers)
        {
          gsize i;
          
          for (i = 0; i < count; i++)
            {
              switch (self->dst_write_state)
                {
                case HTTP_DW_POP_HEADERS:
                  switch (buf[i])
                    {
                    case '\r':
                      self->dst_write_state = HTTP_DW_POP_HEADERS_CR;
                      break;
                    case '\n':
                      self->dst_write_state = HTTP_DW_POP_HEADERS_LF;
                      break;
                    default:
                      break;
                    }
                  break;
                case HTTP_DW_POP_HEADERS_CR:
                  switch (buf[i])
                    {
                    case '\n':
                      self->dst_write_state = HTTP_DW_POP_HEADERS_LF;
                      break;
                    default:
                      self->dst_write_state = HTTP_DW_POP_HEADERS;
                      break;
                    }
                  break;
                }
              if (self->dst_write_state == HTTP_DW_POP_HEADERS_LF)
                {
                  /* new line found */
                  self->dst_eol_count++;
                  if (self->dst_eol_count == 2)
                    {
                      /* end of headers */
                      i++;
                      self->dst_write_state = HTTP_DW_WRITE_PREAMBLE;
                      break;
                    }
                  else
                    {
                      self->dst_write_state = HTTP_DW_POP_HEADERS;
                    }
                }
              else if (self->dst_write_state == HTTP_DW_POP_HEADERS)
                self->dst_eol_count = 0;
                  
            }
          /* at this point we have either processed the whole buffer (i ==
           * count), or a small chunk of data is at the tail of our headers
           * to be ignored. In either case return G_IO_STATUS_NORMAL and the
           * number of processed bytes. */
           
          *bytes_written = i;
          self->stacked_preamble_read_bytes += i;
          return G_IO_STATUS_NORMAL;
        }
      else
        self->dst_write_state = HTTP_DW_WRITE_PREAMBLE;
    }

  if (self->dst_write_state == HTTP_DW_WRITE_PREAMBLE)
    {
      gboolean reformat_preamble = FALSE;
      
      if (owner->content_length_hint_set)
        {
          /* we stay in persistent mode if possible */
	  /*LOG
	    This message reports that the stacked proxy sent a content-length hint on how much
	    data will be sent and the http proxy is using it to set the Content-Length header.
	   */
          z_proxy_log(owner, HTTP_DEBUG, 6, "Stacked proxy sent a content-length hint, using it; expected_content_length='%zd'", owner->content_length_hint);
          g_string_sprintf(self->content_length_hdr->value, "%zd", owner->content_length_hint - self->stacked_preamble_read_bytes);
          self->content_length_hdr->present = TRUE;
          self->transfer_encoding_hdr->present = FALSE;
          self->dst_chunked = FALSE;
          reformat_preamble = TRUE;
        }
      else if (self->persistent_with_cl_hint_only)
        {
          /* when we require content-length hint to remain persistent and we
             have no content-length hint, we force ourselves to
             non-persistent mode */             
          self->force_nonpersistent_mode = TRUE;
        }
        
      if (self->force_nonpersistent_mode && owner->connection_mode == HTTP_CONNECTION_KEEPALIVE)
        {
          owner->connection_mode = owner->server_connection_mode = HTTP_CONNECTION_CLOSE;
          
          /* we must change our connection header right here as our core has
           * already made its decision, but we must override that */
          
          if (owner->connection_hdr)
            {
              g_string_assign(owner->connection_hdr->value, "close");
              reformat_preamble = TRUE;
            }
        }
      if (reformat_preamble)
        self->format_preamble_func(owner, FALSE, self->preamble);

      /* take care about the preamble (request/response itself) */
      res = http_transfer_dst_write_preamble(self, stream, &local_error);
      if (res != G_IO_STATUS_NORMAL)
        {
          goto propagate_exit;
        }
      self->dst_write_state = HTTP_DW_WRITE_INITIAL;
    }

    
  /* ok, now take care about the data, and possibly enchunk it on the way */
  if (self->dst_chunked)
    {
      switch (self->dst_write_state)
        {
        case HTTP_DW_WRITE_INITIAL:
          self->dst_write_state = HTTP_DW_FORMAT_CHUNK_LENGTH;
          self->dst_whole_length = 0;
          /* fallthrough */
        case HTTP_DW_FORMAT_CHUNK_LENGTH:
          {
            if (count == 0)
              { 
                res = G_IO_STATUS_NORMAL;
                break;
              }
            self->dst_chunk_length = 0;
            self->dst_chunk_left = count;
            g_snprintf(self->dst_chunk_length_buf, sizeof(self->dst_chunk_length_buf), "%x\r\n", self->dst_chunk_left);
            self->dst_chunk_length_ofs = 0;
            self->dst_chunk_length_end = strlen(self->dst_chunk_length_buf);
            self->dst_write_state = HTTP_DW_WRITE_CHUNK_LENGTH;
          }
        case HTTP_DW_WRITE_CHUNK_LENGTH:
          {
            res = z_stream_write(stream, &self->dst_chunk_length_buf[self->dst_chunk_length_ofs], self->dst_chunk_length_end - self->dst_chunk_length_ofs, &bw, &local_error);
            if (res == G_IO_STATUS_NORMAL)
              {
                self->dst_chunk_length_ofs += bw;
                if (self->dst_chunk_length_ofs == self->dst_chunk_length_end)
                  self->dst_write_state = HTTP_DW_WRITE_CHUNK;
              }
            else
              {
                break;
              }
          /* fallthrough */
          }
        case HTTP_DW_WRITE_CHUNK:
          {
            /* NOTE: here lies the assumptions that ZTransfer neither
             * changes the buffer nor count when G_IO_STATUS_AGAIN
             * is returned */
            res = z_stream_write(stream, &buf[count - self->dst_chunk_left], self->dst_chunk_left, &bw, &local_error);
            if (res == G_IO_STATUS_NORMAL)
              {
                self->dst_chunk_length += bw;
                self->dst_chunk_left -= bw;
                if (self->dst_chunk_left == 0)
                  {
                    self->dst_write_state = HTTP_DW_WRITE_FOOTER;
                  }
                else
                  {
                    *bytes_written = 0;
                    res = G_IO_STATUS_AGAIN;
                    break;
                  }
              }
            else
              {
                break;
              }
          }
          self->dst_footer_ofs = 0;
        case HTTP_DW_WRITE_FOOTER:
          {
            gchar line_buf[] = "\r\n";
            g_assert(self->dst_footer_ofs < 2);
            res = z_stream_write(stream, &line_buf[self->dst_footer_ofs], 2 - self->dst_footer_ofs, &bw, &local_error);
            if (res == G_IO_STATUS_NORMAL)
              {
                self->dst_footer_ofs += bw;
                if (self->dst_footer_ofs == 2)
                  {
                    self->dst_whole_length += self->dst_chunk_length;
                    *bytes_written = self->dst_chunk_length;
                    self->dst_write_state = HTTP_DW_FORMAT_CHUNK_LENGTH;
                  }
              }
            else
              {
                break;
              }
          }
        }
    }
  else
    {
      res = z_stream_write(stream, buf, count, bytes_written, &local_error);
    }
 propagate_exit:
  if (local_error)
    g_propagate_error(err, local_error);
  return res;
}

static GIOStatus
http_transfer_dst_shutdown(ZTransfer *s, ZStream *stream, gint shutdown_mode, GError **err)
{
  HttpTransfer *self = Z_CAST(s, HttpTransfer);
  GIOStatus res = G_IO_STATUS_NORMAL;
  GError *local_error = NULL;
  gsize bw;
  gboolean delay_transfer;
  
  if (shutdown_mode == SHUT_WR || shutdown_mode == SHUT_RDWR)
    {
      /* write shutdown, otherwise we are not interested */
      
      /* delay preamble and closing chunk if we want to write an error page */
      delay_transfer = (self->dst_write_state == HTTP_DW_INITIAL && 
                        (!!(s->status & ZTS_FAILED) || (((HttpProxy *) s->owner)->stack_decision != Z_ACCEPT))) && 
                        !self->flush_on_shutdown;
      if (delay_transfer)
        {
          s->status |= ZTS_HTTP_TRANSFER_DELAYED;
        }
      else
        {
          if (self->dst_write_state == HTTP_DW_INITIAL || self->dst_write_state == HTTP_DW_WRITE_PREAMBLE)
            res = http_transfer_dst_write_preamble(self, stream, &local_error);
          if (res == G_IO_STATUS_NORMAL)
            {
              if (self->content_length != HTTP_LENGTH_NONE && self->dst_chunked)
                {
                  gchar line_buf[] = "0\r\n\r\n";
                  
                  res = z_stream_write(stream, line_buf, 5, &bw, &local_error);
                  if (bw != 5)
                    {
                      res = G_IO_STATUS_ERROR;
                    }
                }
            }
        }
    }
  if (local_error)
    g_propagate_error(err, local_error);
  return res;
}

static void
http_transfer_select_stacked_proxy(HttpTransfer *self, gint side)
{
  ZPolicyObj *proxy_stack_tuple = NULL, *stack_object = NULL;
  gboolean called;
  
  /* query python for a stacked proxy */
  
  z_policy_lock(self->super.owner->thread);
  
  self->stack_type = HTTP_STK_NONE;
  self->stack_object = NULL;
  proxy_stack_tuple = z_policy_call(self->super.owner->handler, "requestStack", z_policy_var_build("(i)", side), &called, self->super.owner->session_id);
  if (called && !proxy_stack_tuple)
    {
      goto unref_unlock;
    }
  if (!z_policy_var_parse(proxy_stack_tuple, "(iO)", &self->stack_type, &stack_object))
    {
      /*LOG
        This message indicates that the request_stack or response_stack hash
	contains an invalid stacking tuple. It should contain a (stack_type, proxy_class) tuple.
	Check your Zorp configuration.
       */
      z_proxy_log(self->super.owner, HTTP_POLICY, 3, "Invalid stacking tuple returned by policy; side='%d'", side);
      goto unref_unlock;
    }
  if (self->stack_type < HTTP_STK_NONE || self->stack_type > HTTP_STK_MIME)
    {
      /*LOG
        This message indicates that the request_stack or response_stack hash
	contains an invalid stacking type. Check your Zorp configuration.
       */
      z_proxy_log(self, HTTP_POLICY, 3, "Invalid stacking type; type='%d'", self->stack_type);
      self->stack_type = HTTP_STK_NONE;
      goto unref_unlock;
    }
  if (self->stack_type != HTTP_STK_NONE && stack_object)
    {
      self->stack_object = stack_object;
      z_policy_var_ref(stack_object);
    }
    
 unref_unlock:
  z_policy_var_unref(proxy_stack_tuple);
  z_policy_unlock(self->super.owner->thread);
}


static HttpTransfer *
http_transfer_new(HttpProxy *owner, 
                  guint from, ZStream *from_stream, 
                  guint to, ZStream *to_stream, 
                  gboolean expect_data, gboolean suppress_data, 
                  HttpTransferPreambleFunc format_preamble)
{
  HttpTransfer *self;
  HttpHeaders *headers = &owner->headers[from];
  gboolean chunked; /* original entity is chunked */
  
  z_proxy_enter(owner);
  
  self = Z_CAST(z_transfer_new(Z_CLASS(HttpTransfer), 
                               &owner->super, owner->poll, 
                               from_stream, to_stream, NULL, 
                               owner->buffer_size, 
                               owner->timeout, 
                               ZTF_COPY_TO_SERVER | ZTF_SEPARATE_SHUTDOWN), 
                HttpTransfer);
  self->transfer_from = from;
  self->transfer_to = to;
  http_transfer_select_stacked_proxy(self, from);
  self->format_preamble_func = format_preamble;
  self->preamble = g_string_sized_new(0);
  self->stacked_preamble = g_string_sized_new(0);
  self->flush_on_shutdown = !expect_data || suppress_data;
  self->force_nonpersistent_mode = FALSE;
  
  if (!suppress_data)
    {

      if (http_lookup_header(headers, "Transfer-Encoding", &self->transfer_encoding_hdr))
        {
          chunked = strcasecmp(self->transfer_encoding_hdr->value->str, "chunked") == 0;
        }
      else
        {
          chunked = FALSE;
          self->transfer_encoding_hdr = http_add_header(headers, "Transfer-Encoding", 17, "", 0);
        }
        
      if (http_lookup_header(headers, "Content-Length", &self->content_length_hdr))
        {
          gchar *end;
          
          self->content_length = strtoul(self->content_length_hdr->value->str, &end, 10);
          
          if ((guint) (end - self->content_length_hdr->value->str) != self->content_length_hdr->value->len)
            {  
              /* content-length not a number */
              /*LOG
                This message indicates that the Content-Length headers value
                is not a valid number. It is likely caused by a buggy client or server.
               */
              z_proxy_log(owner, HTTP_VIOLATION, 1, "The header 'Content-Length' was present, but is not a number; content_length='%s'", self->content_length_hdr->value->str);
              z_object_unref(&self->super.super);
              z_proxy_leave(owner);
              return NULL;
            }
        }
      else
        {
          self->content_length_hdr = http_add_header(headers, "Content-Length", 14, "", 0);
          self->content_length = expect_data || chunked ? HTTP_LENGTH_UNKNOWN : HTTP_LENGTH_NONE;
        }
        
      self->transfer_encoding_hdr->present = FALSE;
      self->content_length_hdr->present = FALSE;
      
      format_preamble((HttpProxy *) self->super.owner, TRUE, self->stacked_preamble);
      
      self->src_chunked = FALSE;
      self->dst_chunked = FALSE;

      if (self->stack_type != HTTP_STK_NONE)
        {
          if (chunked && (to == EP_SERVER || (owner->proto_version[to] > 0x0100)))
            {
              self->src_chunked = TRUE;
              self->dst_chunked = TRUE;
            }
          else if (to == EP_CLIENT && expect_data)
            {
              if (owner->max_body_length && self->content_length != HTTP_LENGTH_UNKNOWN && 
                  (guint) self->content_length > owner->max_body_length)
                {
                  self->content_length = owner->max_body_length;
                  self->force_nonpersistent_mode = TRUE;
                }
              if (owner->proto_version[to] > 0x0100 && owner->proto_version[from] > 0x0100)
                {
                  /* The original entity is not chunked, we enchunk it to avoid bufferring and 
                   * keeping the connection open.
                   * Transfer-Encoding is added, Content-Length is removed.
                   */
                  self->src_chunked = FALSE;
                  self->dst_chunked = TRUE;
                }
              else
                {
                  /* chunking is not supported, the entity's end is indicated by an EOF
                   * neither Content-Length nor Transfer-Encoding is added, persisency
                   * is retained only if a content-length hint is received.
                   */
                  if (owner->proto_version[from] >= 0x0100)                    
                    self->persistent_with_cl_hint_only = TRUE;
                  else
                    self->force_nonpersistent_mode = TRUE;
                  self->src_chunked = self->dst_chunked = FALSE;

                }
            }
          else
            {
              /* sending to the server, the client sends it unchunked, but we add chunking to avoid buffering */
              
              /* NOTE: some servers do not accept chunked data for methods other
               * than POST, but those do not usually have entity either */
              
              if (self->content_length >= 0 || (expect_data && self->content_length == HTTP_LENGTH_UNKNOWN))
                {
                  self->src_chunked = FALSE;
                  self->dst_chunked = TRUE;
                }
              else
                {
                  self->content_length = HTTP_LENGTH_NONE;
                }
            }
          
        }
      else
        {
          /* there's no stacked proxy */
          if (chunked)
            {
              self->src_chunked = TRUE;
              self->dst_chunked = TRUE;
            }
          else if (self->content_length >= 0)
            {
              /* entity with specified length */
              if (owner->max_body_length && self->content_length != HTTP_LENGTH_UNKNOWN && 
                  (guint) self->content_length > owner->max_body_length)
                {
                  self->content_length = owner->max_body_length;
                  self->force_nonpersistent_mode = TRUE;
                }
            }
          else if (self->content_length == HTTP_LENGTH_UNKNOWN)
            {
              /* EOF terminated entity, can only happen for server->client direction */
              
              g_assert(from == EP_SERVER);
              if (owner->keep_persistent && owner->proto_version[to] > 0x0100)
                {
                  /* client supports chunking, convert it */
                  self->src_chunked = FALSE;
                  self->dst_chunked = TRUE;
                  owner->server_connection_mode = HTTP_CONNECTION_CLOSE;
                }
              else if (owner->connection_mode == HTTP_CONNECTION_KEEPALIVE)
                {
                  /* client does not support chunking or we are not in keep_persistent mode, 
                   * no way to keep it persistent */
                  g_string_assign(owner->connection_hdr->value, "close");
                  owner->connection_hdr->present = TRUE;
                  owner->connection_mode = HTTP_CONNECTION_CLOSE;
                }
            }
        }
      /* NOTE: the headers here are not final, if a content-length hint is received, the
       * content-length header will be added and transfer-encoding removed */
      if (self->dst_chunked)
        {
          self->content_length_hdr->present = FALSE;
          self->transfer_encoding_hdr->present = TRUE;
          g_string_assign(self->transfer_encoding_hdr->value, "chunked");
        }
      else if (self->content_length >= 0)
        {
          self->transfer_encoding_hdr->present = FALSE;
          g_string_sprintf(self->content_length_hdr->value, "%d", self->content_length);
          self->content_length_hdr->present = TRUE;
          
        }
    }
  else
    {
      self->content_length = HTTP_LENGTH_NONE;
    }
  format_preamble((HttpProxy *) self->super.owner, FALSE, self->preamble);
  return self;
}

static gboolean
http_transfer_run(HttpTransfer *self)
{
  GError *local_error = NULL;
  
  if (self->content_length != HTTP_LENGTH_NONE && self->content_length != 0)
    {
      ZStackedProxy *stacked;
      
      if (self->stack_type != HTTP_STK_NONE)
        {
          z_policy_lock(self->super.owner->thread);
          stacked = z_proxy_stack_object(self->super.owner, self->stack_object);
          z_policy_unlock(self->super.owner->thread);
          if (stacked)
            {
              switch (self->stack_type)
                {
                case HTTP_STK_NONE:
                  break;
                case HTTP_STK_MIME:
                  self->push_mime_headers = TRUE;
                  /* fallthrough */
                case HTTP_STK_DATA:
                  z_transfer_set_stacked_proxy(&self->super, stacked);
                  break;
                }
            }
        }
      return z_transfer_run(&self->super);
    }
  if (z_transfer_src_shutdown(&self->super, self->super.endpoints[EP_CLIENT], SHUT_RD, &local_error) != G_IO_STATUS_NORMAL)
    {
      return FALSE;
    }
  if (z_transfer_dst_shutdown(&self->super, self->super.endpoints[EP_SERVER], SHUT_WR, &local_error) != G_IO_STATUS_NORMAL)
    {
      return FALSE;
    }
  return TRUE;
}

static void
http_transfer_free_method(ZObject *s)
{
  HttpTransfer *self = Z_CAST(s, HttpTransfer);
  
  if (self->stack_object)
    {
      z_policy_var_unref(self->stack_object);
    }
  g_string_free(self->preamble, TRUE);
  g_string_free(self->stacked_preamble, TRUE);
  z_transfer_free_method(s);
}

ZTransferFuncs http_transfer_funcs =
{
  {
    Z_FUNCS_COUNT(ZTransfer),
    http_transfer_free_method,
  },
  http_transfer_src_read,
  NULL,
  http_transfer_src_shutdown,
  NULL,
  http_transfer_dst_write,
  http_transfer_dst_shutdown,
  NULL
};

ZClass HttpTransfer__class =
{
  Z_CLASS_HEADER,
  &ZTransfer__class,
  "HttpTransfer",
  sizeof(HttpTransfer),
  &http_transfer_funcs.super
};



gboolean
http_data_transfer(HttpProxy *self, guint from, ZStream *from_stream, guint to, ZStream *to_stream, gboolean expect_data, gboolean suppress_data, HttpTransferPreambleFunc format_preamble)
{
  HttpTransfer *t;
  gboolean res = TRUE;


  self->content_length_hint_set = FALSE;
  self->stack_decision = Z_ACCEPT;
  /*
   * tell transfer not to send the preamble when data is expected (e.g.
   * response & response body). This makes it possible to display an error
   * page instead 
   */
  
  t = http_transfer_new(self, from, from_stream, to, to_stream, expect_data, suppress_data, format_preamble);
  if (!t)
    {
      /*LOG
        This message indicates that the processed request was invalid, and
	the data transfer failed.
       */
      z_proxy_log(self, HTTP_ERROR, 2, "Invalid request, data transfer failed;");
      g_string_assign(self->error_info, "Invalid request, data transfer failed;");
      return FALSE;
    }
  res = http_transfer_run(t);
  if (!res)
    {
      /* transfer was not successful */
      /*LOG
        This message reports that the data transfer failed.
       */
      z_proxy_log(self, HTTP_ERROR, 2, "Data transfer failed;");
      g_string_assign(self->error_info, "Data transfer failed.");
    }
  else
    {
      /* transfer was successful, check if the stacked proxy told us something important */
      switch (self->stack_decision)
        {
        case Z_REJECT:
	  /*LOG
	    This message indicates that the stacked proxy rejected the
	    content. Check the stacked proxy log for further information.
	   */
          z_proxy_log(self, HTTP_ERROR, 2, "Stacked proxy rejected contents; info='%s'", self->stack_info->str);
          g_string_assign(self->error_info, self->stack_info->str);
          break;
        default:
          /* nothing was received */
          break;
        }
    }

  if (t->super.status & ZTS_HTTP_TRANSFER_DELAYED)
    {
      /* transfer was delayed, we can _AND_ have to display an error page */
      self->error_code = HTTP_MSG_BAD_CONTENT;
      res = FALSE;
    }
  else if (!res)
    {
      /* unable to write error page */
      self->error_code = 0;
    }

  z_object_unref(&t->super.super);
  return res;
}

