############################################################################
##
## Copyright (c) 2000, 2001, 2002 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: Auth.py,v 1.69 2004/05/07 12:57:53 bazsi Exp $
##
## Author  : Bazsi
## Auditor : kisza
## Last audited version: 1.3
## Notes:
##
############################################################################

import Zorp
from Zorp import *
from Cache import TimedCache

import types

Z_AUTH_UNKNOWN    = 0
Z_AUTH_GETMETHODS = 1
Z_AUTH_METHODS    = 2
Z_AUTH_SETMETHOD  = 3
Z_AUTH_REQUEST    = 4
Z_AUTH_RESPONSE   = 5
Z_AUTH_ACCEPT     = 6
Z_AUTH_ABORT      = 7
Z_AUTH_REJECT     = 8

Globals.auth_policies[None] = None
Globals.auth_caches[None] = None

class AuthCache:
	
	def __init__(self, name=None, timeout=600, update_stamp=TRUE, service_equiv=FALSE, cleanup_threshold=100):
		"""
		  name            -- [QSTRING]
		  timeout         -- [INTEGER]
		  update_stamp    -- [BOOLEAN]
		  service_equiv   -- [BOOLEAN]
		  cleanup_threshold -- [INTEGER]
		"""
		self.name = name
		self.cache = TimedCache('authcache(%s)' % name, timeout, update_stamp, cleanup_threshold)
		self.service_equiv = service_equiv
		if name:
			if Globals.auth_caches.has_key(name):
				raise ValueError, "Duplicate AuthCache name: %s" % name
			Globals.auth_caches[name] = self
		
	
	def makeKey(self, session):
		if self.service_equiv:
			return session.client_address.ip_s
		else:
			return (session.client_address.ip_s, session.service.name)

	def lookup(self, session):
		return self.cache.lookup(self.makeKey(session))
	
	def store(self, session, entity):
		return self.cache.store(self.makeKey(session), entity)

class AbstractAuthentication:
	"""Abstract class encapsulating interface different authentication means.
	
	This class encapsulates interfaces for inband and outband
	authentication procedures. You should not use this class directly,
	use 'InbandAuthentication' or 'SatyrAuthentication'.
	
	Attributes
	
	  authentication_provider -- authentication provider to authenticate against
	"""
	def __init__(self, authentication_provider, auth_cache=None):
		"""Constructor to initialize an AbstractAuthentication instance.
		
		This constructor initializes an 'AbstractAuthentication' instance
		by storing arguments as attributes.
		
		Arguments
		
		  self -- this instance
		  
		  authentication_provider -- [CLASS_authprov] The authentication
		                             provider class
		                             
		  auth_cache              -- [INST_authcache] Deprecated: The
					     authentication Cache Object

		"""

		if type(auth_cache) == types.StringType:
	                try:
                		self.cache = Globals.auth_caches[auth_cache]
	                except KeyError:
        	                raise AuthException, 'Authentication cache not found.'
		else:
                	self.cache = auth_cache

		try:
			self.authentication_provider = Globals.authentication_pv[authentication_provider]
		except KeyError:
			raise AuthException, 'Authentication db not found.'

	def performAuth(self, proxy):
		raise AuthException, 'Authentication not implemented'
		
		
	def performOutbandAuth(self, proxy):
		"""Function called to perform outband authentication before the session is started.
		
		This function is called to perform outband authentication
		before a session is started. It should raise AuthException
		if the authentication was not successful.
		
		Arguments
		
		  self -- this instance
		
		  session -- the session object which is to be started
		"""
		raise AuthException, 'Outband authentication not implemented'
		
	def performInbandAuth(self, proxy):
		"""Function called to enable inband authentication in a session.
		
		This function is called when a proxy was started, and it should
		enable inband authentication in the proxy, given it has support
		to perform it.
		
		Arguments
		
		  self -- this instance
		  
		  proxy -- proxy instance to enable authentication in
		"""
		raise AuthException, 'Inband authentication not implemented'


class InbandAuthentication(AbstractAuthentication):
	"""Class encapsulating inband authentication.
	
	This class encapsulates inband authentication. Instances
	of this class can be used as the 'auth' parameter for 
	Service definitions.
	
	Example
	
	  r = TISAuthorization(SockAddrInet('192.168.1.1', 7777))
	  Service("intra_http", HttpProxy,
	          auth=InbandAuthentication(r))
	
	"""

	def __init__(self, authentication_provider, auth_cache=None):
		"""Constructor to initialize an AbstractAuthentication instance.

		This constructor initializes an 'AbstractAuthentication' instance
		by storing arguments as attributes.

		Arguments

		  self -- this instance

		  authentication_provider -- [CLASS_authprov] The authentication
		                              provider class

		  auth_cache              -- [INST_authcache] Deprecated: The
					     authentication Cache Object
		"""
		AbstractAuthentication.__init__(self, authentication_provider, auth_cache)

	def performAuth(self, proxy):
		return self.performInbandAuth(proxy)

	def performInbandAuth(self, proxy):
		if hasattr(proxy, 'auth_inband_supported') and proxy.auth_inband_supported:
			proxy.auth = self.authentication_provider
#			self.authentication_provider.startSession(proxy.session.session_id, proxy.session)
		else:
			raise AuthException, 'Inband authentication not supported by the underlying proxy'
		return TRUE


Globals.auth_policies[None] = None

class AuthPolicy:
	"""Class encapsulating an authentication and authorization policy.

	This class encapsulates authentication and authorization policy,
	that is it contains an authentication method (for example
	SatyrAuthentication), and an authorization list (for example
	BasicAuthorization).
		
	Arguments
		
	  self           -- this instance

	  name           -- [QSTRING] Name of this AuthPolicy instance

	  authentication -- [INST_auth] An authentication instance

	"""

	def __init__(self, name, authentication=None, authorization=None, auth_cache=None):
		self.name = name
		self.authentication = authentication
		self.auth_cache = auth_cache
		
		# FIXME: this is a hack for compatibility and might be removed as soon
		# as SatyrAuthentication.cache is removed.
		if not auth_cache and hasattr(authentication, 'cache'):
			self.auth_cache = authentication.cache
		if name:
			if Globals.auth_policies.has_key(name):
				raise ValueError, "Duplicate AuthPolicy name: %s" % name
			Globals.auth_policies[name] = self

	def performAuth(self, proxy):
		entity = None
		if self.auth_cache:
			try:
				entity = self.auth_cache.lookup(proxy.session)
			except KeyError:
				pass
		if not entity:
			res = self.authentication.performAuth(proxy)
		else:
			res = TRUE
			proxy.userAuthenticated(entity[0], entity[1], 'cached')

		if res:
			entity = (proxy.session.auth_user, proxy.session.auth_groups)
			if self.auth_cache:
				self.auth_cache.store(proxy.session, entity)
		return res

	def isAuthorized(self, session):
		return TRUE

