############################################################################
##
## Copyright (c) 2000, 2001, 2002 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: AuthDB.py,v 1.5 2004/02/06 17:29:49 sasa Exp $
##
## Author  : Bazsi
## Auditor : kisza
## Last audited version: 1.3
## Notes:
##
############################################################################


import Zorp
from Zorp import *

class AbstractAuthenticationProvider:
	"""Class encapsulating an authentication provider.
	
	This is an abstract class to encapsulate an authentication provider,
	which is responsible for checking authentication credentials based
	on a backend database. 

	Authentication methods (Satyr or Inband) use this class to verify
	credentials.
	
	You shouldn't need to instantiate this class, use a derived one
	instead.
	
	Attributes
	
	  methods  -- set of allowed methods set by the administrator
	"""
	def __init__(self, name):
		"""Constructor to initialize an 'AbstractAuthorization' instance.
		
		This constructor initializes an 'AbstractAuthorization' instance
		by storing its methods argument as an attribute.
		
		Arguments
		
		  self -- this instance
		
		  name -- [QSTRING]
		"""

		if Globals.authentication_pv.has_key(name):
			raise AuthException, "Duplicate authorization db"
		Globals.authentication_pv[name] = self
		
	def startSession(self, session_id, session):
		"""Method to be called when an authentication session starts.
		
		This method is called when an authentication session
		identified by 'session_id' starts. 'session_id' can be used
		to associate data with this session, as each subsequent
		calls to AbstractAuthorization methods will get this value.
		
		Arguments
		
		  self -- this instance
		  
		  session_id -- session identifier represented as a string
		"""
		pass
	
	def stopSession(self, session_id):
		"""Method to be called when an authentication session ends.
		
		This method is called when an authentication session is ended.
		It's a placeholder for freeing up any resources associated to
		a given session.
		"""
		pass
	
	def getMethods(self, session_id, entity):
		"""Function returning the allowed set of methods.
		
		This function calculates and returns the set of allowed methods
		a user is allowed to authenticate with. We return an empty
		set here, overridden methods should return something more
		interesting.
		
		Arguments
		
		  self -- this instance
		  
		  session_id -- authentication session id
		  
		  entity -- username
		  
		Returns
		
		  return a tuple. First value is Z_AUTH_*, the second is a
		  array of applicable methods. (if any)
		"""
		return ()
	
	def setMethod(self, session_id, method):
		"""Function to set the authentication Method.
		
		This function should return a challenge for a given entity
		using the given method, or None if challenge is not
		applicable for the given method.
		
		Arguments
		
		  self -- this instance
		  
		  session_id -- authentication session id
		  
		  entity -- username
		  
		  method -- authentication method
		  
		Returns
		
		  return a tuple. First value is one of Z_AUTH*, second
		  value is a string containing the challenge, or None if not applicable
		"""
		raise NotImplementedError
	
	def converse(self, session_id, credentials):
		"""Function checking the presented credentials of an entity.
		
		This function is called to check the credentials presented
		by the client for validity. It should return either TRUE, if
		the credentials for the given challenge & method & username
		are valid.
		
		Arguments
		
		  self -- this instance
		  
		  session_id -- authentication session identifier
		  
		  entity -- username
		  
		  challenge -- a previously issued challenge (might be None or an empty string)
		  
		  credentials -- response for the given challenge

		Returns

		  return a tuple. First value is one of Z_AUTH_*, second is
		  depending on the first.
		"""
		raise NotImplementedError

