############################################################################
##
## Copyright (c) 2000, 2001, 2002 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: Cache.py,v 1.6 2004/01/05 10:54:27 bazsi Exp $
##
## Author  : Bazsi
## Auditor : 
## Last audited version:
## Notes:
##
############################################################################

"""
General cache management module. Used internally by the policy layer.
"""

from Zorp import *

import time

class AbstractCache:
	def __init__(self, name):
		self.name = name

	def store(self, key, value):
		pass
		
	def lookup(self, key):
		pass
	
	def clear(self):
		pass

class ShiftCache(AbstractCache):
	def __init__(self, name, shift_threshold):
		AbstractCache.__init__(self, name)
		self.cache = {}
		self.old_cache = {}
		self.shift_threshold = shift_threshold
	
	def store(self, key, value):
		if len(self.cache) > self.shift_threshold:
			## LOG ##
			# This message indicates that the cache size(threshold) is reached, and cache is shifted.
			# @see: Cache.ShiftCache
			##
			log(None, CORE_MESSAGE, 3, "Cache over shift-threshold, shifting; cache='%s', threshold='%d'", (self.name, self.shift_threshold,))
			self.old_cache = self.cache
			self.cache = {}
		self.cache[key] = value
		try:
			del self.old_cache[key]
		except KeyError:
			pass
			
	def lookup(self, key):
		val = None
		try:
			return self.cache[key]
		except KeyError:
			pass
		
		val = self.old_cache[key]
		self.cache[key] = val
		del self.old_cache[key]
		return val
	
	def clear(self):
		self.cache = {}
		self.old_cache = {}


class TimedCache(AbstractCache):
	def __init__(self, name, timeout, update_stamp=TRUE, cleanup_threshold=100):
		AbstractCache.__init__(self, name)
		self.timeout = timeout
		self.update_stamp = update_stamp
		self.cleanup_threshold = cleanup_threshold
		self.cache = {}

	def cleanup(self):
		now = time.time()
		for x in self.cache.keys():
			if now - self.cache[x][0] > self.timeout:
				del self.cache[x]

	def lookup(self, key):
		if len(self.cache) > self.cleanup_threshold:
			self.cleanup()
		if self.cache.has_key(key):
			entry = self.cache[key]
			if time.time() - entry[0] > self.timeout:
				del self.cache[key]
			else:
				if self.update_stamp:
					entry[0] = time.time()
				return entry[1]
		return None
	
	def store(self, key, value):
		self.cache[key] = [time.time(), value]

	def clear(self):
		self.cache = {}

