############################################################################
##
## Copyright (c) 2000, 2001, 2002 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: Chainer.py,v 1.95 2004/05/07 14:23:56 bazsi Exp $
##
## Author  : Bazsi
## Auditor : kisza
## Last audited version: 1.3
## Notes:
##
############################################################################

"""Module defining the chainer interface for chaining up to parent proxies.

This module defines the 'AbstractChainer' interface, whose implementations
are responsible to connect a child proxy to its parent.
"""

from Zorp import *
from Session import MasterSession
from Attach import Attacher
from Stream import Stream
from Session import StackedSession
from SockAddr import SockAddrInet
from NAT import NAT_SNAT, NAT_DNAT
from Cache import TimedCache
import types

class AbstractChainer:
	"""Abstract class encapsulating the chaining process.
	
	This abstract class encapsulates the process of chaining to
	the parent proxy.
	"""
	def __init__(self):
		pass

	def setupFastpath(self, proxy):
		"""Function called to setup fast path handlers.
		
		This function is called to provide the C-level fast path
		handlers to the proxy, which are called when subsequent
		connections are added to the proxy (currently only used
		for UDP proxies)
		"""

		pass

	def chainParent(self, session):
		"""Function to be called when a proxy wants to connect to its parent.
		
		This function is called to actually perform chaining to the parent.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session we belong to
		"""
		raise NotImplementedError

class ConnectChainer(AbstractChainer):
	"""Class encapsulating a connection establishment.
	
	This class encapsulates a real TCP/IP connection establishment, and
	is used when a top-level proxy wants to perform chaining.
	
	When no explicit top-level chainer is specified for services, this
	one is used by default.	
	"""
	
	def __init__(self, protocol=ZD_PROTO_AUTO):
		"""
		   
		Arguments

		  protocol  -- [ENUM;ZD_PROTO] Type of Connecting method
		"""
		AbstractChainer.__init__(self)
		self.protocol = protocol

	def setupFastpath(self, proxy):
		protocol = self.protocol
		if protocol == 0:
			protocol = proxy.session.protocol
		setupConnectChainer(proxy, proxy.session.session_id, protocol)

	def establishConnection(self, session, local, remote):
		"""Function to actually establish a connection.
		
		Internal function to establish a connection with the given
		local and remote addresses. It is used by derived chainer
		classes after finding out the destination address to connect
		to. This function performs access control checks.
		   
		Arguments

		  self         -- this instance
		  
		  session      -- Session object
		  
		  local	       -- bind address
		  
		  remote       -- host to connect to

                  protocol     -- protocol to connect to
		  
		Returns
		
		  The stream of the connection to the server

		"""
		protocol = self.protocol
		if protocol == 0:
			protocol = session.protocol
		if remote.port == 0:
			remote.port = session.client_local.port
		session.setServer(remote)
		if session.isServerPermitted() == Z_ACCEPT:
			#remote.options = session.client_address.options
			try:
				conn = Attacher(session.session_id, protocol, local, remote, local_loose=session.server_local_loose, tos=session.server_tos)
				session.server_stream = conn.block()
				session.server_local = conn.local
			except IOError:
				session.server_stream = None
			if session.server_stream == None:
				## LOG ##
				# This message indicates that the connection to the server failed.
				##
				log(session.session_id, CORE_SESSION, 3, 
                                    "Server connection failure; server_address='%s', server_zone='%s', server_local='%s', server_protocol='%s'",
                                    (session.server_address, session.server_zone, session.server_local, session.protocol_name))
			else:
				session.server_stream.name = session.session_id + "/server"
				## LOG ##
				# This message indicates that the connection to the server succeeded.
				##
				log(session.session_id, CORE_SESSION, 3, 
				    "Server connection established; server_fd='%d', server_address='%s', server_zone='%s', server_local='%s', server_protocol='%s'",
				    (session.server_stream.fd, session.server_address, session.server_zone, session.server_local, session.protocol_name))
			return session.server_stream
		raise DACException

	def chainParent(self, session):
		"""Overridden function to perform connection establishment.
		
		This function is called by the actual Proxy implementation
		to actually connect to the server-endpoint of the session. 
		The destination address is 'session.server_address' (which
		is previously set by the Router used for the service, and
		optionally overridden by the Proxy). The local address to
		bind to is determined with the help of a NAT object if one
		is provided, or allocated dynamically by the kernel.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session we belong to
		
		"""
		if session.server_local:
			local = session.server_local.clone(0)
		else:
			local = None

		if session.server_address == None:
			## LOG ##
			# This message indicates that the connection to the
			# server can not be established, because no server
			# address is set.
			##
			log(session.session_id, CORE_SESSION, 3, "Server connection failure, no destination;")
			return None
		elif type(session.server_address) == types.TupleType or type(session.server_address) == types.ListType:
			## LOG ##
			# This message indicates that the connection to the
			# server can not be established, the routing
			# meachnism specified several destination addresses
			# and ConnectChainer only supports a single address.
			##
			log(session.session_id, CORE_SESSION, 3, "Server connection failure, multiple destination addresses;")
			return None
		remote = session.server_address.clone(0)

		if session.service.snat_policy:
			local = session.service.snat_policy.performTranslation(local, NAT_SNAT)
		elif session.service.snat:
			if not local:
				## LOG ##
				# This message indicates that SNAT is being set, but there is no server side local address to use for SNAT.
				##
				log(session.session_id, CORE_ERROR, 3, "SNAT is present and local address is None;")
				return None
			local = session.service.snat.performTranslation(session, local, NAT_SNAT)

		if session.service.dnat_policy:
			remote = session.service.dnat_policy.performTranslation(remote, NAT_DNAT)
		elif session.service.dnat:
			remote = session.service.dnat.performTranslation(session, remote, NAT_DNAT)

		return self.establishConnection(session, local, remote)

class FailoverChainer(ConnectChainer):
	"""Class encapsulating a connection establishment with failover capability.
	
	This class encapsulates a real TCP/IP connection establishment, and
	is used when a top-level proxy wants to perform chaining. In 
        addition to ConnectChainer this class adds the capability to perform
	failover, given a destination is not working.

	Attributes

	  current_host -- index of the last attempted destination
	"""
	def __init__(self, protocol=ZD_PROTO_TCP, timeout=0):
		"""Constructor to initialize a FailoverChainer class.

		This constructor initializes a FailoverChainer class by
		filling attributes with appropriate values and calling the
		inherited constructor.

		Arguments

		  self -- this instance

		  protocol  -- [ENUM;ZD_PROTO] Type of Connecting method
		  
		  timeout   -- [INTEGER] the down state of remote hosts is kept for this interval in seconds
		"""
		ConnectChainer.__init__(self, protocol)
		if timeout:
			self.state = TimedCache('failover-state', timeout, update_stamp=FALSE)
		else:
			self.state = None
		self.current_host = 0

	def chainParent(self, session):
		"""Overridden function to perform connection establishment.
		
		This function is called by the actual Proxy implementation
		to actually connect to the server-endpoint of the session. 
		The destination address is 'session.server_address' (which
		is previously set by the Router used for the service, and
		optionally overridden by the Proxy). The local address to
		bind to is determined with the help of a NAT object if one
		is provided, or allocated dynamically by the kernel.

		The failover capability of FailoverChainer is implemented
		here.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session we belong to
		
		"""
		if session.server_local:
			local = session.server_local.clone(0)
		else:
			local = None
		hostlist = session.server_address
		if type(hostlist) != types.TupleType and type(hostlist) != types.ListType:
			return ConnectChainer.chainParent(self, session)

		if len(hostlist) == 1:
			session.server_address = hostlist[0]
			return ConnectChainer.chainParent(self, session)

		if session.service.snat:
			local = session.service.snat.performTranslation(session, local, NAT_SNAT)

		stream = None
		try_count = 0
		while (try_count < 2)  and (stream == None):
			first_attempt = self.current_host % len(hostlist)
			remote = None
	 		while stream == None and (remote == None or (self.current_host != first_attempt)):
				remote = hostlist[self.current_host].clone(0)
				self.current_host = (self.current_host + 1) % len(hostlist)

				if session.service.dnat:
					remote = session.service.dnat.performTranslation(session, remote, NAT_DNAT)

				if self.state:
					is_host_down = self.state.lookup(remote.ip_s)
				else:
					is_host_down = 0
				if not is_host_down:
					stream = self.establishConnection(session, local, remote)
					if not stream and self.state:
						## LOG ##
						# This message reports that the remote end is down and Zorp stores the 
						# down state of the remote end, so Zorp wont try to connect to it within the 
						# timeout latter.
						##
						log(None, CORE_MESSAGE, 4, "Destination is down, keeping state; remote='%s'", (remote,))
						self.state.store(remote.ip_s, 1)
				else:
					## LOG ##
					# This message reports that the remote end is down, but Zorp does not store the
					# down state of the remote end, so Zorp will try to connect to it next time.
					##
					log(None, CORE_MESSAGE, 4, "Destination is down, skipping; remote='%s'", (remote,))
			if (stream == None) and (try_count == 0):
				log(None, CORE_MESSAGE, 4, "All destinations are down, clearing cache and trying again;")
				self.state.clear()
			try_count = try_count + 1
		return stream


class SideStackChainer(AbstractChainer):
	"""Class encapsulating proxy chaining side-by-side.

	This class encapsulates a special case of chaining. Instead of 
	establishing a connection to some kind of server it creates
	another Proxy instance and connects the server side of the current
	and the client side of the new Proxy to each other.

	Attributes

	  right_class -- proxy class to side-stack

	  right_chainer -- chainer used by right_class to chain further

	"""
	def __init__(self, right_class, right_chainer = None):
		"""Constructor to initialize a SideStackChainer class.

		This constructor performs initialization of the
		SideStackChainer class.

		Arguments

		  right_class -- [CLASS_proxy] proxy class to side-stack

		  right_chainer -- [INST_chainer] chainer used by right_class to chain further
		"""
		AbstractChainer.__init__(self)
		self.right_class = right_class
		if right_chainer == None:
			right_chainer = ConnectChainer()
		self.right_chainer = right_chainer

	def chainParent(self, session):
		"""Overridden function to perform chaining.

		This function is called by a Proxy instance to establish its
		server side connection. Instead of connecting to a server
		this chainer creates another proxy instance and connects
		this new proxy with the current one.

		Arguments
		
		  self -- this instance
		  
		  session -- session we belong to
		
		"""
		try:
			streams = streamPair(AF_UNIX, SOCK_STREAM)
		except IOError:
			## LOG ##
			# This message indicates that side stacking failed, because Zorp was unable to create a socketPair.
			# It is likely that there is now resource available. Try increase fd limits.
			##
			log(session.session_id, CORE_SESSION, 3, "Side stacking failed, socketPair failed;")
			return None

		try:
			# convert our tuple to an array, to make it possible
			# to modify items
			streams = [streams[0], streams[1]]
			ss = None

			session.server_stream = streams[0]
			session.server_stream.name = session.owner.session_id + "/leftside"
			ss = StackedSession(session, self.right_chainer)
			streams[0] = None
			ss.client_stream = streams[1]
			ss.client_stream.name = ss.session_id + "/rightside"
			ss.server_stream = None
			streams[1] = None
			## LOG ##
			# This message indicates that side stacking was successful.
			##
			log(session.session_id, CORE_SESSION, 4, 
                                "Side-stacking proxy instance; server_fd='%d', client_fd='%d', proxy_class='%s'",
                                (session.server_stream.fd, ss.client_stream.fd, self.right_class.__name__))
			self.right_class(ss)
		except:
			## LOG ##
			# This message indicates that side stacking failed. 
			##
			log(session.session_id, CORE_ERROR, 3, "Side-stacking failed; proxy_class='%s'", (self.right_class.__name__))
			if ss:
				ss.destroy()
			if (streams[0] != None):
				streams[0].close()
			if (streams[1] != None):
				streams[1].close()
