/***************************************************************************
 *
 * Copyright (c) 2000, 2001 BalaBit IT Ltd, Budapest, Hungary
 * All rights reserved.
 *
 * $Id: main.c,v 1.171 2004/09/03 12:16:02 bazsi Exp $
 *
 * Author  : bazsi
 * Auditor : kisza
 * Last version : 1.24
 * Notes   : 
 *
 ***************************************************************************/

#include <zorp/zorp.h>
#include <zorp/io.h>
#include <zorp/stream.h>
#include <zorp/policy.h>
#include <zorp/registry.h>
#include <zorp/thread.h>
#include <zorp/log.h>
#include <zorp/proxyvars.h>
#include <zorp/cap.h>
#include <zorp/ssl.h>
#include <zorp/sysdep.h>
#include <zorp/poll.h>
#include <zorp/szig.h>
#include <zorp/tpsocket.h>
#include <zorp/dispatch.h>
#include <zorp/process.h>
#include <zorp/blob.h>

#include <zorp/conntrack.h>
#include <zorp/stackdump.h>

#include <sys/types.h>
#include <sys/resource.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include <sys/wait.h>

#include <sys/time.h>
#include <unistd.h>
                     
#include <sys/ioctl.h>

#include <popt.h>

#ifdef USE_DMALLOC
#include <dmalloc.h>
#endif

#if HAVE_SYS_PRCTL_H
#include <sys/prctl.h>
#endif

#include <sys/termios.h>

#include "logtags.gperf.c"

static gint 
z_logtag_lookup(const gchar *tag, gsize len)
{
  struct tagid *p = NULL;
  
  p = z_logtag_lookup_gperf(tag, len);
  if (G_LIKELY(p))
    return p->id;
  else
    return -1;
}

void
z_sigterm_handler(int signo G_GNUC_UNUSED)
{
  z_main_loop_initiate_termination(TRUE);
}

void 
z_ignore_signal_handler(int signo G_GNUC_UNUSED)
{
}


void
z_sigchild_handler(int signo G_GNUC_UNUSED)
{
  while (waitpid(-1, NULL, WNOHANG) > 0)
    ;
}

void
z_fatal_signal_handler(int signo)
{
  ZSignalContext *p = z_stackdump_get_context(p);
  struct sigaction act;

  memset(&act, 0, sizeof(act));
  act.sa_handler = SIG_DFL;
  sigaction(signo, &act, NULL);

  /*LOG
    This message is logged when Zorp caught a fatal signal.
    Possible reason is bad RAM or other hardware.
   */
  z_log(NULL, CORE_ERROR, 0, "Signal received, stackdump follows; signo='%d'", signo);
  z_mem_trace_stats();
  z_stackdump_log(p);
  kill(getpid(), signo);
}

void
z_sigusr1_handler(int signo G_GNUC_UNUSED)
{
  usr1_received = 1;
}

void
z_sigusr2_handler(int signo G_GNUC_UNUSED)
{
  usr2_received = 1;
}

void
z_write_pid_file(const gchar *fname)
{
  FILE *fd;
  
  fd = fopen(fname, "w");
  if (fd) 
    {
      fprintf(fd, "%d\n", getpid());
      fclose(fd);
    }
  else
    {
      fprintf(stderr, "Error opening pidfile; fname='%s', error='%s'\n", fname, g_strerror(errno));
    }
}

void
z_remove_pid_file(const gchar *fname)
{
  if (unlink(fname) == -1)
    /*LOG
      This message indicates that a stale pidfile was found, but Zorp is unable to remove it.
     */
    z_log(NULL, CORE_ERROR, 3, "Error removing pidfile; fname='%s', error='%s'", fname, g_strerror(errno));
}

void
z_setup_signals(void)
{
  struct sigaction  act;
  sigset_t          ss;
  
  sigemptyset(&ss);


  
  memset(&act, 0, sizeof(act));
  act.sa_handler = z_ignore_signal_handler;
  sigaction(SIGPIPE, &act, NULL);
  sigaddset(&ss, SIGPIPE);
  sigaction(SIGINT, &act, NULL);
  sigaddset(&ss, SIGINT);
  sigaction(SIGTERM, &act, NULL); 
  sigaddset(&ss, SIGTERM);

  memset(&act, 0, sizeof(act));
  act.sa_handler = z_sigterm_handler;
  sigaction(SIGINT, &act, NULL);
  sigaction(SIGTERM, &act, NULL); 
  
  memset(&act, 0, sizeof(act));
  act.sa_handler = z_fatal_signal_handler;
  sigaction(SIGSEGV, &act, NULL);
  sigaddset(&ss, SIGSEGV);
  sigaction(SIGABRT, &act, NULL);
  sigaddset(&ss, SIGABRT);
  sigaction(SIGILL, &act, NULL);
  sigaddset(&ss, SIGILL);
  
  memset(&act, 0, sizeof(act));
  act.sa_handler = z_ignore_signal_handler;
  sigaction(SIGTRAP, &act, NULL);
  sigaddset(&ss, SIGTRAP);

  memset(&act, 0, sizeof(act));
  act.sa_handler = z_sigchild_handler;
  sigaction(SIGCHLD, &act, NULL);
  sigaddset(&ss, SIGCHLD);
  
  memset(&act, 0, sizeof(act));
  act.sa_handler = z_sigusr1_handler;
  sigaction(SIGUSR1, &act, NULL);
  sigaddset(&ss, SIGUSR1);
  
  memset(&act, 0, sizeof(act));
  act.sa_handler = z_sigusr2_handler;
  sigaction(SIGUSR2, &act, NULL);
  sigaddset(&ss, SIGUSR2);
  
  sigprocmask(SIG_UNBLOCK, &ss ,NULL);
}

gboolean 
z_resolve_user(const gchar *user, uid_t *uid)
{
  struct passwd *pw;
  gchar *errstr;

  *uid = 0;
  pw = getpwnam(user);
  if (pw) 
    {
      *uid = pw->pw_uid;
    }
  else 
    {
      *uid = strtol(user, &errstr, 10);
      if (!errstr || *errstr)
        {
          return FALSE;
        }
    }
  return TRUE;
}

int 
z_resolve_group(const gchar *group, gid_t *gid)
{
  struct group *gr;
  gchar *errstr;

  *gid = 0;
  gr = getgrnam(group);
  if (gr) 
    {
      *gid = gr->gr_gid;
    }
  else 
    {
      *gid = strtol(group, &errstr, 10);
      if (!errstr || *errstr)
        {
          return FALSE;
        }
    }
  return TRUE;
}

#define ON_OFF_STR(x) (x ? "on" : "off")

void
z_version(void)
{
  printf("Zorp %s\n"
         "Compile-Date: %s %s\n"
         "Config-Date: %s\n"
         "Trace: %s\n"
         "Debug: %s\n"
         "IPOptions: %s\n"
         "IPFilter-Tproxy: %s\n"
         "Netfilter-Tproxy: %s\n"
         "Netfilter-Linux22-Fallback: %s\n"
         "Linux22-Tproxy: %s\n"
         "Conntrack: %s\n\n"
         "%s",
         VERSION, __DATE__, __TIME__,
         ZORP_CONFIG_DATE,
         ON_OFF_STR(ENABLE_TRACE),
         ON_OFF_STR(ENABLE_DEBUG),
         ON_OFF_STR(ENABLE_IPOPTIONS),
         ON_OFF_STR(ENABLE_NETFILTER_TPROXY),
         ON_OFF_STR(ENABLE_NETFILTER_LINUX22_FALLBACK),
         ON_OFF_STR(ENABLE_LINUX22_TPROXY),
         ON_OFF_STR(ENABLE_CONNTRACK),
         z_zorplib_version_info());
}

#if 0
void
z_enable_core(void)
{
  struct rlimit limit;

  limit.rlim_cur = RLIM_INFINITY;
  limit.rlim_max = RLIM_INFINITY;

  chdir("/var/tmp");
  setrlimit(RLIMIT_CORE, &limit);
}
#endif

#define ZORP_OPT_ESCAPE  0x100

static struct poptOption zorp_options[] = 
{
  { "as",           'a', POPT_ARG_STRING, NULL, 'a',              "Set instance name", "<instance>" },
  { "policy",       'p', POPT_ARG_STRING, NULL, 'p',              "Set policy file", "<policy>" },
  { "verbose",      'v', POPT_ARG_INT,    NULL, 'v',              "Set verbosity level", "<verbosity>" },
  { "pidfile",      'P', POPT_ARG_STRING, NULL, 'P',              "Set path to pid file", "<pidfile>" },
  { "no-syslog",    'l', POPT_ARG_NONE,   NULL, 'l',              "Do not send messages to syslog", NULL },
  { "help",         'h', POPT_ARG_NONE,   NULL, 'h',              "Display this help", NULL },
  { "version",      'V', POPT_ARG_NONE,   NULL, 'V',              "Display version number", NULL },
  { "log-spec",     's', POPT_ARG_STRING, NULL, 's',              "Set log specification", "<logspec>" },
  { "logspec",      's', POPT_ARG_STRING, NULL, 's',              "Alias for log-spec", "<logspec>" },
  { "log-tags",     'T', POPT_ARG_NONE,   NULL, 'T',              "Enable logging of message tags", NULL },
  { "log-escape",   0  , POPT_ARG_NONE,   NULL, ZORP_OPT_ESCAPE,  "Escape log messages to avoid non-printable characters", NULL },
  { "threadpools",  'O', POPT_ARG_NONE,   NULL, 'O',              "Enable the use of threadpools", NULL },
  { "threads",      't', POPT_ARG_INT,    NULL, 't',              "Set the maximum number of threads", "<thread limit>" },
  { "idle-threads", 'I', POPT_ARG_INT,    NULL, 'I',              "Set the maximum number of idle threads (applies to threadpools only)", "<idle-threads limit>" },
  { "uid",          'u', POPT_ARG_STRING, NULL, 'u',              "Set the uid to run as", "<uid>" },
  { "gid",          'g', POPT_ARG_STRING, NULL, 'g',              "Set the gid to run as", "<gid>" },
  { "chroot",       'R', POPT_ARG_STRING, NULL, 'R',              "Chroot to this directory", "<dir>" },
  { "tproxy",       'Y', POPT_ARG_STRING, NULL, 'Y',              "Force the TPROXY implementation to use", "<netfilter|linux22|ipf>" },
  { "foreground",   'F', POPT_ARG_NONE,   NULL, 'F',              "Do not go into the background after initialization", NULL },
#if ENABLE_NETFILTER_TPROXY || ENABLE_IPFILTER_TPROXY
  { "autobind-ip",  'B', POPT_ARG_STRING, NULL, 'B',              "Set autobind-ip", "<ipaddr>" },
#endif
#if ZORPLIB_ENABLE_CAPS
  { "caps",         'C', POPT_ARG_STRING, NULL, 'C',              "Set default capability set", "<capspec>" },
  { "no-caps",      'N', POPT_ARG_NONE,   NULL, 'N',              "Disable managing Linux capabilities", NULL },
#endif
  { "stack-size",   'S', POPT_ARG_INT,    NULL, 'S',              "Set the stack size in kBytes", "<stacksize>" },
  { NULL,           0  , 0,               NULL,   0,              NULL, NULL }
};

#define MAX_SOFT_INSTANCES 128


int 
main(int argc, const char *argv[])
{
  int opt;
  const gchar *instance_name = "zorp";
  const gchar *instance_policy_list[MAX_SOFT_INSTANCES + 1];
  gint instance_count = 1;
  const gchar *policy_file = ZORP_POLICY_FILE;
  const gchar *log_spec = "";
  gchar pid_file_buf[128];
  const gchar *pid_file = NULL;
  const gchar *zorp_user = "root", *zorp_group = NULL;
  gint zorp_uid = -1, zorp_gid = -1;
  gboolean use_syslog = TRUE;
  gboolean detach_terminal = TRUE;
  gboolean log_tags = FALSE, log_escape = FALSE;
  const gchar *root_dir = NULL;
  const gchar *tproxy_impl = NULL, *verbosity = NULL;
  gint verbose_level = 3;
  gboolean foreground = FALSE;
  poptContext ctx;
  gint stack_size = 256*1024; /* Set default stack size to 256 KByte */
  struct rlimit limit;
  const gchar *license_serial;

  z_mem_trace_init(NULL);
  instance_name = "zorp";
  instance_policy_list[0] = "zorp";
  
  ctx = poptGetContext("zorp", argc, argv, zorp_options, 0);
  while ((opt = poptGetNextOpt(ctx)) > 0)
    {
      switch (opt)
	{
	case 'a':
	  instance_name = poptGetOptArg(ctx);
	  instance_policy_list[0] = instance_name;
	  instance_count = 1;
	  break;
	case 'p':
	  policy_file = poptGetOptArg(ctx);
	  break;
	case 'v':
	  if ((verbosity = poptGetOptArg(ctx)))
	    verbose_level = atoi(verbosity);
	  else
	    verbose_level++;
	  break;
	case 's':
	  log_spec = poptGetOptArg(ctx);
	  break;
        case ZORP_OPT_ESCAPE:
          log_escape = TRUE;
          break;
	case 'P':
	  pid_file = poptGetOptArg(ctx);
	  break;
	case 'h':
	  poptPrintHelp(ctx, stdout, 0);
	  return 0;
	case 'l':
	  use_syslog = FALSE;
	  foreground = TRUE;
	  detach_terminal = FALSE;
	  break;
	case 'F':
	  foreground = TRUE;
	  break;
	case 'T':
	  log_tags = 1;
	  break;
	case 'V':
	  z_version();
	  return 0;
	case 't':
	  max_threads = atoi(poptGetOptArg(ctx));
	  break;
	case 'u':
	  zorp_user = poptGetOptArg(ctx);
	  if (!z_resolve_user(zorp_user, &zorp_uid))
	    {
              fprintf(stderr, "%s: Invalid uid: %s\n", instance_name, zorp_user);
              return 1;
            }
	  break;
	case 'g':
	  zorp_group = poptGetOptArg(ctx);
	  if (!z_resolve_group(zorp_group, &zorp_gid))
	    {
	      fprintf(stderr, "%s: Invalid gid: %s\n", instance_name, zorp_group);
	      return 1;
	    }
	  break;
	case 'I':
	  idle_threads = atoi(poptGetOptArg(ctx));
	  break;
	case 'R':
	  root_dir = poptGetOptArg(ctx);
	  break;
	case 'Y':
	  tproxy_impl = poptGetOptArg(ctx);
	  break;
	case 'O':
	  use_threadpools = TRUE;
	  break;
#if ZORPLIB_ENABLE_CAPS
	case 'C':
	  zorp_caps = poptGetOptArg(ctx);
	  break;
	case 'N':
	  zorp_caps = NULL;
	  break;
#endif
#if ENABLE_NETFILTER_TPROXY|| ENABLE_IPFILTER_TPROXY
        case 'B':
          auto_bind_ip = poptGetOptArg(ctx);
          break;
#endif
	case 'S':
	  stack_size = atoi(poptGetOptArg(ctx)) * 1024;
	  break;
	}
    }
  if (opt < -1 || poptPeekArg(ctx))
    {
      fprintf(stderr, "%s: Invalid arguments.\n", instance_name);
      return 1;
    }
  poptFreeContext(ctx);
  instance_policy_list[instance_count] = NULL;

  memset(&limit, 0, sizeof(limit));
  limit.rlim_cur = stack_size;
  limit.rlim_max = stack_size;
  
  setrlimit(RLIMIT_STACK, &limit);

  if (pid_file == NULL)
    {
      g_snprintf(pid_file_buf, sizeof(pid_file_buf), "%s/zorp-%s.pid", ZORP_PID_FILE_DIR, instance_name);
      pid_file = pid_file_buf;
    }
  if (!foreground && !z_process_daemonize(instance_name, -1, -1, pid_file))
    {
      z_process_failed(1, TRUE);
    }
  else
    {
      z_write_pid_file(pid_file);
      if (detach_terminal)
        {
          /* we stay in foreground but detach ourselves from the tty */
          if (isatty(STDIN_FILENO))
            {
              ioctl(STDIN_FILENO, TIOCNOTTY, 0);
              setsid();
            }
          close(STDIN_FILENO);
        }
    }
    
  if (root_dir)
    {
      if (chroot(root_dir) < 0)
        {
          fprintf(stderr, "%s: Error in chroot(); error='%s'\n", instance_name, g_strerror(errno));
          z_process_failed(1, TRUE);
        }
    }


#if HAVE_PRCTL && HAVE_PR_SET_KEEPCAPS
  prctl(PR_SET_KEEPCAPS, 1, 0, 0, 0);
#endif
  
  if (zorp_gid != -1)
    {
      if (setgid(zorp_gid) < 0)
        {
          fprintf(stderr, "%s: Error in setgid(); error='%s'\n", instance_name, g_strerror(errno));
          z_process_failed(1, TRUE);
        }
      if (initgroups(zorp_user, zorp_gid) < 0)
        {
          fprintf(stderr, "%s: Error in initgroups(); error='%s'\n", instance_name, g_strerror(errno));
          z_process_failed(1, TRUE);
        }
    }
    
  if (zorp_uid != -1) 
    {
      if (setuid(zorp_uid) < 0)
        {
          fprintf(stderr, "%s: Error in setuid(); error='%s'\n", instance_name, g_strerror(errno));
          z_process_failed(1, TRUE);
        }
    }


#if HAVE_PRCTL && HAVE_PR_SET_DUMPABLE
  if (!prctl(PR_GET_DUMPABLE, 0, 0, 0, 0))
    {
      gint rc;
      
      rc = prctl(PR_SET_DUMPABLE, 1, 0, 0, 0);
      
      if (rc < 0)
	/*LOG
	  This message indicates that Zorp is unable to set it's state do
	  dumpable, so that no core dump is possible latter. Check under
	  what uid does zorp run.
	 */
        z_log(NULL, CORE_ERROR, 2, "Cannot set process to be dumpable; error='%s'", g_strerror(errno));
    }
#endif
    
#if ZORPLIB_ENABLE_CAPS
  if (zorp_caps) 
    {
      cap_t cap = cap_from_text(zorp_caps);
    
      if (cap == NULL)
        {
          fprintf(stderr, "%s: Error parsing capabilities: %s\n", instance_name, zorp_caps);
          z_process_failed(1, TRUE);
        }
      else
        {
          if (cap_set_proc(cap) == -1)
            {
              fprintf(stderr, "%s: Error setting capabilities; error='%s'\n", instance_name, g_strerror(errno));
            }
          cap_free(cap);
        }
    }
#endif
  
  /* NOTE: after z_log_init() we are detached from our terminal as it closes
   * the remaining stdout/stderr fds */

  z_thread_init();
  g_main_context_acquire(NULL);

  /*NOLOG*/
  if (!z_log_init(log_spec, verbose_level, instance_name, 
                  (use_syslog ? ZLF_SYSLOG : 0) |
                  (log_tags ? ZLF_TAGS : 0) |
                  ZLF_THREAD | 
                  (log_escape ? ZLF_ESCAPE : 0)))
    {
      z_log(NULL, CORE_ERROR, 0, "Error initializing logging subsystem;");
      exit_code = 1;
      goto deinit_exit;
    }

  z_log_enable_tag_map_cache(z_logtag_lookup, TOTAL_KEYWORDS);

  /*LOG
    This message reports the current verbosity level of Zorp.
   */
  z_log(NULL, CORE_DEBUG, 0, "Starting up; verbose_level='%d', version='%s'", verbose_level, VERSION);

  if (!z_sysdep_init(tproxy_impl))
    {
      /*LOG
        This message indicates that some of the system dependent subsystems
        (tproxy for example) could not be initialized.
       */
      z_log(NULL, CORE_ERROR, 0, "Error initializing system dependent subsystems;");
      exit_code = 1;
      goto deinit_exit;
    }
  z_ssl_init();
  z_szig_init(instance_name);
  if (use_syslog)
    z_log_enable_stderr_redirect(TRUE);
  
  z_dispatch_init();
  if (!z_conntrack_init())
    {
      exit_code = 1;
      goto deinit_exit;
    }
  z_registry_init();
  
  /* only used for PORT allocation within a given range */
  srand(time(NULL) ^ getpid()); 

  if (!z_python_init())
    {
      /*LOG
	This message indicates that Zorp was unable to initialize the Python engine.
	It is likely that your installation is broken. Check your packages and there version number.
       */
      z_llog(CORE_ERROR, 0, "Error initializing Python policy engine.");
      exit_code = 1;
      goto deinit_exit;
    }
  
  z_setup_signals();

  /*NOLOG*/
  z_main_loop(policy_file, instance_name, instance_policy_list);

 deinit_exit:

  /*NOLOG*/ 
  z_llog(CORE_INFO, 3, "Shutting down; version='%s'", VERSION);
  
  z_blob_system_default_destroy();
  
  z_thread_destroy();
  z_python_destroy();
  z_dispatch_destroy();
  z_conntrack_destroy();
  z_sysdep_destroy();
  z_ssl_destroy();
  z_remove_pid_file(pid_file);
  z_log_destroy();
  z_mem_trace_dump();
#ifdef USE_DMALLOC
  dmalloc_shutdown();
  /* avoid second dump of dmalloc */
  rename("logfile", "logfile.dm");
#endif
  if (exit_code != 0)
    z_process_failed(exit_code, TRUE);
  return exit_code;
}
