/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: fastpath.c,v 1.19 2004/01/05 10:54:26 bazsi Exp $
 *
 * Author  : bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/fastpath.h>
#include <zorp/proxy.h>
#include <zorp/log.h>

/**
 * z_proxy_fastpath_destroy:
 * @self: ZProxyFastpath pointer
 *
 * Destroy fastpath related data embedded into a ZProxy. Called by
 * z_proxy_destroy when the proxy finishes.
 **/
void
z_proxy_fastpath_destroy(ZProxyFastpath *self)
{
  if (self->router_data)
    self->router_free(self->router_data);
  if (self->chainer_data)
    self->chainer_free(self->chainer_data);
  if (self->snat_data)
    self->snat_free(self->snat_data);
  if (self->dnat_data)
    self->dnat_free(self->dnat_data);
}

/**
 * z_proxy_fastpath_session_destroy:
 * @self: ZProxyFastpathSession instance
 *
 * This function is called to destroy a fastpath session.
 **/
void
z_proxy_fastpath_session_destroy(ZProxyFastpathSession *self)
{
  z_sockaddr_unref(self->client_address);
  z_sockaddr_unref(self->client_local);
  z_sockaddr_unref(self->server_address);
  z_sockaddr_unref(self->server_local);
}

/**
 * z_proxy_fastpath_connect_server:
 * @proxy: ZProxy instance
 * @conn: connection description
 *
 * This function can be used by proxies to establish server-side connection
 * without touching the policy layer. It is usually called by proxies when 
 * secondary sessions are in use.
 **/
ZStream *
z_proxy_fastpath_connect_server(ZProxy *proxy, ZConnection *conn)
{
  ZProxyFastpath *self = &proxy->fastpath;
  ZStream *stream;
  gchar buf1[MAX_SOCKADDR_STRING], buf2[MAX_SOCKADDR_STRING];
  ZProxyFastpathSession session;
  
  z_proxy_enter(self);

  if (self->router == NULL ||
      self->chainer == NULL)
    {
      /*LOG
        This message attempt when proxy try to use fast path, but this
        not possible. Then it's falling back to normal path.
       */
      z_proxy_log(self, CORE_ERROR, 4, "Error starting secondary session, no fastpath chainer or router defined;");
      z_proxy_leave(self);
      return NULL;
    }
  memset(&session, 0, sizeof(session));
  
  session.client_address = z_sockaddr_clone(conn->remote, FALSE);
  session.client_local = z_sockaddr_clone(conn->dest, FALSE);
  session.server_address = NULL;
  session.server_local = NULL;
  
  (self->router)(&session, self->router_data);
  if (self->snat)
    (self->snat)(&session, self->snat_data);
  if (self->dnat)
    (self->dnat)(&session, self->dnat_data);
  
  stream = (self->chainer)(&session, self->chainer_data);

  if (stream)
    {
      gint fd;
      
      fd = z_stream_get_fd(stream);
      /*LOG
	This message reports that a secondary session server connection successfully established.
       */
      z_log(NULL, CORE_SESSION, 3, 
            "Secondary session, server connection established; server_fd='%d', server_address='%s', server_local='%s'", 
            fd, 
            session.server_address ? z_sockaddr_format(session.server_address, buf1, sizeof(buf1)) : "None", 
            session.server_local ? z_sockaddr_format(session.server_local, buf2, sizeof(buf2)) : "None");
    }
  else
    {
      /*LOG
	This message indicates that a secondary session connection failed to the server.
       */
      z_log(NULL, CORE_SESSION, 3, 
            "Secondary session, server connection failure; server_address='%s', server_local='%s'", 
            session.server_address ? z_sockaddr_format(session.server_address, buf1, sizeof(buf1)) : "None", 
            session.server_local ? z_sockaddr_format(session.server_local, buf2, sizeof(buf2)) : "None");
    }

  z_proxy_fastpath_session_destroy(&session);
  z_proxy_leave(self);
  
  return stream;
}
