############################################################################
##
## Copyright (c) 2000-2001 BalaBit IT Ltd, Budapest, Hungary
## All rights reserved.
##
## $Id: NAT.py,v 1.36 2004/07/22 07:48:17 bazsi Exp $
##
## Author  : Bazsi
## Auditor :
## Last audited version:
## Notes:
##
############################################################################

"""Module defining interfaces for network address translation.

This module defines 'AbstractNAT' an abstract interface for performing
address translation for client addresses.
"""

from Zorp import *
from SockAddr import SockAddrInet, inet_ntoa
from Domain import InetDomain
from Cache import ShiftCache
import Globals
import types

from whrandom import choice

NAT_SNAT = 1
NAT_DNAT = 2

Globals.nat_policies[None] = None

class NATPolicy:
	"""Class encapsulating named NAT instances.
	
	This class encapsulates a name and an associated NAT instance. NAT
	policies can be used where a NAT instance was used until now which
	avoids having to define NAT mappings for each service individually.

	"""
	def __init__(self, name, nat, cacheable=TRUE):
		"""Constructor to initialize a NAT policy

		This contructor initializes a NAT policy by storing
		arguments as attributes and registering this policy in the
		global nat_policies hash.

		Arguments

		  self      -- this instance

		  name      -- [QSTRING] name to identify this NAT policy

		  nat       -- [INST_nat] NAT instance

		  cacheable -- [BOOLEAN] Unset this attribute if NAT
		               decision is not cacheable. (It's depend's
		               on another thing than target (or source) IP
			       address).

		"""

		self.name = name
		self.nat = nat
		self.cacheable = cacheable
		if self.cacheable:
			self.nat_cache = ShiftCache('nat(%s)' % name, 1000)
		if Globals.nat_policies.has_key(name):
			raise ValueError, "Duplicate NATPolicy name: %s" % name
		Globals.nat_policies[name] = self

	def performTranslation(self, addr, nat_type=NAT_SNAT):
		## LOG ##
		# This message reports that the NAT type and the old address before the NAT mapping occurs.
		##
		log(None, CORE_DEBUG, 4, "Before NAT mapping; nat_type='%d', old_addr='%s'", (nat_type, str(addr)))
		if self.cacheable:
			try:
				addr.ip_s = self.nat_cache.lookup(addr.ip_s)
				v = addr
			except KeyError:
				ip = addr.ip_s
				v = self.nat.performTranslation(None, addr, nat_type)
				self.nat_cache.store(ip, v.ip_s)
		else:
			v = self.nat.performTranslation(None, addr, nat_type)

		## LOG ##
		# This message reports that the NAT type and the new address after the NAT mapping occurred.
		##
		log(None, CORE_DEBUG, 4, "After NAT mapping; nat_type='%d', new_addr='%s'", (nat_type, str(v)))
		return v

class AbstractNAT:
	"""Abstract class encapsulating interface for address translation.
	
	This class encapsulates an interface for application level network
	address translation. This NAT is not the same you are used to with
	regard to packet filters, it basically specifies/modifies outgoing
	source/destination addresses just before Zorp connects to the
	server.
	
	Source and destination NATs can be specified when a 'Service' is
	created.
	
	Currently it is used by the 'ConnectChainer' class just before 
	connecting to the server.
	"""

	def __init__(self):
		"""Constructor to initialize an AbstractNAT instance.

		This constructor initializes an AbstractNAT instance.
		Currently it does nothing, serves as a placeholder for
		future extensions.

		Arguments

		  self -- this instance
		"""
		pass

	def setupFastpath(self, proxy):
		"""Function called to setup fast path handlers.
		
		This function is called to provide the C-level fast path
		handlers to the proxy, which are called when subsequent
		connections are added to the proxy (currently only used
		for UDP proxies)
		"""
		pass

	def performTranslation(self, session, addr, nat_type = NAT_SNAT):
		"""Function to actually perform address translation.
		
		This function is called when a session is to be connected
		to its server endpoint. It should return the address to
		bind to before connecting.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session which is about to connect
		  
		  addr -- address to translate
		
		Returns
		
		  A SockAddr instance to bind to.
		"""
		raise NotImplementedError

class GeneralNAT(AbstractNAT):
	def __init__(self, mapping):
		"""

		Arguments

		  self     -- this instance

		  mapping  -- [INETDOMAIN_PAIR_LIST] List of pairs of
		              InetDomains in fro, to sequence.
		"""
		AbstractNAT.__init__(self)
		self.mapping = mapping

	def performTranslation(self, session, addr, nat_type):
		if type(self.mapping) != types.TupleType and type(self.mapping) != types.ListType:
			self.mapping = (self.mapping,)
			
		for (src_dom, dst_dom) in self.mapping:
			if src_dom.contains(addr):
				# addr is in domain, do translation
				hostaddr = src_dom.getHostAddr(addr)
				addr.ip = dst_dom.mapHostAddr(hostaddr)
				return addr
		return addr

class ForgeClientSourceNAT(AbstractNAT):
	"""Class to fake original client address for outgoing connections.
	
	This class implements the 'AbstractNAT' interface to perform address
	translation. It effectively uses the client address, so the connection
	seem to come from the original client instead of the firewall.
	
	It can be used when the original client address has some value (for
	example WWW and other servers in the DMZ performing address based
	access control).

	This class is OBSOLETE. Please use the forge_addr parameter of
	different Router classes. This may be removed in future.
	
	"""
	def performTranslation(self, session, addr, nat_type = NAT_SNAT):
		"""Function to actually perform address translation.
		
		This function returns the original client address with
		a dynamically allocated port number.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session which is about to connect
		  
		  addr -- address to translate
		  
		Returns
		
		  The address to bind to
		"""
		
		return SockAddrInet(session.client_address.ip_s, 0)

class StaticNAT(AbstractNAT):
	"""Class to set the source address of outgoing connections to a predefined value.
	
	This class implements the 'AbstractNAT' interface to perform address
	translation. It effectively assigns a predefined value as the
	address of the connection to be initiated.
	
	Attributes

	  addr	-- address to translate all addresses to

	"""
	def __init__(self, addr):
		"""Constructor to initialize a StaticNAT instance.

		This constructor initializes a StaticNAT instance by storing
		its arguments as attributes.

		Arguments

		  addr  -- [SOCKADDR] address to translate all addresses to
		"""
		AbstractNAT.__init__(self)
		self.addr = addr

	def performTranslation(self, session, addr, nat_type = NAT_SNAT):
		"""Function to actually perform address translation.
		
		This function returns the predefined address.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session which is about to connect
		  
		  addr -- address to translate
		  
		Returns
		
		  The address to bind to
		"""
		
		return SockAddrInet(self.addr.ip_s, self.addr.port)


class OneToOneNAT(AbstractNAT):
	"""Class to translate from a given address range to another.
	
	This class performs 1:1 address translation from the given source
	domain to the also given destination domain. If the source address
	is outside of the given source address range, DACException is raised.
	The supplied address ranges must have the same size.
	
	Instances of this class can be used to direct traffic destined to
	a block of IPs to another block, like having a lot of Web servers
	in your DMZ with dedicated IP aliases on your firewall.
	
	Attributes
	
	  from_domain  -- source subnet (InetDomain instance)
	  
	  to_domain    -- destination subnet (InetDomain instance)

	  default_reject -- whether to reject all connections not within the required range
	"""
        def __init__(self, from_domain, to_domain, default_reject=TRUE):
		"""Constructor to initialize a OneToOneNAT instance
		
		This constructor initializes a OneToOneNAT instance by
		storing arguments in attributes. Arguments must be
		InetDomain instances specifying two non-overlapping IP subnets
		with the same size.
		
		Arguments
		
		  from_domain  -- [INST_domain] source address range to translate addresses from
		  
		  to_domain    -- [INST_domain] destination address range to translate addresses into

		  default_reject -- [BOOLEAN] whether to reject all connections not within the required range
		"""
		AbstractNAT.__init__(self)
		self.from_domain = from_domain
		self.to_domain = to_domain
		self.default_reject = default_reject
		if from_domain.mask_bits != to_domain.mask_bits:
			raise ValueError, 'OneToOneNAT requires two domains of the same size'
		
	def performTranslation(self, session, addr, nat_type=NAT_SNAT):
		"""Function to actually perform 1:1 NAT
		
		This function actually performs the required changes to
		project an address in the source address range to the
		destination address range.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session we belong to
		  
		  addr -- address to translate
		  
		Returns
		
		  This function returns the calculated new address.
		"""
		try:
			return self.mapAddress(addr, self.from_domain, self.to_domain, nat_type)
		except ValueError:
			pass
		if self.default_reject:
			raise DACException, 'IP not within the required range.'
		else:
			return addr

	def mapAddress(self, addr, from_domain, to_domain, nat_type=NAT_SNAT):
		"""Function to map an address to another subnet.

		This function maps the address 'addr' in the domain
		'from_domain' to another domain 'to_domain'.

		Arguments

		  self        -- this instance

		  from_domain -- source domain

		  to_domain   -- destination domain

		  nat_type    -- specifies the NAT type

		Returns

		  a SockAddrInet in the destination domain or None

		"""
		if addr < from_domain:
			ip = (addr.ip & ~to_domain.mask) + (to_domain.ip & to_domain.mask)
			if nat_type == NAT_SNAT:
				return SockAddrInet(inet_ntoa(ip), 0)
			elif nat_type == NAT_DNAT:
				return SockAddrInet(inet_ntoa(ip), addr.port)

class OneToOneMultiNAT(OneToOneNAT):
	"""Class to translate from a given address range to another.

	This class is similar to OneToOneNAT as it performs 1:1 address
	translation from a given source domain to an also given destination
	domain. The difference is that this class supports multiple mappings
	at the same time by using a sequence of mapping pairs.

	If the source address is outside of the given source address range,
	DACException is raised.  The supplied address ranges must have the
	same size.

	Instances of this class can be used to direct traffic destined to
	a block of IPs to another block, like having a lot of Web servers
	in your DMZ with dedicated IP aliases on your firewall.

	Attributes

	  mapping  -- (src1,dst1),(src2,dst2)

	  default_reject -- whether to reject all connections not within the required range
	"""
        def __init__(self, mapping, default_reject=TRUE):
		"""Constructor to initialize a OneToOneMultiNAT instance

		This constructor initializes a OneToOneMultiNAT instance by
		storing arguments in attributes. Arguments must be
		InetDomain instances specifying two non-overlapping IP subnets
		with the same size.

		Arguments

		  mapping  -- [INETDOMAIN_PAIR_LIST] (from1, to1), (from2, to2)

		  default_reject -- [BOOLEAN] whether to reject all connections not within the required range
		"""
		AbstractNAT.__init__(self)
		self.mapping = mapping
		self.default_reject = default_reject
		for (from_domain, to_domain) in mapping:
			if from_domain.mask_bits != to_domain.mask_bits:
				raise ValueError, 'OneToOneMultiNAT requires two domains of the same size'

	def performTranslation(self, session, addr, nat_type=NAT_SNAT):
		"""Function to actually perform 1:1 NAT

		This function actually performs the required changes to
		project an address in the source address range to the
		destination address range.

		Arguments

		  self -- this instance

		  session -- session we belong to

		  addr -- address to translate

		Returns

		  This function returns the calculated new address.
		"""
		for (from_domain, to_domain) in self.mapping:
			try:
				return self.mapAddress(addr, from_domain, to_domain, nat_type)
			except ValueError:
				pass
		if self.default_reject:
			raise DACException, 'IP not within the required range.'
		else:
			return addr

class RandomNAT(AbstractNAT):
	"""Class to choose one of several addresses randomly.
	
	This class chooses one of the supplied addresses randomly.
	It can be used for load-balancing several lines by binding
	to different interfaces for each session.
	
	Attributes
	
	  addresses -- sequence of addresses to choose one from
	"""
	def __init__(self, addresses):
		"""Constructor to initialize a RandomNAT instance.
		
		This constructor initializes a RandomNAT instance by
		calling the inherited constructor and storing arguments
		in appropriate attributes.
		
		Arguments
		
		  addresses -- [SOCKADDR_list] sequence of addresses to choose one from, 
		               each item in this sequence must be a SockAddr
			       (or derived class) instance.
		"""
		AbstractNAT.__init__(self)
		self.addresses = addresses
	
	def performTranslation(self, session, addr, nat_type = NAT_SNAT):
		"""Function to actually perform random-address translation.
		
		This function actually performs address translation by
		choosing one of the supplied addresses randomly.
		
		Arguments
		
		  self -- this instance
		  
		  session -- session we belong to
		  
		  addr -- address to translate
		  
		Returns
		
		  The translated address is returned.
		"""
		return choice(self.addresses)

class HashNAT(AbstractNAT):
	def __init__(self, ip_hash, default_reject=TRUE):
		"""Constructor to initialize a HashNAT instance
		
		This constructor initializes a HashNAT instance by
		calling the inherited constructor and storing arguments
		in appropriate attributes.

		The purpose of HashNAT is to statically map an IP address to
		another. The implementation uses a hash table where the
		index is a string representation of the IP address, and the
		value is the translated IP address also in string form.
		
		Arguments
		
		  ip_hash -- [HASH;QSTRING;QSTRING] Hash of ip address.

		  default_reject -- [BOOLEAN] whether to reject all connections not within the required range

		"""
		AbstractNAT.__init__(self)
		self.ip_hash = ip_hash
		self.default_reject = default_reject

	def performTranslation(self, session, addr, nat_type=NAT_SNAT):
		"""Function to actually perform hash based translation.

		This function looks up the incoming address in the hash, and
		returns the mapped address.

		Arguments

		  self -- this instance

		  session -- session we belong to

		  addr -- address to map

		  nat_type -- nat type

		"""
		try:
			if nat_type == NAT_SNAT:
				ip = self.ip_hash[addr.ip_s]
				return SockAddrInet(ip, 0)
			else:
				ip = self.ip_hash[addr.ip_s]
				return SockAddrInet(ip, addr.port)
		except KeyError:
			if self.default_reject:
				raise DACException, 'IP not within the required range.'
			else:
				return addr

