/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: pyproxy.c,v 1.60 2004/07/26 12:31:58 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor : kisza
 * Last audited version: 1.15
 * Notes:
 *
 ***************************************************************************/

#include <zorp/pyproxy.h>
#include <zorp/proxy.h>
#include <zorp/policy.h>
#include <zorp/registry.h>
#include <zorp/modules.h>
#include <zorp/pystream.h>
#include <zorp/log.h>
#include <zorp/szig.h>

#include <ExtensionClass.h>

static PyObject *
z_py_zorp_proxy_new(ZorpProxy *self, PyObject *args);

static PyMethodDef z_py_zorp_proxy_methods[] =
{
  { "__init__", (PyCFunction) z_py_zorp_proxy_new, METH_VARARGS, NULL },
  { NULL, NULL, 0, NULL }
};



/**
 * z_py_zorp_proxy_new:
 * @self this
 * @args Python args: module_name, session_id, client, parent
 *
 * Constructor of ZorpProxy. Class is an abstract one, so the instance
 * already exists, only initialisation has to be done.
 *
 * Searches the registry for module_name, and creates its (C-side) self->proxy
 * using the returned constructor. For details see documentation about the
 * instantiation and invocation of custom proxies.
 *
 * Returns:
 * NULL on error, PyNone on success.
 */ 
static PyObject *
z_py_zorp_proxy_new(ZorpProxy *self, PyObject *args)
{
  gchar *module_name, *session_id;
  ZorpStream *client;
  PyObject *parent;
  ZProxy *parent_proxy = NULL;
  ZProxyParams params;
  gpointer proxy_create;
  int proxy_type = ZR_NONE;

  z_enter();
  if (!PyArg_ParseTuple(args, "ssOO", &module_name, &session_id, &client, &parent))
    {
      z_leave();
      return NULL;
    }

  if (!z_py_zorp_stream_check(client))
    {
      PyErr_SetString(PyExc_TypeError, "client must be a ZorpStream");
      z_leave();
      return NULL;
    }
    

  if (parent != Py_None)
    {
      parent_proxy = ((ZorpProxy *) parent)->proxy;
    }

  proxy_create = z_registry_get(module_name, &proxy_type);
  if (!proxy_create)
    {
      z_load_module(module_name);
      proxy_create = z_registry_get(module_name, &proxy_type);
    }
  if (!proxy_create)
    {
      /*LOG
	This message indicates that Zorp was unable to find the required proxy module.
	Check your installation, or contact your Zorp support for assistance.
       */
      z_log(NULL, CORE_ERROR, 1, "Cannot find proxy module; module='%s', type='%d'", module_name, proxy_type);
      PyErr_SetString(PyExc_RuntimeError, "Error loading proxy module.");
      return NULL;
    }
  Py_XINCREF(self);
  params.session_id = session_id;
  params.pyclient = (PyObject *) client;
  params.client = client->stream;
  params.handler = (PyObject *) self;
  params.parent = parent_proxy;
  switch (proxy_type)
    {
    case ZR_PROXY:
    case ZR_PYPROXY:
      Py_BEGIN_ALLOW_THREADS;
      self->proxy = (*(ZProxyCreateFunc) proxy_create)(&params);
      Py_END_ALLOW_THREADS;
      break;
    default:
      Py_XDECREF(self);
      /*LOG
        This module indicates that the given proxy module couldn't be found
        or an error occurred during the loading of the module.
       */
      z_log(NULL, CORE_ERROR, 1, "Cannot find proxy module; module='%s', type='%d'", module_name, proxy_type);
      PyErr_SetString(PyExc_RuntimeError, "Error loading proxy module.");
      return NULL;
    }
  z_policy_thread_ready(self->proxy->thread);
  Py_XDECREF(self);
  Py_XINCREF(Py_None);
  z_leave();
  return Py_None;
}


/**
 * z_py_zorp_proxy_get_proxy:
 * @obj this
 *
 * Get the embedded proxy.
 *
 * Returns:
 * The embedded proxy
 */
ZProxy *
z_py_zorp_proxy_get_proxy(PyObject *obj)
{
  return ((ZorpProxy *) obj)->proxy;
}


/**
 * z_py_zorp_proxy_free:
 * @self this
 *
 * Destructor of ZorpProxy
 */
static void
z_py_zorp_proxy_free(ZorpProxy *self)
{
  if (self->proxy)
    z_proxy_unref(self->proxy);
  PyObject_Del(self);
}


/**
 * z_py_zorp_proxy_getattr:
 * @self this
 * @name Attribute name
 *
 * Get an attribute of the proxy. Actually there is nothing to get from the
 * abstract ZorpProxy, rather from its descendants, so if the embedded proxy
 * is initialised, its vars (or session_vars->vars) is searched for the name.
 *
 * Returns:
 * The attribute value
 */
static PyObject *
z_py_zorp_proxy_getattr(ZorpProxy *self, char *name)
{
  PyObject *v;

  /* NOTE: we don't support fetching proxy attributes as long as the proxy
   * is not initialized */
  if (self->proxy)
    {
      v = z_proxy_vars_getattr(self->proxy, name);
      if (v)
        {
          if (z_log_enabled(CORE_DEBUG, 6))
            {
              PyObject *repr = PyObject_Repr(v);
              /*LOG
                This message reports that the given proxy-exported 
                attribute was fetched, and it contained the also given value.
               */
              z_log(self->proxy->session_id, CORE_DEBUG, 6, "Attribute fetched; attribute='%s', value='%s'", name, PyString_AsString(repr));
              Py_XDECREF(repr);
            }
          return v;
        }
    }
  
  return Py_FindMethod(z_py_zorp_proxy_methods, (PyObject *) self, name);
}

/**
 * z_py_zorp_proxy_setattr:
 * @self this
 * @name Attribute name
 * @value New attribute value
 *
 * Set an attribute of the proxy. The recognised attribute names are the same
 * as for _getattr, the type of @value is the same as _getattr would return
 * for @name.
 *
 * Returns:
 * The attribute value
 */
static gint
z_py_zorp_proxy_setattr(ZorpProxy *self, char *name, PyObject *value)
{
  if (self->proxy && z_proxy_get_state(self->proxy) > ZPS_INITIAL)
    { 
      if (z_proxy_vars_setattr(self->proxy, name, value))
        {
          if (z_log_enabled(CORE_DEBUG, 6))
            {
              PyObject *repr = PyObject_Repr(value);
              /*LOG
                This message reports that the given proxy-exported attribute
                was changed to the given value.
               */
              z_log(self->proxy->session_id, CORE_DEBUG, 6, "Attribute changed; attribute='%s', newvalue='%s'", name, PyString_AsString(repr));
              Py_XDECREF(repr);
            }
          return 0;
        }
      else
        if (PyErr_Occurred())
          {
            PyErr_Print();
            return 1;
          }
    }
  if (PyErr_Occurred())
    PyErr_Print(); 
  return PyEC_SetAttrString((PyObject *) self, name, value);
}

static PyExtensionClass z_py_zorp_proxy_type = 
{
  PyObject_HEAD_INIT(&PyType_Type)
  0,                                        /*ob_size*/
  "ZorpProxy",                              /*tp_name*/
  sizeof(ZorpProxy),                        /*tp_basicsize*/
  0,                                        /*tp_itemsize*/
  /* methods */
  (destructor)z_py_zorp_proxy_free,         /*tp_dealloc*/
  (printfunc)0,                             /*tp_print*/
  (getattrfunc)z_py_zorp_proxy_getattr,     /*tp_getattr*/
  (setattrfunc)z_py_zorp_proxy_setattr,     /*tp_setattr*/
  (cmpfunc)0,                               /*tp_compare*/
  (reprfunc)0,                              /*tp_repr*/
  0,                                        /*tp_as_number*/
  0,                   		            /*tp_as_sequence*/
  0,                   		            /*tp_as_mapping*/
  (hashfunc)0,         		            /*tp_hash*/
  (ternaryfunc)0,      		            /*tp_call*/
  (reprfunc)0,         		            /*tp_str*/

  /* Space for future expansion */
  0L,0L,0L,0L,
  "ZorpProxy class", /* Documentation string */
  METHOD_CHAIN(z_py_zorp_proxy_methods),
  0, 0, 0, 0
};


/**
 * z_py_zorp_proxy_check:
 * @obj this
 *
 * Checks if @obj is really compatible with ZorpProxy.
 *
 * Returns:
 * TRUE if compatible
 */
gboolean
z_py_zorp_proxy_check(PyObject *obj)
{
  return (void *) obj->ob_type == (void *) &z_py_zorp_proxy_type;
}

/**
 * z_py_zorp_proxy_init:
 *
 * Module initialisation.
 * FIXME: some words about GetDict...
 */
void
z_py_zorp_proxy_init(void)
{
  PyObject *m, *d;

  m = PyImport_AddModule("Zorp.Zorp");
  /* Py_InitModule("Zorp.Proxy", z_py_zorp_proxy_funcs); */
  d = PyModule_GetDict(m);
  PyExtensionClass_Export(d, "ZorpProxy", z_py_zorp_proxy_type);
}
